/**
 ******************************************************************************
 *
 * @file        Sample_I2C_ByteMode.h
 *
 * @brief       Sample I2C Byte Mode Transfer Head File 
 *
 * @par         Project
 *              MG32
 * @version     V2.22
 * @date        2021/05/25
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2019 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer 
 *       The Demo software is provided "AS IS"  without any warranty, either 
 *       expressed or implied, including, but not limited to, the implied warranties 
 *       of merchantability and fitness for a particular purpose.  The author will 
 *       not be liable for any special, incidental, consequential or indirect 
 *       damages due to loss of data or any other reason. 
 *       These statements agree with the world wide and local dictated laws about 
 *       authorship and violence against these laws. 
 ******************************************************************************
 *******************************************************************************
 */



#ifndef __MG32_I2C_ByteMode_H
#define __MG32_I2C_ByteMode_H
//#define __MG32_I2C_ByteMode_H_VER                            0.01

#ifdef __cplusplus
 extern "C" {
#endif


/* Includes ------------------------------------------------------------------*/
#include "Sample__Common.h"
#include "MG32_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/**
 * @name    I2C Initialze TypeDef
 *          I2C Initialze TypeDef
 */ 
///@{

/** 
  * @brief  I2C Sample Initial Structure definition
  */
typedef struct __attribute__((packed))
{
    __IO uint32_t            CK_I2C_PR_Frequency;   /*!< I2C Module Input Freq.             */
    __IO uint32_t            SCL_Clock;             /*!< SCL Output Freq.                   */
    __IO uint8_t             OwnAddress1;           /*!< Own Address 1                      */
    __IO uint8_t             DualAddressMode;       /*!< Dual Address Enable / Disable      */
    __IO uint8_t             OwnAddress2;           /*!< Own Address 2                      */
    __IO uint8_t             GeneralCallMode;       /*!< Generalcall Mode Enable / Disable  */
}Sample_I2C_InitTypeDef;
///@}

/**
 * @name    I2C Control TypeDef
 *          I2C Control TypeDef
 */ 
///@{
/** 
  * @brief  I2C Sample XferNext enum definition
  */
typedef enum{
    Sample_I2C_XferNext_Mask        = 0x0003UL,
    Sample_I2C_XferNext_RepeatStart = 0x0001UL,
    Sample_I2C_XferNext_STOP        = 0x0002UL,
}Sample_I2C_XferNextTypeDef;
///@}

/** 
  * @brief  I2C Sample Control Structure definition
  */
///@{
typedef union __attribute__((packed)){
    uint16_t H;                       /*!< Halfword Access    */
    uint8_t  B[2];                    /*!< Byte Access        */
    struct __attribute__((packed)){
       uint16_t XferNext      :2;    /*!< Master Trnasfer End Type, Repeat Start or Stop */
       uint16_t NACKNext      :1;    /*!< Master Transmit Data NACK not Error            */
       uint16_t Listen        :1;    /*!< Always Listen Own Addrsss Enable / Disable     */
       uint16_t               :8;    /*!< RESERVED                                       */
       uint16_t DMA_STx       :1;    /*!< Slave Transmit Data transfer use DMA           */
       uint16_t DMA_SRx       :1;    /*!< Slave Receive Data transfer use DMA            */
       uint16_t DMA_MRx       :1;    /*!< Master Receive Data transfer use DMA           */
       uint16_t DMA_MTx       :1;    /*!< Master Transmit Data transfer use DMA          */
    }MBit;                           /*!< Function Bit Access                            */
}Sample_I2C_ControlTypeDef;
///@}

/** 
  * @brief  I2C Sample State Structure definition
  */
///@{
typedef union __attribute__((packed)){
    __IO uint8_t B;                       /*!< Byte Access                                            */
    struct __attribute__((packed)){ 
        uint8_t TX             :1;   /*!< Bit 0 : Tx state                                       */
                                     /*           0 :Ready (no Tx operation ongoing)             */
                                     /*           1 :Busy (Tx operation ongoing)                 */
        uint8_t RX             :1;   /*!< Bit 1 : Rx state                                       */
                                     /*           0 :Ready (no Rx operation ongoing)             */
                                     /*           1 :Busy (Tx operation ongoing)                 */
        uint8_t Error          :2;   /*!< Bit [3:2] Error information                            */
                                     /*           00:No Error                                    */
                                     /*           01:Abort (Abort user request on going)         */
                                     /*           10:Timeout                                     */
                                     /*           11:Error                                       */
        uint8_t                :1;   /*!< Bit 4 : Reserved                                       */
                                     /*           x :Should be set to 0                          */
        uint8_t Listen         :1;   /*!< Bit 5 : Listen (peripheral in Address Listen Mode)     */
                                     /*           0 :Ready or Busy (No Listen mode ongoing)      */
                                     /*           1 :Listen (peripheral in Address Listen Mode)  */
        uint8_t Busy           :1;   /*!< Bit 6 : Intrinsic process state                        */
                                     /*           0 :Ready or Busy (No Listen mode ongoing)      */
                                     /*           1 :Listen (peripheral in Address Listen Mode)  */
        uint8_t Ready          :1;   /*!< Bit 7 : Peripheral initialization status               */
    }MBit;                           /*!< Function Bit Access                                    */
}Sample_I2C_StateTypeDef;
///@}

/** 
  * @brief  I2C Sample Error State Structure definition
  */
///@{
typedef union __attribute__((packed)){
    __IO uint16_t H;                         /*!< Halfword Access    */
    struct __attribute__((packed)){
        uint16_t BusError           :1; /*!< Bit 0 :Bus error                                       */
        uint16_t ArbitrationLost    :1; /*!< Bit 1 :ArbitrationLost error                           */
        uint16_t NACK               :1; /*!< Bit 2 :NACK error                                      */
        uint16_t Over               :1; /*!< Bit 3 :Over error                                      */
        uint16_t Timeout            :1; /*!< Bit 4 :Timeout error                                   */
        uint16_t Size               :1; /*!< Bit 5 :Size Management error                           */
        uint16_t Callback           :1; /*!< Bit 6 :Invalid Callback error                          */
        uint16_t Parameter          :1; /*!< Bit 7 :Invalid Parameters error                        */
        uint16_t DMACH              :1; /*!< Bit 8 :DMA transfer error                              */
        uint16_t DMA_PARAM          :1; /*!< Bit 9 :DMA Parameter Error                             */
        uint16_t                    :6; /*!< Bit [15:10] Reserved                                   */
    }MBit;                              /*!< Function Bit Access                                    */
}Sample_I2C_ErrorTypeDef;
///@}

/** 
  * @brief  I2C Handle Structure definition
  */
///@{
typedef struct __attribute__((packed)){
    I2C_Struct                 *Instance;    /*!< I2C registers base address                */
    Sample_I2C_InitTypeDef     Init;         /*!< I2C communication parameters              */
    uint8_t                    *pBuffer;     /*!< Pointer to I2C transfer buffer            */
    DMAChannel_Struct          *DMACHx;      /*!< DMA Channel registers base address        */
    Sample_I2C_ControlTypeDef  Control;      /*!< I2C Opration Control                      */
    uint16_t                   XferCount;    /*!< I2C transfer counter                      */
    uint16_t                   XferSize;     /*!< I2C transfer size                         */
    Sample_I2C_ErrorTypeDef    Error;        /*!< I2C Error code                            */
    Sample_I2C_StateTypeDef    State;        /*!< I2C communication state                   */
    uint8_t                    SlaveAddress; /*!< Bit[7:1]Device Address                    */
                                             /*   Bit0 Read / Write                         */
}Sample_I2C_HandleTypeDef;
///@}

/* Private define ------------------------------------------------------------*/
#define Sample_I2C_DualAddress_Enable     (0x00UL)                    /*!< DualAddress_Enable  */
#define Sample_I2C_DualAddress_Disable    I2C_CR0_SADR2_EN_enable_b0; /*!< DualAddress_Disable */
    
#define Sample_I2C_GENERALCALL_Enable     (0x00UL)                    /*!< GENERALCALL_Enable  */
#define Sample_I2C_GENERALCALL_Disable    I2C_CR0_SADR2_EN_enable_b0; /*!< GENERALCALL_Disable */

/** 
 * @name I2C_DMA_ACCESS
 *       I2C DMA Access Types
 */
///@{
#define Sample_I2C_Listen                 0x0008UL      /*!< Control Listen     */
#define Sample_I2C_DMA_MTx                0x8000UL      /*!< Control MTx        */
#define Sample_I2C_DMA_MRx                0x4000UL      /*!< Control MRx        */
#define Sample_I2C_DMA_SRx                0x2000UL      /*!< Control SRx        */
#define Sample_I2C_DMA_STx                0x1000UL      /*!< Control STx        */
///@}

/**
 * @name    I2C State TypeDef
 *          I2C State TypeDef
 */ 
///@{
#define Sample_I2C_State_Reset            (uint8_t)0x00 /*!< Peripheral is not yet Initialized         */
#define Sample_I2C_State_Ready            (uint8_t)0x80 /*!< Peripheral Initialized and ready for use  */
        
#define Sample_I2C_State_Busy             (uint8_t)0x40 /*!< An internal process is ongoing            */
#define Sample_I2C_State_Busy_Rx          (uint8_t)0xC2 /*!< Data Reception process is ongoing         */
#define Sample_I2C_State_Busy_Tx          (uint8_t)0xC1 /*!< Data Transmission process is ongoing      */
        
#define Sample_I2C_State_Listen           (uint8_t)0x20 /*!< Address Listen Mode is ongoing            */
#define Sample_I2C_State_Listen_Rx        (uint8_t)0xE2 /*!< Address Listen Mode and Data Reception process is ongoing */
#define Sample_I2C_State_Listen_Tx        (uint8_t)0xE1 /*!< Address Listen Mode and Data Transmission process is ongoing */
        
#define Sample_I2C_State_Rx               (uint8_t)0x02 /*!< Address Data Reception process is ongoing */
#define Sample_I2C_State_Tx               (uint8_t)0x01 /*!< Address Data Transmission process is ongoing */
        
#define Sample_I2C_State_Abort            (uint8_t)0x04 /*!< Abort user request ongoing                */
#define Sample_I2C_State_Timeout          (uint8_t)0x08 /*!< Timeout state                             */
#define Sample_I2C_State_Error            (uint8_t)0x0C /*!< Error                                     */
///@}
    
/** 
 * @name    I2C Error TypeDef 
 *          I2C Error TypeDef 
 */   
///@{ 
#define Sample_I2C_Error_None             (0x0000UL)    /*!< None                       */
#define Sample_I2C_Error_BusError         (0x0001UL)    /*!< Bus Error                  */
#define Sample_I2C_Error_ArbitrationLost  (0x0002UL)    /*!< ArbitrationLost error      */
#define Sample_I2C_Error_NACK             (0x0004UL)    /*!< Data NACK error            */
#define Sample_I2C_Error_Over             (0x0008UL)    /*!< SRx / STx Over Size error  */
#define Sample_I2C_Error_Timeout          (0x0010UL)    /*!< Timeout error              */
#define Sample_I2C_Error_Size             (0x0020UL)    /*!< Size Management error      */
#define Sample_I2C_Error_Callback         (0x0040UL)    /*!< Callback                   */
#define Sample_I2C_Error_Parameter        (0x0080UL)    /*!< Parameter                  */
#define Sample_I2C_Error_DMA              (0x0100UL)    /*!< DMA                        */
#define Sample_I2C_Error_DMA_PARAM        (0x0200UL)    /*!< DMA_PARAM                  */
///@}

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
void Sample_I2C0_ByteMode(void);

#if defined(MG32_1ST) || defined(MG32_3RD) || defined(MG32_4TH) || defined(MG32_5TH)
  void Sample_I2C1_ByteMode(void);
#endif

SMP_Return Sample_I2C_DeInit(Sample_I2C_HandleTypeDef *SI2C);
SMP_Return Sample_I2C_ByteMode_Init(Sample_I2C_HandleTypeDef *SI2C);

/******* Blocking mode: Polling */
SMP_Return Sample_I2C_IsDeviceReady(Sample_I2C_HandleTypeDef *SI2C, uint16_t DevAddress, uint32_t Trials);

SMP_Return Sample_I2C_ByteMode_MasterTransmit(Sample_I2C_HandleTypeDef *SI2C, uint16_t SlaveAddress, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_MasterTransmit_RepeatStart(Sample_I2C_HandleTypeDef *SI2C, uint16_t SlaveAddress, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_MasterReceive(Sample_I2C_HandleTypeDef *SI2C, uint16_t SlaveAddress, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_SlaveReceive(Sample_I2C_HandleTypeDef *SI2C, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_SlaveTransmit(Sample_I2C_HandleTypeDef *SI2C, uint8_t *pData, uint16_t Size);

/******* Non-Blocking mode: Interrupt */
SMP_Return Sample_I2C_ByteMode_MasterTransmit_IT(Sample_I2C_HandleTypeDef *SI2C, uint16_t SlaveAddress, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_MasterTransmit_IT_RepeatStart(Sample_I2C_HandleTypeDef *SI2C, uint16_t SlaveAddress, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_MasterReceive_IT(Sample_I2C_HandleTypeDef *SI2C, uint16_t SlaveAddress, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_SlaveReceive_IT(Sample_I2C_HandleTypeDef *SI2C, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_SlaveTransmit_IT(Sample_I2C_HandleTypeDef *SI2C, uint8_t *pData, uint16_t Size);

/******* Non-Blocking mode: DMA */
SMP_Return Sample_I2C_ByteMode_MasterTransmit_DMA(Sample_I2C_HandleTypeDef *SI2C, uint16_t SlaveAddress, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_MasterTransmit_DMA_RepeatStart(Sample_I2C_HandleTypeDef *SI2C, uint16_t SlaveAddress, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_MasterReceive_DMA(Sample_I2C_HandleTypeDef *SI2C, uint16_t SlaveAddress, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_SlaveReceive_DMA(Sample_I2C_HandleTypeDef *SI2C, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_ByteMode_SlaveTransmit_DMA(Sample_I2C_HandleTypeDef *SI2C, uint8_t *pData, uint16_t Size);
SMP_Return Sample_I2C_DMA_Handle(Sample_I2C_HandleTypeDef *SI2C);

/******* Non-Blocking mode: Interrupt and DMA */
SMP_Return Sample_I2C_Enable_Listen(Sample_I2C_HandleTypeDef *SI2C);
SMP_Return Sample_I2C_Disable_Listen(Sample_I2C_HandleTypeDef *SI2C);
SMP_Return Sample_I2C_Get_Transfer_State(Sample_I2C_HandleTypeDef *SI2C);
SMP_Return Sample_I2C_Get_Listen_State(Sample_I2C_HandleTypeDef *SI2C);

/******* Byte mode: Polling and Interrupt and DMA Handle */
void Sample_I2C_ByteMode_Handle(Sample_I2C_HandleTypeDef *Sample_I2C);

/* External vairables --------------------------------------------------------*/



#ifdef __cplusplus
}
#endif

#endif


