/**
 ******************************************************************************
 *
 * @file        MG32_LCD_DRV.h
 *
 * @brief       This is the C code format driver head file.
 *
 * @par         Project
 *              MG32
 * @version     V1.11
 * @date        2025/06/16
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2024 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer 
 *      The Demo software is provided "AS IS"  without any warranty, either 
 *      expressed or implied, including, but not limited to, the implied warranties 
 *      of merchantability and fitness for a particular purpose.  The author will 
 *      not be liable for any special, incidental, consequential or indirect 
 *      damages due to loss of data or any other reason. 
 *      These statements agree with the world wide and local dictated laws about 
 *      authorship and violence against these laws. 
 ******************************************************************************
 ******************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#include "MG32.h"

#if defined(ModuleExist_LCD)

#ifndef __MG32_LCD_DRV_H
#define __MG32_LCD_DRV_H

#ifdef __cplusplus
extern "C" {
#endif

/** @name MG32_DRV
  * @{
  */

/* Includes ------------------------------------------------------------------*/
#include "MG32__Common_DRV.h"
#include "MG32_LCD.h"

/** @name LCD LCD
  * @{
  */

/* Exported types ------------------------------------------------------------*/

/** @name LCD_Exported_Types LCD Exported Types
  * @{
  */


/**
  * @}
  */

/* Exported constants --------------------------------------------------------*/

/** @name LCD_Exported_Constants LCD Exported Constants
  * @{
  */

/** @name LCD_ClcokPreDivider LCD Clock PreDivider
  * @{
  */
typedef enum{
    PreDivider_8 = LCD_CLK_CK_PDIV_div8_w,
    PreDivider_16 = LCD_CLK_CK_PDIV_div16_w,
    PreDivider_32 = LCD_CLK_CK_PDIV_div32_w,
    PreDivider_64 = LCD_CLK_CK_PDIV_div64_w,
    PreDivider_128 = LCD_CLK_CK_PDIV_div128_w,
    PreDivider_256 = LCD_CLK_CK_PDIV_div256_w,
    PreDivider_512 = LCD_CLK_CK_PDIV_div512_w,
    PreDivider_1024 = LCD_CLK_CK_PDIV_div1024_w,
}LCD_ClcokPreDivider_TypeDef;
/**
  * @}
  */


/** @name LCD_ClcokSelect LCD input clock CK_LCD source select.
  * @{
  */
typedef enum{
    ClcokSource_CK_LS = LCD_CLK_CK_SEL_ck_ls_w,
    ClcokSource_NCO_P0 = LCD_CLK_CK_SEL_nco_p0_w,
    ClcokSource_PROC = LCD_CLK_CK_SEL_proc_w,
    ClcokSource_TM01_TRGO = LCD_CLK_CK_SEL_tm01_trgo_w,
}LCD_ClcokSource_TypeDef;
/**
  * @}
  */


/** @name LCD_ClcokDivider LCD Clock Divider
  * @{
  */
typedef enum{
    Divider_1 = LCD_CLK_CK_DIV_div1_w,
    Divider_2 = LCD_CLK_CK_DIV_div2_w,
    Divider_4 = LCD_CLK_CK_DIV_div4_w,
    Divider_8 = LCD_CLK_CK_DIV_div8_w,
    Divider_16 = LCD_CLK_CK_DIV_div16_w,
    Divider_32 = LCD_CLK_CK_DIV_div32_w,
    Divider_64 = LCD_CLK_CK_DIV_div64_w,
}LCD_ClcokDivider_TypeDef;
/**
  * @}
  */

//=================================================================================================

/** @name LCD_VxPinControl LCD voltage rail LCD_VT / LCD_V1 / LCD_V2 / LCD_V3 pin connection control.
  * @{
  */
#define VTPin_NC    LCD_CR1_VT_CTL_nc_w
#define VTPin_Pin1  LCD_CR1_VT_CTL_pin1_w
#define V1Pin_NC    LCD_CR1_V1_CTL_nc_w
#define V1Pin_Pin1  LCD_CR1_V1_CTL_pin1_w
#define V2Pin_NC    LCD_CR1_V2_CTL_nc_w
#define V2Pin_Pin1  LCD_CR1_V2_CTL_pin1_w
#define V3Pin_NC    LCD_CR1_V3_CTL_nc_w
#define V3Pin_Pin1  LCD_CR1_V3_CTL_pin1_w
/**
  * @}
  */


/** @name LCD_VTPowerSourceSelect LCD bias power source selection.
  * @{
  */
typedef enum{
    PowerSource_OFF = LCD_CR0_VS_SEL_off_w,
    PowerSource_AVDD = LCD_CR0_VS_SEL_avdd_w,
    PowerSource_EXT = LCD_CR0_VS_SEL_ext_w,
    PowerSource_CP = LCD_CR0_VS_SEL_cp_w,
}LCD_VTPowerSource_TypeDef;
/**
  * @}
  */


/** @name LCD_Constrast_Level LCD Constrast Level
  * @{
  */

typedef enum{
    CPContrast_Level0 = LCD_CR1_VT_SEL_lvl0_w,
    CPContrast_Level1 = LCD_CR1_VT_SEL_lvl1_w,
    CPContrast_Level2 = LCD_CR1_VT_SEL_lvl2_w,
    CPContrast_Level3 = LCD_CR1_VT_SEL_lvl3_w,
    CPContrast_Level4 = LCD_CR1_VT_SEL_lvl4_w,
    CPContrast_Level5 = LCD_CR1_VT_SEL_lvl5_w,
    CPContrast_Level6 = LCD_CR1_VT_SEL_lvl6_w,
    CPContrast_Level7 = LCD_CR1_VT_SEL_lvl7_w,
    CPContrast_Level8 = LCD_CR1_VT_SEL_lvl8_w,
    CPContrast_Level9 = LCD_CR1_VT_SEL_lvl9_w,
    CPContrast_Level10 = LCD_CR1_VT_SEL_lvl10_w,
    CPContrast_Level11 = LCD_CR1_VT_SEL_lvl11_w,
    CPContrast_Level12 = LCD_CR1_VT_SEL_lvl12_w,
    CPContrast_Level13 = LCD_CR1_VT_SEL_lvl13_w,
    CPContrast_Level14 = LCD_CR1_VT_SEL_lvl14_w,
}LCD_ChargePumpContrast_TypeDef;

/**
  * @}
  */


/** @name LCD_ChargePumpClockDivider LCD Charge Pump Clock Selection
  * @{
  */
typedef enum{
    CPClock_Div1 = LCD_CLK_CK_CDIV_div1_w,
    CPClock_Div2 = LCD_CLK_CK_CDIV_div2_w,
    CPClock_Div3 = LCD_CLK_CK_CDIV_div3_w,
    CPClock_Div4 = LCD_CLK_CK_CDIV_div4_w,
}LCD_ChargePumpClcokDivider_TypeDef;
/**
  * @}
  */


/** @name LCD_PinOffState LCD off state LCD pin control.
  * @{
  */
typedef enum{
    OffState_Hiz = LCD_CR1_OFF_CTL_hiz_w,
    OffState_Low = LCD_CR1_OFF_CTL_low_w,
}LCD_PinOffState_TypeDef;
/**
  * @}
  */


/** @name LCD_ResisterLadderSelect LCD reference voltage resister ladder selection.
  * @{
  */
typedef enum{
    Resister_Int = LCD_CR0_RL_SEL_int_w,
    Resister_Ext = LCD_CR0_RL_SEL_ext_w,
}LCD_ResisterLadder_TypeDef;
/**
  * @}
  */


/** @name LCD_BiasSelect LCD bias voltage select.
  * @{
  */
typedef enum{
    BIAS_Static = LCD_CR0_BIAS_static_w,
    BIAS_B12 = LCD_CR0_BIAS_b12_w,
    BIAS_B13 = LCD_CR0_BIAS_b13_w,
    BIAS_B14 = LCD_CR0_BIAS_b14_w,
}LCD_BiasSelect_TypeDef;
/**
  * @}
  */

//=================================================================================================

/** @name LCD_FrameModeSelect LCD frame control mode selection.
  * @{
  */
typedef enum{
    FrameTypeA = LCD_CR0_FRM_MDS_typea_w,
    FrameTypeB = LCD_CR0_FRM_MDS_typeb_w,
}LCD_FrameMode_TypeDef;
/**
  * @}
  */


/** @name LCD_DutyCycleSelect LCD duty cycle select bits.
  * @{
  */
typedef enum{
    DCStatic = LCD_CR0_DUTY_static_w,
    DC12Duty = LCD_CR0_DUTY_d12_w,
    DC13Duty = LCD_CR0_DUTY_d13_w,
    DC14Duty = LCD_CR0_DUTY_d14_w,
    DC15Duty = LCD_CR0_DUTY_d15_w,
    DC16Duty = LCD_CR0_DUTY_d16_w,
    DC17Duty = LCD_CR0_DUTY_d17_w,
    DC18Duty = LCD_CR0_DUTY_d18_w,
}LCD_DutyCycle_TypeDef;
/**
  * @}
  */


/** @name LCD_WaveformInverse LCD COM/SEG waveform inverse enable.
  * @{
  */
typedef enum{
    WaveformNormal = LCD_CR0_CS_INV_normal_w,
    WaveformInverse = LCD_CR0_CS_INV_inverse_w,
}LCD_WaveformInverse_TypeDef;
/**
  * @}
  */


/** @name LCD_DeadTimeModeSelect LCD output dead time mode selection.
  * @{
  */
typedef enum{
    DTFrame = LCD_CR0_DT_MDS_frame_w,
    DTDuty = LCD_CR0_DT_MDS_duty_w,
}LCD_DeadTime_TypeDef;
/**
  * @}
  */


/** @name LCD_DriveStrengthMode LCD drive strength mode selection.
  * @{
  */
typedef enum{
    DSNormal = LCD_CR1_DRV_MDS_normal_w,
    DSHigh = LCD_CR1_DRV_MDS_high_w,
}LCD_DriveStrengthMode_TypeDef;
/**
  * @}
  */

//=================================================================================================

/** @name LCD_InterruptEnable LCD Interrupt Enable Definition
  * @{
  */
#define LCD_IT_IEA           LCD_INT_IEA_mask_w             /* LCD interrupt all enable. When disables */
#define LCD_IT_SOF           LCD_INT_SOF_IE_mask_w          /* LCD start of frame interrupt enable. */
#define LCD_IT_UDC           LCD_INT_UDC_IE_mask_w          /* LCD update display data completed interrupt enable. */
#define LCD_IT_CPR           LCD_INT_CPR_IE_mask_w          /* LCD charge pump power ready interrupt enable. */
#define LCD_IT_BLKON         LCD_INT_BLKON_IE_mask_w        /* LCD blinking on interrupt enable. */
#define LCD_IT_BLKOFF        LCD_INT_BLKOFF_IE_mask_w       /* LCD blinking off interrupt enable. */
/**
  * @}
  */


/** @name LCD_FlagStatus  LCD Flag Status Definition
  * @{
  */
#define LCD_FLAG_BLKOFF      LCD_STA_BLKOFF_mask_w          /* LCD blinking off interrupt flag. */
#define LCD_FLAG_BLKONF      LCD_STA_BLKONF_mask_w          /* LCD blinking on interrupt flag. */
#define LCD_FLAG_CPRF        LCD_STA_CPRF_mask_w            /* LCD charge pump power ready interrupt flag. */
#define LCD_FLAG_UDCF        LCD_STA_UDCF_mask_w            /* LCD update display data completed interrupt flag. */
#define LCD_FLAG_SOF         LCD_STA_SOF_mask_w             /* LCD start of frame interrupt flag. */
#define LCD_FLAG_PRDYF       LCD_STA_PRDYF_mask_w           /* LCD internal power source ready statue. */
#define LCD_FLAG_DEF         LCD_STA_DEF_mask_w             /* LCD display enabled status. */
/**
  * @}
  */

//=================================================================================================

/** @name LCD_SegmentOff LCD segments all off enable.
  * @{
  */
typedef enum{
    Segment_ON = LCD_CR2_SEG_OFF_on_w,
    Segment_OFF = LCD_CR2_SEG_OFF_off_w,
}LCD_SegmentOff_TypeDef;
/**
  * @}
  */

//=================================================================================================

/** @name LCD_BlinkClockDivider LCD Blink Clock Divider
  * @{
  */
typedef enum{
    BDivider_4 = LCD_CLK_CK_BDIV_div4_w,
    BDivider_8 = LCD_CLK_CK_BDIV_div8_w,
    BDivider_16 = LCD_CLK_CK_BDIV_div16_w,
    BDivider_32 = LCD_CLK_CK_BDIV_div32_w,
}LCD_BlinkClcokDivider_TypeDef;
/**
  * @}
  */


/** @name LCD_BlinkModeSelect LCD Blink Clock Divider
  * @{
  */
typedef enum{
    BlinkDisable = LCD_CR2_BLK_MDS_disable_w,
    BlinkAll = LCD_CR2_BLK_MDS_all_w,
}LCD_BlinkMode_TypeDef;
/**
  * @}
  */

/**
  * @}
  */

/* Exported macro ------------------------------------------------------------*/

/** @name LCD_Exported_Macros LCD Exported Macros
  * @{
  */

/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/

/** @name LCD_Exported_Functions
  * @{
  */


/** @name LCD_Exported_Functions_Clock
  * @{
  */

void LCD_SetClockPreDivider(LCD_Struct* LCDx, LCD_ClcokPreDivider_TypeDef PreDivider);
void LCD_SetClockSource(LCD_Struct* LCDx, LCD_ClcokSource_TypeDef SourceSelect);
void LCD_SetClockDivider(LCD_Struct* LCDx, LCD_ClcokDivider_TypeDef Divider);
void LCD_SetClockPrescaler(LCD_Struct* LCDx, uint32_t Prescaler);

/**
  * @}
  */


/** @name LCD_Exported_Functions_Power
  * @{
  */

void LCD_OverVoltageDetection(LCD_Struct* LCDx, FunctionalState OVDState);
void LCD_VxPinControl(LCD_Struct* LCDx, uint32_t VxPin);
void LCD_VTBiasPowerSource(LCD_Struct* LCDx, LCD_VTPowerSource_TypeDef VTBPSource);

void LCD_ChargePumpConstrast_Config(LCD_Struct* LCDx, LCD_ChargePumpContrast_TypeDef CPContranst);
void LCD_SetChargePumpDivider(LCD_Struct* LCDx, LCD_ChargePumpClcokDivider_TypeDef CPClockDivx);
void LCD_SetChargePumpCount(LCD_Struct* LCDx, uint32_t CPCount);
DRV_Return LCD_ChargePump_Cmd(LCD_Struct* LCDx, FunctionalState CPState);
void LCD_ChargePumpSync_Cmd(LCD_Struct* LCDx, FunctionalState CPSyncState);

void LCD_SetPinOffState(LCD_Struct* LCDx, LCD_PinOffState_TypeDef OffState);
void LCD_ReferenceResister(LCD_Struct* LCDx, LCD_ResisterLadder_TypeDef RLSelect);
void LCD_BiasVoltege(LCD_Struct* LCDx, uint32_t BiasSelect);

/**
  * @}
  */


/** @name LCD_Exported_Functions_Mode
  * @{
  */

void LCD_FrameMode(LCD_Struct* LCDx, LCD_FrameMode_TypeDef TypeSelect);
void LCD_DutyCycle(LCD_Struct* LCDx, LCD_DutyCycle_TypeDef DCSelect);
void LCD_WaveformInverse(LCD_Struct* LCDx, uint32_t InvSelect);

void LCD_DeadTimeMode(LCD_Struct* LCDx, LCD_DeadTime_TypeDef DTMode);
void LCD_DeadTimePeriodWidth(LCD_Struct* LCDx, uint32_t DTPeriodWidth);

void LCD_DriveStrengthMode(LCD_Struct* LCDx, LCD_DriveStrengthMode_TypeDef DSMode);
void LCD_HighDrivePulseWidth(LCD_Struct* LCDx, uint32_t DriveHighPulseWidth);

/**
  * @}
  */


/** @name LCD_Exported_Functions_Interript & Event
  * @{
  */

#define __LCD_IT_Enable(LCDx__, LCD_ITSrc__)    ((LCDx__)->INT.W |= (LCD_ITSrc__))
#define __LCD_IT_Disable(LCDx__, LCD_ITSrc__)   ((LCDx__)->INT.W &= (~(LCD_ITSrc__)))
void LCD_IT_Config(LCD_Struct* LCDx, uint32_t LCD_ITSrc, FunctionalState State);
#define  LCD_IT_Cmd     LCD_IT_Config

#define __LCD_ITEA_Enable(LCDx__)               ((LCDx__)->INT.B[0] |= LCD_CR0_EN_enable_b0)
#define __LCD_ITEA_Disable(LCDx__)              ((LCDx__)->INT.B[0] &= (~LCD_CR0_EN_enable_b0))
void LCD_ITEA_Cmd(LCD_Struct* LCDx, FunctionalState State);

#define __LCD_GetITSource(LCDx__)               ((LCDx__)->INT.W)
uint32_t LCD_GetITSource(LCD_Struct* LCDx);

#define __LCD_GetAllFlagStatus(LCDx__)          ((LCDx__)->STA.W)
uint32_t LCD_GetAllFlagStatus(LCD_Struct* LCDx);

#define __LCD_GetFlagStatus(LCDx__, LCD_FLAG__) ((((LCDx__)->STA.W & (LCD_FLAG__)) == (LCD_FLAG__))? SET : CLR)
DRV_Return LCD_GetFlagStatus(LCD_Struct* LCDx, uint32_t LCD_FLAG);

#define __LCD_ClearFlag(LCDx__, LCD_FLAG__)     ((LCDx__)->STA.W = (LCD_FLAG__))
void LCD_ClearFlag(LCD_Struct* LCDx, uint32_t LCD_FLAG);

/**
  * @}
  */


/** @name LCD_Exported_Functions_Start_STOP
  * @{
  */

DRV_Return LCD_Start(LCD_Struct* LCDx);
void LCD_Stop(LCD_Struct* LCDx);
void LCD_Cmd(LCD_Struct* LCDx, FunctionalState LCDState);

/**
  * @}
  */


/** @name LCD_Exported_Functions_Display Memory
  * @{
  */

void LCD_IOConfigCOMSEG(LCD_Struct* LCDx, uint32_t* MUXn);
void LCD_WriteData(LCD_Struct* LCDx, uint32_t* WData);
void LCD_ClearMemory(LCD_Struct* LCDx);
void LCD_DisplayCTL(LCD_Struct* LCDx, LCD_SegmentOff_TypeDef SegmentCTL);

/**
  * @}
  */


/** @name LCD_Exported_Functions_Blink
  * @{
  */

void LCD_SetBlinkDivider(LCD_Struct* LCDx, LCD_BlinkClcokDivider_TypeDef BDivider);
void LCD_SetBlinkCount(LCD_Struct* LCDx, uint32_t BCount);
void LCD_BlinkMode_Config(LCD_Struct* LCDx, LCD_BlinkMode_TypeDef BMode);
DRV_Return LCD_GetBlinkMode(LCD_Struct* LCDx);

/**
  * @}
  */

/**
  * @}
  */


/* Private defines -----------------------------------------------------------*/

/** @name LCD_Private_Functions
  * @{
  */

/**
  * @}
  */


/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __TEPMLATE_H */



#endif

