
/**
 ******************************************************************************
 *
 * @file        BSP_ARGB_WS2812.c
 * @brief       This is ARGB_WS2812 C file(use driver).
 
 * @par         Project
 *              MG32
 * @version     V1.03
 * @date        2023/02/03
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2022 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/*==============================================================================
                                 User Notes
How To use this function:
-----------------------
   1. Use BSP_ARGB_Init() function to initial.
   2. By calling "BSP_ARGB_Update" function to refresh ARGB LED.

Driver architecture:
--------------------
   + MG32_GPIO_DRV
   + MG32_URT_DRV
   
Known Limitations:
------------------
   1. In transmiting ARGB data byte to byte interval time can't above 50us
      (ARGB reset time).      
   2. UART baud-rate must be 9.6MHz.
   
Require parameter
------------------
    Require module : CSC / GPIO / URT0
    
    GPIO pin configuration : 
        Pin / IO mode / AFS
        ---  --------  -----
        PA5 / PPO     / URT0_TX

    URT0 Module : 
        Mode              : Normal mode + 8bit data + parity bit (always High) + 2bit stop.
		Baud-rate         : 9.6MHz in (URT clock source is 48MHz)
        
Example codes:
------------------

==============================================================================*/



/* Includes ------------------------------------------------------------------*/
#include "BSP_2_ARGB_WS2812.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
// GPIO (PA5, PPO, AFS=URT0_RX)
#define ARGB_PIN                   PX_Pin_5
#define ARGB_PIN_MODE              PINX_Mode_PushPull_O
#define ARGB_PIN_AFS               11
#define ARGB_IOM                   IOMA

// URT module
#define  ARGB_URTn                 URT0                     /*!< Define using URT.*/
#define  ARGB_URTn_IRQ             URT0_IRQn                /*!< Define Interrupt Number.*/
#define  BSP_ARGB_URTn_IRQHandler  URT0_IRQHandler          /*!< Define IRQHandler.*/
    
// ARGB 
#define ARGB_TRES                  50                       /*!< Reset time (min time).*/
    
// For ARGB WS2812  
#define ARGB_LOGIC_URT_0           0x1F                     /*!< WS2812 logic 0 data proportion for URT.*/
#define ARGB_LOGIC_URT_1           0x07                     /*!< WS2812 logic 1 data proportion for URT.*/
    
// For ARGB WS2812B 
//#define ARGB_LOGIC_URT_0           0x3F                   /*!< WS2812B logic 0 data proportion.*/
//#define ARGB_LOGIC_URT_1           0x00                   /*!< WS2812B logic 1 data proportion.*/
//                                                              
// For SK6812                                                  
//#define ARGB_LOGIC_URT_0           0x1F                   /*!< SK6812 logic 0 data proportion.*/
//#define ARGB_LOGIC_URT_1           0x07                   /*!< SK6812 logic 1 data proportion.*/

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/ 
static uint8_t  *ARGB_updateBuf;
static uint8_t  ARGB_DataBitCmp;                            /*!< Just ARGB data bit index (keep function). Emulates the ARGB protocol using URT. */
static uint32_t ARGB_TotalCount;
static uint32_t ARGB_DataCount;
static uint8_t  ARGB_TXData;

/* Private function prototypes -----------------------------------------------*/
static void BSP_ARGB_IRQHandler(void);
    
void BSP_ARGB_URTn_IRQHandler(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/
                                                           
/**
 *******************************************************************************
 * @brief	    UART0 IRQHandle function.
 * @details     Just weak function.
 * @return      None
 * @exception   No 
 * @note        No
 *******************************************************************************
 */  
__WEAK void BSP_ARGB_CpltCallback(void) 
{
    //=========================================================
    //NOTE : This function should not be modified , when the 
    //       callback is needed, the MID_URT_ErrorCallback can
    //       be implemented in the user file.
}
/**
 *******************************************************************************
 * @brief	    UART0 IRQHandle function.
 * @details     Jump to BSP_ARGB_IRQHandler routine.
 * @return      None
 * @exception   No 
 * @note        No
 *******************************************************************************
 */                                                          
void BSP_ARGB_URTn_IRQHandler(void)
{
    BSP_ARGB_IRQHandler();
}

/**
 *******************************************************************************
 * @brief	    ARGB Initial
 * @details     Use UART to control ARGB.
 * @return      None
 * @exception   No 
 * @note        No
 *******************************************************************************
 */
void BSP_ARGB_Init(void)
{
    URT_BRG_TypeDef  URT_BRG;
    URT_Data_TypeDef DataDef;
    PIN_InitTypeDef  ARGB_PINX;
    
    // ------------------------------------------------------------------------
    // 1. CSC Initial :
    //    *. The URT sample need URT clock source is 48MHz.
    //    *. CSC Initial :
    //        1. Checked CSC checkbox of MG32_ChipInit in Manage Run-Time Environment.
    //        2. In MG32_ChipInit_Wizard of Project window use MG32_CSC_Init.h file
    //           to select that you want CSC environment (APB - 48MHz , and AHB <- 24MHz).
    //
    // ------------------------------------------------------------------------
    // 2. GPIO Initial can :
    //    1. Checked GPIO checkbox of MG32_ChipInit in Manage Run-Time Environment.
    //       In MG32_ChipInit_Wizard of Project window use MG32_GPIO_Init.h file
    //       to select that you want GPIO environment.
    //    or
    //    2. Use GPIO_PinMode_Config to setting GPIO mode.
    // ------------------------------------------------------------------------
    ARGB_PINX.PINX_Pin                = ARGB_PIN;
    ARGB_PINX.PINX_Mode               = ARGB_PIN_MODE;
    ARGB_PINX.PINX_PUResistant        = PINX_PUResistant_Disable;
    ARGB_PINX.PINX_Speed              = PINX_Speed_Low;
    ARGB_PINX.PINX_Inverse            = PINX_Inverse_Disable;
    ARGB_PINX.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    ARGB_PINX.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    ARGB_PINX.PINX_Alternate_Function = ARGB_PIN_AFS; 
    
    GPIO_PortMode_Config(ARGB_IOM ,&ARGB_PINX);
                                                 
    // ------------------------------------------------------------------------
    // URT Initial.
    // ------------------------------------------------------------------------
    // Set Baud-Rate generator
    URT_BRG.URT_InternalClockSource = URT_BDClock_PROC;
    URT_BRG.URT_BaudRateMode = URT_BDMode_Separated;
    URT_BRG.URT_PrescalerCounterReload = 0;	                //Set PSR
    URT_BRG.URT_BaudRateCounterReload  = 0;	                //Set RLR
        
    URT_BaudRateGenerator_Config(ARGB_URTn, &URT_BRG);      //Baudrate = f(CK_URTx)/(PSR+1)/(RLR+1)/(OS_NUM+1)
    URT_BaudRateGenerator_Cmd(ARGB_URTn, ENABLE);	        //Enable BaudRateGenerator
    
    // ------------------------------------------------------------------------
    // TX/RX Clock
    URT_TXClockSource_Select(ARGB_URTn, URT_TXClock_Internal);  //URT_TX use BaudRateGenerator
    URT_RXClockSource_Select(ARGB_URTn, URT_RXClock_Internal);  //URT_RX use BaudRateGenerator
    URT_TXOverSamplingSampleNumber_Select(ARGB_URTn, 4);	    //Set TX OS_NUM
    URT_RXOverSamplingSampleNumber_Select(ARGB_URTn, 4);	    //Set RX OS_NUM
    URT_RXOverSamplingMode_Select(ARGB_URTn, URT_RXSMP_3TIME);  
    URT_TX_Cmd(ARGB_URTn, ENABLE);	                            //Enable TX
    URT_RX_Cmd(ARGB_URTn, ENABLE);	                            //Enable RX (for Timeout timer )

    // ------------------------------------------------------------------------
    // Set Data character config
    DataDef.URT_TX_DataLength  = URT_DataLength_8;          // Data bit is 8bit.
    DataDef.URT_RX_DataLength  = URT_DataLength_8;          
    DataDef.URT_TX_DataOrder   = URT_DataTyped_MSB;         // Data order is MSB.
    DataDef.URT_RX_DataOrder   = URT_DataTyped_MSB;         
    DataDef.URT_TX_Parity      = URT_Parity_All_H;          // Parity bit Enable and alway is H
    DataDef.URT_RX_Parity      = URT_Parity_All_H;          
    DataDef.URT_TX_StopBits    = URT_StopBits_2_0;          // Stop bit is 2=bit.
    DataDef.URT_RX_StopBits    = URT_StopBits_2_0;          
    DataDef.URT_TX_DataInverse = DISABLE;                   // Data bit no invert.
    DataDef.URT_RX_DataInverse = DISABLE;
    
    URT_DataCharacter_Config(ARGB_URTn, &DataDef);
    
    // ------------------------------------------------------------------------
    // Set Mode Select
    URT_Mode_Select(ARGB_URTn, URT_URT_mode);
    
    // ------------------------------------------------------------------------
    // Set DataLine Select And Pin Control.
    URT_DataLine_Select(ARGB_URTn, URT_DataLine_2);
    URT_RxTxSwap_Cmd(ARGB_URTn , ENABLE);                   // RX / TX Pin Swap.
    
    // ------------------------------------------------------------------------
    // Set Data Control
    URT_TXGuardTime_Select(ARGB_URTn, 0);    
    URT_TXInverse_Cmd(ARGB_URTn, ENABLE);                   // TX output signal invert.
    
    // ------------------------------------------------------------------------
    // TRst time setting. (Use timeout timer of URT )
    URT_TimeoutMode_Select(ARGB_URTn, URT_TMOMDS_General);              
    URT_TimeoutTimerClockSource_Select(ARGB_URTn, URT_TMOCLK_BitTime);
    URT_SetTimeoutTimerCompare(ARGB_URTn, (ARGB_TRES *12));  
    URT_TimeroutTimer_Cmd(ARGB_URTn, DISABLE);
    
    // ------------------------------------------------------------------------
    // Enable URT Interrupt
    URT_IT_Config(ARGB_URTn, (URT_IT_TMO | URT_IT_TX | URT_IT_TC), DISABLE);
    URT_IT_Config(ARGB_URTn, URT_IT_UG , ENABLE);
    URT_ITEA_Cmd( ARGB_URTn, ENABLE);
    NVIC_EnableIRQ(ARGB_URTn_IRQ );
    
    // ------------------------------------------------------------------------
    // Enable URT
    URT_Cmd(ARGB_URTn, ENABLE);       
    
}
/**
 *******************************************************************************
 * @brief	    Update ARGB status.
 * @details     Use uart to update ARGB.
 * @param[in]   sARGB : Control ARGB data line .
 * @param[in]   UpDate_ARGBCount : Update ARGB pieces.   
 * @return      None
 * @exception   No 
 * @note        No
 *******************************************************************************
 */
void BSP_ARGB_SendData(uint8_t *ARGB_Buf , uint32_t UpDate_ARGBCount )
{
    /* Set ARGB handle parameter. */
    ARGB_updateBuf    = ARGB_Buf;
    ARGB_DataBitCmp   = 0;
    ARGB_TotalCount   = (UpDate_ARGBCount * 3);
    ARGB_DataCount    = 0;
    
    /* Start update */
    WRITE_REG(ARGB_URTn->STA.W, (URT_IT_TMO | URT_IT_UG));  // URT_IT_TMO : Timeout timer timout.
                                                            // URT_IT_UG: URT genernal event relationship interrupt.
    
    SET_BIT(ARGB_URTn->TMOUT.W, URT_TMOUT_TMO_RST_mask_w);  // force reset timeout timer
    SET_BIT(ARGB_URTn->CR4.W, URT_CR4_TDAT_CLR_mask_w);     // transmitted data buffer will be flushed.
    
    
    WRITE_REG(ARGB_URTn->INT.W, (URT_IT_TMO | URT_IT_UG | URT_INT_IEA_mask_w)); // Enable interrupt {TMO | UG | IEA}.
    SET_BIT(ARGB_URTn->TMOUT.W, URT_TMOUT_TMO_EN_mask_w);   // Enable URT's timeout timer.
}
/**
 *******************************************************************************
 * @brief	    Handle ARGB (URT) interrupt request.
 * @details     Write ARGB data via URT.
 * @return      None   
 * @exception   No 
 * @note        No
 *******************************************************************************
 */
static void BSP_ARGB_IRQHandler(void)
{
    uint32_t URTIRQ_Flag;
    uint8_t  DataBitCmp;
    ctype    DataBit;

    // ------------------------------------------------------------------------
    // URTIRQ_Flag=STA & INT
    // Only INT enable bit equal STA flag.
    // ------------------------------------------------------------------------
    URTIRQ_Flag = READ_REG(ARGB_URTn->STA.W) & READ_REG(ARGB_URTn->INT.W);
    
    // ------------------------------------------------------------------------
    // In transmitting.
    // ------------------------------------------------------------------------
    if(URTIRQ_Flag & URT_IT_TX)                             // URT_IT_TX : Transmit data register empty.
    {
        // --------------------------------------------------------------------
        // Transmit bit7 ~ bit4.
        if(ARGB_DataBitCmp == 0x00)
        {
            ARGB_TXData = ARGB_updateBuf[ ARGB_DataCount];  // reflesh transmitted data           
            DataBitCmp  = 0x80;                             // reflesh bit index (0x80->0x40,0x20,0x10,0x08,0x04,0x02,0x01)       
        }
        // --------------------------------------------------------------------
        // Transmit bit3 ~ bit0.
        else
        {
            DataBitCmp      = ARGB_DataBitCmp; 
            ARGB_TotalCount = ARGB_TotalCount - 1;
            ARGB_DataCount  = ARGB_DataCount  + 1;
        }
        
        // --------------------------------------------------------------------
        // Regroup data. (bit 7~4) or (bit 3~0)
        // --------------------------------------------------------------------
        // DataBit.B[0]
        if((ARGB_TXData & DataBitCmp) == 0)                 // for bit 7 or bit 3
        {
            DataBit.B[0] = ARGB_LOGIC_URT_0;                // ARGB_LOGIC_URT_0=0x1F (WS2812 logic 0 data proportion for URT)
        }
        else
        {
            DataBit.B[0] = ARGB_LOGIC_URT_1;                // ARGB_LOGIC_URT_1=0x07 (WS2812 logic 1 data proportion for URT)
        }
        
        // --------------------------------------------------------------------
        // DataBit.B[1]
        DataBitCmp = DataBitCmp >> 1;
        if((ARGB_TXData & DataBitCmp) == 0)                 // for bit 6 or bit 2
        {
            DataBit.B[1] = ARGB_LOGIC_URT_0;
        }
        else
        {
            DataBit.B[1] = ARGB_LOGIC_URT_1;
        }
        
        // --------------------------------------------------------------------
        // DataBit.B[2]
        DataBitCmp = DataBitCmp >> 1;
        if((ARGB_TXData & DataBitCmp) == 0)                 // for bit 5 or bit 1
        {
            DataBit.B[2] = ARGB_LOGIC_URT_0;
        }
        else
        {
            DataBit.B[2] = ARGB_LOGIC_URT_1;
        }
        
        // --------------------------------------------------------------------
        // DataBit.B[3]
        DataBitCmp = DataBitCmp >> 1;
        if((ARGB_TXData & DataBitCmp) == 0)                 // for bit 4 or bit 0
        {
            DataBit.B[3] = ARGB_LOGIC_URT_0;
        }
        else
        {
            DataBit.B[3] = ARGB_LOGIC_URT_1;
        }
        
        // --------------------------------------------------------------------
        DataBitCmp = DataBitCmp >> 1;
        
		WRITE_REG(ARGB_URTn->TDAT.W , DataBit.W);           // Write ARGB data via URT (Emulates ARGB potocol)
        ARGB_DataBitCmp = DataBitCmp;                       // save bit index

        // --------------------------------------------------------------------
        // data transmit complete ?
        if(ARGB_TotalCount != 0)
        {
            return;
        }
        // --------------------------------------------------------------------
        // Last data write to TX data.
        else
        {
            // Enable URT interrupt {UG | TC | IEA}
            WRITE_REG(ARGB_URTn->INT.W , ((URT_IT_UG | URT_IT_TC | URT_INT_IEA_mask_w)));
        }
    }
    // ------------------------------------------------------------------------
    // Reset time handle.
    // ------------------------------------------------------------------------
    else if(URTIRQ_Flag & URT_IT_TMO)
    {
        // Disable Timeout timer
        CLEAR_BIT(ARGB_URTn->TMOUT.W , URT_TMOUT_TMO_EN_mask_w); 
        // Clear Timout & UG flag
        WRITE_REG(ARGB_URTn->STA.W   , (URT_IT_TMO | URT_IT_UG));
        // Enable URT interrupt {UG | TC | IEA}
        WRITE_REG(ARGB_URTn->INT.W   , (URT_IT_UG  | URT_IT_TX | URT_INT_IEA_mask_w));
        
        return;
    }
    
    // ------------------------------------------------------------------------
    // Wrap up transmission ?
    // ------------------------------------------------------------------------
    if(READ_REG(ARGB_URTn->STA.W & URT_IT_TC))              // (URT_STA_TCF_mask_w) Transmission complete interrupt?
    {
        // Enable URT_INT (URT genernal event relationship interrupt + IEA)
        WRITE_REG(ARGB_URTn->INT.W , (URT_IT_UG | URT_INT_IEA_mask_w)); 
        // Run BSP_ARGB_CpltCallback routine.
        BSP_ARGB_CpltCallback();
        
        return;
    }
    
}








