
/**
 ******************************************************************************
 *
 * @file        main.c
 * @brief       The main C file.
 *              
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2025/07/14
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 *******************************************************************************
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 */
 
/*==============================================================================
                                 User NOTES
Please reference : https://www.waveshare.net/wiki/Pico-ePaper-2.13
                                 
WAVESHARE 2.13" E-Paper Module
------------------------------
    1. Partial Refresh Support
    2. Low Power
    3. Wide Viewing Angle : > 170 degree
    4. Paper-Like Effect
    5. Resolution : 250x122
    6. Display Color : Black and White (Grey Scale : 2)
    7. Size : 2.13"
    
E-Paper character
-----------------
    1. Communicatiom : SPI
    2. Display Panel : E-paper
    3. Experience : Paper-like
    4. Comfortably : Eye Care, No Blue ...
    5. Enviroment : Ambient Light Request
    6. Display Type : Passively Reflective
    7. Power Consumption : Ultra Low
    8. Display Duration : Persists whithout Power
    9. Refreshing : Partial Refresh 
    
Features At A Glance
--------------------
    1. No backlight, keeps displaying last content for a long time even when power down
    2. Ultra low power consumption, basically power is only required for refreshing
    3. SPI interface, requires minimal IO pins
    4. Comes with development resources and manual (Raspberry Pi Pico C/C++ and MicroPython examples)
    
Pinout Definition (Board :Pico-ePaper-2.13)
------------------------------------------
    VSYS                : Power input (3.3V / 5V)
    GND                 : Ground
    GP8 (E-Paper DC)    : Data/Command control pin (high for data, low for command)
    GP9 (E-Paper CS)    : SPI chip select (low active)
    GP10 (E-Paper CLK)  : SPI SCK pin
    GP11 (E-Paper DIN)  : SPI MOSI pin
    GP12 (E-Paper RST)  : External reset (low active)
    GP13 (E-Paper BUSY) : Busy status output

                                 
How To use this function:
-----------------------
   + EPD_2in13_Full_Init() - Initial WaveShare 2.13" E-Paper with all screen.
   + EPD_2IN13_Draw_All_Screen(uint8_t) - Display an all screen image on WaveShare 2.13" E-Paper.
   
   
Driver architecture:
--------------------
   + MG32_GPIO_DRV
   + MG32_SPI_DRV

Known Limitations:
------------------
   1- Run in 48MHz. (CSC control).
   2- Enable SysTick in IRQ_Handler.
==============================================================================*/

/*==============================================================================
                                 Require parameter
Require module : CSC / SPI / GPIO 

CSC Module :
    Clock source                : CK_IHRCO (12M Hz)
    Clock frequency             : CK_MAIN select CK_PLLO (PLL to 48MHz)
    CK_APB frequency            : 48MHz
    CK_AHB frequency            : 48MHz
    Clock enable in 'On Mode'   : DAC / GPIOB / DMA / TM10

GPIO pin configuration : 
    Pin / IO mode / AFS
    ---  --------  -----
    PB0 / PPO     / GPB0 (Default = Software SPI CS)
    PB2 / PPO     / SPI0_CLK
    PB3 / PPO     / SPI0_MOSI
    PB4 / PPO     / GPB4 (Default = E-Paper DC : Data/Command control pin (high for data, low for command))
    PB5 / PPO     / GPB5 (Default = E-Paper RST : External reset (low active))
    PB6 / DIN     / GPB6 (Default = E-Paper BUSY : Busy status output)
           
SPI Module :                   
    Master SPI
    SPI CLk bit time : ~2M Hz
    CHPA = CPOL = 0
    
==============================================================================*/
/* USER CODE END Header */

/* Includes ------------------------------------------------------------------*/
#include "MG32_MID.h"
#include "RTE_Components.h"
#include "EPD_2in13V3_Function.h"


/* External function prototypes ----------------------------------------------*/
extern void ChipInit(void);

/* Private includes ----------------------------------------------------------*/

/* Private typedef -----------------------------------------------------------*/

/* Private define ------------------------------------------------------------*/
// uintptr_t: Conversion to 'void *'
#define UN_CONSTIFY(_t, _v) ((_t)(uintptr_t)(_v))


/* Private macro -------------------------------------------------------------*/

/* Private variables ---------------------------------------------------------*/
extern uint8_t EPD_rawData_20230508[];          // Size : 122 x 250
extern uint8_t EPD_MCU_20231107[];              // Size : 80 x 70
extern uint8_t Font24_Table[];                  // Size : 17 x 24


/* Private function prototypes -----------------------------------------------*/

/* Private user code ---------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief  The application entry point.
 * @retval int
 *******************************************************************************
 */
int main(void)
{
    uint8_t *ImagePtr8;

    // ------------------------------------------------------------------------
    // chip initial (User can enable CSC, GPIO, TM, ADC, EXIC ... wizard)
    // ------------------------------------------------------------------------
    // MG32_CSC_Init.c / .h : IHRCO + PLLO to 48MHz, APB=CK_MAIN/1=24M, AHB=24M
    // MG32_GPIO_Init.c / MG32_GPIO_LQFP80U_Init.h : Handles I/O configuration.
    // MG32_SPI_Init.c / .h : Handles SPI0 configuration.
    // ------------------------------------------------------------------------
    ChipInit();

    // ------------------------------------------------------------------------
    // Initializes the Systick. 
    // ------------------------------------------------------------------------
    MID_Init();

    // ------------------------------------------------------------------------
    // Initial EPD with full screen to desplay the base image.
    // ------------------------------------------------------------------------
    EPD_2in13_V3_Full_Init();

    // ------------------------------------------------------------------------
    // Display the full screen image.
    // ------------------------------------------------------------------------
    EPD_2IN13_V3_Draw_All_Screen(UN_CONSTIFY(uint8_t *, &EPD_rawData_20230508));
    
    
    // ------------------------------------------------------------------------
    // countdown function : Show '3' -> '2' -> '1' 
    // ------------------------------------------------------------------------
    for (uint8_t CNT=0; CNT < 3; CNT++)
    {
        // --------------------------------------------------------------------
        // Initial E-Paper for partial screen.
        // --------------------------------------------------------------------
        ImagePtr8 = (uint8_t *) &Font24_Table;
        if (CNT == 0)
            // Show '3' text character
            ImagePtr8 += (24 * 3) * ('3' - 0x20);
        else if (CNT == 1)
            // Show '2' text character
            ImagePtr8 += (24 * 3) * ('2' - 0x20);
        else if (CNT == 2)
            // Show '1' text character
            ImagePtr8 += (24 * 3) * ('1' - 0x20);
        
        // --------------------------------------------------------------------
        // Show text character (turn right 90 degrees for font24.c)
        // Function: EPD_2IN13_V3_Draw_Font17x24 (XStart, YStart, Bitmap of Character)
        //  XStart address must be multiples of 8.
        // --------------------------------------------------------------------
        EPD_2IN13_V3_Draw_Font17x24(24, 160+30, ImagePtr8);
        
        // --------------------------------------------------------------------
        // fully refresh EPD once
        // --------------------------------------------------------------------
        EPD_RefreshDisplay();
        
        // --------------------------------------------------------------------
        MID_Delay(500);        
    } 
    
    // ------------------------------------------------------------------------
    // Display part of the image.
    // ------------------------------------------------------------------------
    EPD_2IN13_V3_Draw_Partial_Pattern(0,160, 80-1, 70-1, UN_CONSTIFY(uint8_t *, &EPD_MCU_20231107));
    
    // ------------------------------------------------------------------------
    // fully refresh EPD once
    // ------------------------------------------------------------------------
    EPD_RefreshDisplay();

    // ------------------------------------------------------------------------
    // Enter Sleep Mode for E-Paper.
    // ------------------------------------------------------------------------
    EPD_2in13_V3_EntrySLEEP();
    
    
    while (1)
    {
        // to do ...
    }
}

/************************ (C) COPYRIGHT megawin ****************END OF FILE****/



