/**
  ******************************************************************************
 *
 * @file        Sample_TM36_IC3_DMA.c
 *
 * @brief       Input Capture for TM36 IC3 with DMA access.
 *
 * @par         Project
 *              MG32
 * @version     V1.12
 * @date        2022/04/14
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
* @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_TM_DRV.h"
#include "MG32_DMA_DRV.h"


/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
// user define capture quantity
#define CaptureNtimes   10

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static uint16_t TM36_IC3_CaptureData[CaptureNtimes];

/* Private function prototypes -----------------------------------------------*/
void Sample_TM36_IC3_DMA(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief	    Capture main counter value when happen External Trigger event 
 * @details     1.TimeBase structure initial
 *    \n        2.config TM36 channel 3 function 
 *    \n        3.config DMA (P2M) (TM36_IC3 to SRAM)
 *    \n        4. Enable DMA & Set DMA request
 *    \n        5.Enable Timer  
 * @return      None
 * @code
                uint16_t CaptureData;
                
                // initial IC3 (PB7) 
                // to do ...
                
                // config TM36 for IC3-16 bit mode 
                Sample_TM36_IC3_DMA();
                
                // wait for DMA complete event
                while(DMA_GetChannelFlag(DMAChannel0, DMA_Chx_TCF) == DRV_UnHappened);
                
                // Get IC3 capture data (16-bits), total 10 times
                CaptureData = TM36_IC3_CaptureData[0];
                CaptureData = TM36_IC3_CaptureData[1];
                CaptureData = TM36_IC3_CaptureData[2];
                ...
                CaptureData = TM36_IC3_CaptureData[9];

                
 * @endcode               
 * @note        User be careful of pulse width.
 *              The pulse width must large 32T AHB clock. This is the DMA access
 *              time (2 bytes). 
 *******************************************************************************
 */
void Sample_TM36_IC3_DMA(void)
{  
    TM_TimeBaseInitTypeDef TM_TimeBase_InitStruct;
    DMA_BaseInitTypeDef    TM_IC3_DMA;

    // make sure :
	
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12M
    //Select CK_MAIN Source = CK_HS
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->TM36 = Enable
    //Configure Peripheral On Mode Clock->Port B = Enable
	
    //==Set GPIO init 
    //TM36_IC3 pin config:
    //MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin7
    //GPIO port initial is 0xFFFF
    //PB7 mode is ODO
    //PB7 function TM36_IC3
		 
    TM_DeInit(TM36);
    
    // ----------------------------------------------------
    // 1.TimeBase structure initial
    TM_TimeBaseStruct_Init(&TM_TimeBase_InitStruct);
    
    // modify parameter
    TM_TimeBase_InitStruct.TM_MainClockDirection    = TM_UpCount;
    TM_TimeBase_InitStruct.TM_Period                = 65535; 
    TM_TimeBase_InitStruct.TM_Prescaler             = 65535;
    TM_TimeBase_InitStruct.TM_CounterMode           = Separate;
    
    TM_TimeBase_Init(TM36, &TM_TimeBase_InitStruct);
   
    // ----------------------------------------------------
    // 2.config TM36 channel 3 function 
    TM_CH3Function_Config(TM36, TM_InputCapture);
    TM_IN3Source_Select(TM36, TMx_InputMUX_Pin);
    TM_IN3TriggerEvent_Select(TM36, IC_DualEdge);
    
    // ----------------------------------------------------
    // 3.config DMA (P2M) (TM36_IC3 to SRAM)
    DMA_Cmd(ENABLE);
    
    // clear DMAChannel0 flags
    DMA_ClearChannelFlag(DMAChannel0, (DMA_ChannelFlagDef) (DMA_Chx_TCF | DMA_Chx_THF));
    
    // initial DMA struct
    DMA_BaseInitStructure_Init(&TM_IC3_DMA);
    {   // modify parameter
        TM_IC3_DMA.DMAChx               = DMAChannel0;
        // P2M (TM36_IC3 to SRAM)
        TM_IC3_DMA.SrcSymSel            = DMA_TM36_IC3;
        TM_IC3_DMA.DestSymSel           = DMA_MEM_Write;
        
        // declare destination address
        TM_IC3_DMA.DMADestinationAddr   = &TM36_IC3_CaptureData;
        TM_IC3_DMA.DestDINCSel          = ENABLE;
        
        // another DMA parameters
        TM_IC3_DMA.DMALoopCmd           = DISABLE;
        TM_IC3_DMA.DMATransferNUM       = (CaptureNtimes * 2);  
        TM_IC3_DMA.BurstDataSize        = DMA_BurstSize_2Byte;
    }
    // inital DMAChannel0 
    DMA_Base_Init(&TM_IC3_DMA);
    
    DMA_Channel_Cmd(DMAChannel0, ENABLE);
    
    // ----------------------------------------------------
    // 4. Enable DMA & Set DMA request
    TM_DMAChannel_Cmd(TM36, (uint32_t) TMx_DMA_IC3, ENABLE);       // Enable TM36 IC3 DMA
    DMA_StartRequest(DMAChannel0);
    
    // ----------------------------------------------------
    // 5.enable Timer 
    TM_Timer_Cmd(TM36,ENABLE);

    return;
}



