

/**
 ******************************************************************************
 *
 * @file        BSP_RotaryEncoder.c
 * @brief       This is Rotary Encoder C file.
 
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2022/06/28
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2022 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/*==============================================================================
                                 User Notes
How To use this function:
-----------------------
    1. Use BSP_RotaryEncoder_Init() to initial.
    2. Periodic call BSP_RotaryEncoder_main() function to update Rotary encoder 
       status.
       (1) If Rotary_CTR.ChangeFlag = 0 means Rotary Encoder status no change.
       (2) If Rotary_CTR.ChangeFlag & ROTARY_CHANGEFLAG_BUTTON no equal zero
           means button status change. Read Rotary_CTR.SWButton_Status to get
           button status ( 0 = make, 1 = break).
       (3) If Rotary_CTR.ChangeFlag & ROTARY_CHANGEFLAG_ENCODE no equal zero
           means button status change. Read Rotary_CTR.Encoder_Status to get 
           rotary encoder status ( positive value is clockwise n step, 
           negative value is anticlockwise n step). 
           
Driver architecture:
--------------------
    + MG32_GPIO_DRV
    + MG32_TM_DRV
   
Known Limitations:
------------------

Require parameter
------------------
    Require module : CSC / GPIO / TM26
    
    GPIO pin configuration : 
        Pin  / IO mode / AFS
        ---  --------  -----
        PD13 / DIN     / GPIO
        PD14 / DIN     / TM26_IC0
        PD15 / DIN     / TM26_IC1

    TM26 Module :
        Timer Mode   : Cascade Mode.
        Control Mode : QEI Control IN0/1 Both Trigger Mode.

Example codes:
------------------

==============================================================================*/


/* Includes ------------------------------------------------------------------*/
#include "BSP_9_RotaryEncoder.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
//GPIO
#define ROTARYENCODER_PHA_PIN        PIND(14)
#define ROTARYENCODER_PHB_PIN        PIND(15)
#define ROTARYENCODER_SW_PIN         PIND(13)
                                     
#define ROTARYENCODER_PHA_AFS        7
#define ROTARYENCODER_PHB_AFS        7
#define ROTARYENCODER_SW_AFS         0
                                     
#define ROTARYENCODER_MODE           PINX_Mode_Digital_I
#define ROTARYENCODER_IOM            IOMD
                                     
#define ROTARYENCODER_SW             PD13

//Timer
#define ROTARYENCODER_TM             TM26
       
//Rotary Encoder control       
#define ROTARYENCODER_COUNT_MAX      65535
#define ROTARYENCODER_CHANGE_RANGE   10                         /*!< The vaule can't above ROTARYENCODER_COUNT_MAX/2.*/


#define ROTARY_SWSTATUS_MAKE         0                          /*!< Button status is make.*/ 
#define ROTARY_SWSTATUS_BREAK        1                          /*!< Button status is break.*/ 

#define ROTARY_SW_MAKE_DEBOUNCE      10                         /*!< Button make debounce time.*/
#define ROTARY_SW_BREAK_DEBOUNCE     5                          /*!< Button break debounce time.*/

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
RotaryCTR_TypeDef Rotary_CTR;

static uint32_t RotaryEncoder_Cmp;                              /*!< The previous rotation status. Initial state is 0.*/
static uint32_t RotarySWDebounceCounter;

/* Private function prototypes -----------------------------------------------*/
static void BSP_Rotary_GetRotaryEncodeStatus(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/
/**
 *******************************************************************************
 * @brief	    Rotary Encoder initial.
 * @details     
 * @return      
 * @exception   No
 * @note        No
 *******************************************************************************
 */
void BSP_RotaryEncoder_Init(void)
{
    PIN_InitTypeDef         Encoder_GPIOX;
    
    /*GPIO Initial.*/
    //Set Port filter clock source. (AHB / 8)
    GPIO_PortFilterClockSource_Select(ROTARYENCODER_IOM,GPIO_FT_CLK_AHB_DIV8); //The port filter clock source is AHB/8
    //Set PHA , PHB and SW pin mode
    Encoder_GPIOX.PINX_Mode				  = ROTARYENCODER_MODE;
    Encoder_GPIOX.PINX_PUResistant        = PINX_PUResistant_Enable;           //Enable internal pull-up resistance.
    Encoder_GPIOX.PINX_Speed              = PINX_Speed_Low;
    Encoder_GPIOX.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    Encoder_GPIOX.PINX_FilterDivider      = PINX_FilterDivider_16;             /*Set filter divider, so the pins can filter less than 
                                                                                 filter clock source frequence / filter divider signal
                                                                                 (if AHB = 24MHz, can filter less than 5us signal.*/ 
    Encoder_GPIOX.PINX_Inverse            = PINX_Inverse_Disable;
    Encoder_GPIOX.PINX_Alternate_Function = ROTARYENCODER_PHA_AFS;
    
    GPIO_PinMode_Config( ROTARYENCODER_PHA_PIN ,&Encoder_GPIOX); 
    GPIO_PinMode_Config( ROTARYENCODER_PHB_PIN ,&Encoder_GPIOX);
    
    Encoder_GPIOX.PINX_Alternate_Function = ROTARYENCODER_SW_AFS;
    GPIO_PinMode_Config( ROTARYENCODER_SW_PIN ,&Encoder_GPIOX);

    /* TM QEI function initial
       *. According to rotary encoder PHA and PHB state to control prescaler counter + main counter 
          down count or up count.
    */
    TM_DeInit(ROTARYENCODER_TM);
    TM_TimerMode_Select( ROTARYENCODER_TM,Cascade);
    TM_SetCounterDirection( ROTARYENCODER_TM,TM_UpCount);
    TM_SetPrescalerDirection(ROTARYENCODER_TM,TM_UpCount);
    TM_Prescaler_Config(ROTARYENCODER_TM,3,3);                         // Prescaler counter compare value.
    TM_Counter_Config(ROTARYENCODER_TM,0,ROTARYENCODER_COUNT_MAX);     /* Main counter compare value(according to refresh rotary encoder status 
                                                                          frequency to modify this parameter).*/
    
    // TM26 clock setting
    TM_PrescalerClock_Select(ROTARYENCODER_TM,TM_CK_INT);
    TM_CounterClock_Select(ROTARYENCODER_TM,TM_CK_INT);
    TM_ExternalClock_Select(ROTARYENCODER_TM,TM_CKETR);
    TM_InternalClockSource_Select(ROTARYENCODER_TM,TM_PROC);
    TM_SetInternalClockDivider(ROTARYENCODER_TM,TM_IntDIV1);
    
    
    // QEI function initial
    TM_QEI_Select(ROTARYENCODER_TM,TM_BOTH);                           // QEI BOTH mode.         
    TM_InverseQEIDirection(ROTARYENCODER_TM,DISABLE);                  // Disable QEI inverse function
    TM_QEIReset_Cmd(ROTARYENCODER_TM,DISABLE);                         // Disable QEI INDEX function
    TM_IN0Source_Select(ROTARYENCODER_TM,TMx_InputMUX_Pin);            // QEI PHA signal from IC0
    TM_IN1Source_Select(ROTARYENCODER_TM,TMx_InputMUX_Pin);            // QEI PHB signal from IC1
 
    ROTARYENCODER_TM->PSCNT.H[0] = 1;                                  //Set prescaler counter default value.
 
    TM_Timer_Cmd(ROTARYENCODER_TM,ENABLE);

    /*Parameter initial*/
    Rotary_CTR.ChangeFlag      = 0;
    Rotary_CTR.Encoder_Status  = 0;
    Rotary_CTR.SWButton_Status = ROTARY_SWSTATUS_BREAK;
    
    RotaryEncoder_Cmp       = 0;
    RotarySWDebounceCounter = 0;
}
/**
 *******************************************************************************
 * @brief	  Rotary main function
 * @details   According visiting function frequency to control detect RotaryEncoder
              status frequency.
 * @return    
 * @exception No  
 * @note        
 *******************************************************************************
 */
void BSP_RotaryEncoder_main(void)
{
    uint8_t BSP_Rotary_mainTmp;
    
    /*Detect Rotary encoder etatus*/
    BSP_Rotary_GetRotaryEncodeStatus();
    
    /*Detect Rotary button status */
    BSP_Rotary_mainTmp = ROTARYENCODER_SW;
    
    if( Rotary_CTR.SWButton_Status == BSP_Rotary_mainTmp)
    {
        return;
    }
    
    if( Rotary_CTR.SWButton_Status == ROTARY_SWSTATUS_BREAK)
    {
        if( RotarySWDebounceCounter != ROTARY_SW_MAKE_DEBOUNCE)
        {
            RotarySWDebounceCounter = RotarySWDebounceCounter + 1;
            return;
        }
    }
    else if( RotarySWDebounceCounter != ROTARY_SW_BREAK_DEBOUNCE)
    {
        RotarySWDebounceCounter = RotarySWDebounceCounter + 1;
        return;
    }
    
    RotarySWDebounceCounter     = 0;
    Rotary_CTR.SWButton_Status  = BSP_Rotary_mainTmp;
    Rotary_CTR.ChangeFlag      |= ROTARY_CHANGEFLAG_BUTTON;
    
}
/**
 *******************************************************************************
 * @brief	  Get Rotary status  
 * @details     
 * @return    Rotary status:
 *  \n        -1 ~ -32767 : Anti clockwise rotate 1 ~ 32767 step.
 *  \n         1 ~  32767 : Clockwise rotate 1 ~ 32767 step.
 *  \n         0          : No action.
 * @exception   
 * @note        
 *******************************************************************************
 */
static void BSP_Rotary_GetRotaryEncodeStatus(void)
{
    uint32_t BSP_Rotary_GetStatusTMCount;
    int32_t  BPS_Rotary_GetStatusTmp;
    
    /*Get TIme main counter (Rotary : current state)*/
    BSP_Rotary_GetStatusTMCount = TM_GetCounter(ROTARYENCODER_TM);
    
    /*If the same as the previous rotation status then exit.*/
    if( BSP_Rotary_GetStatusTMCount == RotaryEncoder_Cmp)
    {
        return;
    }
    /*Current rotation status > previous rotation status*/
    if( BSP_Rotary_GetStatusTMCount > RotaryEncoder_Cmp)
    {
        BPS_Rotary_GetStatusTmp = (int32_t)(BSP_Rotary_GetStatusTMCount - RotaryEncoder_Cmp);
        
        if( BPS_Rotary_GetStatusTmp > ROTARYENCODER_CHANGE_RANGE)
        {
            BPS_Rotary_GetStatusTmp = (int32_t)(-((ROTARYENCODER_COUNT_MAX - BSP_Rotary_GetStatusTMCount) + RotaryEncoder_Cmp + 1));
        }
    }
    else if( (RotaryEncoder_Cmp - BSP_Rotary_GetStatusTMCount) > ROTARYENCODER_CHANGE_RANGE)
    {
        BPS_Rotary_GetStatusTmp = (int32_t)((ROTARYENCODER_COUNT_MAX - RotaryEncoder_Cmp) + BSP_Rotary_GetStatusTMCount + 1);
    }        
    else
    {
        BPS_Rotary_GetStatusTmp = (int32_t)(-( RotaryEncoder_Cmp - BSP_Rotary_GetStatusTMCount));
    }
    /*Update previous rotation status*/
    RotaryEncoder_Cmp       = BSP_Rotary_GetStatusTMCount;
    
    /*Change Encoder_Status & ChangeFlag*/
    Rotary_CTR.Encoder_Status  =  BPS_Rotary_GetStatusTmp;
    Rotary_CTR.ChangeFlag     |=  ROTARY_CHANGEFLAG_ENCODE;
}










