/**
 ******************************************************************************
 *
 * @file        main.c
 * @brief        
 *
 * @par         Project
 *              MG32F02K128
 * @version     V1.01
 * @date        2025/04/11
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 * the code for TH274A Board
 *     Remove Part R55, R56, R97, R98, R99 and R100 
 *     Adding Part J18, DB09-02(LCD_STN02 Board) and GDC05720S(C8S24) xTN Glass.
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32_ChipInit.h"
#include "MG32_GPIO_DRV.h"
#include "Global.h"
#include "BSP_LED.h"
#include "BSP_Button.h"
#include "BSP_LCD.h"
#include "BSP_VariableResistor.h"
//#include "BSP_ARGB_WS2812.h"
#include "BSP_TemperatureSensor.h"
#include "Calendar.h"
#include "Drama.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
typedef struct
{
    union
    {
        uint8_t  B;
        struct
        {
            uint8_t LCD_UpdateFlag   : 1;
            uint8_t ARGB_UpdateFlag  : 1;
            uint8_t LED_UpdateFlag   : 1;
            uint8_t                  : 5;
        }MBIT;
    }Status;
}UpdateFlag_TypeDef;

/* Private define ------------------------------------------------------------*/
#define LCD_STATUS_ENABLE           0
#define LCD_STATUS_ToBLINK          1
#define LCD_STATUS_BLINK            2
#define LCD_STATUS_ToBREAK          3
#define LCD_STATUS_BLANK            4

#define LCD_BLANK_KEY               0
#define LCD_BLINK_KEY               1
#define LCD_CALENDAR_KEY            2
#define LCD_MEASURE_KEY             3

#define DRAMA_MODE_SPLASH_SCREEN    0
#define DRAMA_MODE_NORMAL           1
#define DRAMA_MODE_SLEEP            2

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static __IO UpdateFlag_TypeDef  UpdateFlag;

static      uint8_t  LCD_Status;
static      uint8_t  LCD_StatusUpdateFlag;

static      uint8_t  Drama_Mode;

static      uint32_t Drama_ActCmp;
static      uint32_t Drama_ActCnt;

static      uint8_t  Drama_LCD_Mode;

static      uint8_t  Drama_ADC_Act;

/* Private function ----------------------------------------------------------*/
void SysTick_Handler(void);
void APX_IRQHandler(void);

static void ADC_Init(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @exception   
 * @note        
 *******************************************************************************
 */ 
void SysTick_Handler(void)
{
    /*Calendar handler*/
    Calendar_Handler();

    /*Button sample handler*/
    BSP_Button_Handler();

    /*ADC related*/
    switch(Drama_ADC_Act)
    {
        case 6: 
                BSP_TemperatureSensor_Handler();
                break;

        case 5: 
                BSP_TemperatureSensor_SkipADCData();
                BSP_TemperatureSensor_Trigger();
                break;

        case 4: 
                BSP_TemperatureSensor_SkipADCData();
                BSP_TemperatureSensor_Trigger();
                break;

        case 3:
                BSP_TemperatureSensor_DischargeTrigger();
                break;

        case 2: 
                /*Variable Resistor handler*/
                BSP_VR_Handler();
                break;

        case 1:
                BSP_VR_SkipADCData();
                BSP_VR_Trigger();
                break;

        case 0:
                BSP_VR_DischargeTrigger();
                break;

        default:
                Drama_ADC_Act = 0;
                break;
    }
    if( Drama_ADC_Act == 7)
    {
        Drama_ADC_Act = 0;
    }
    else
    {
        Drama_ADC_Act ++;
    }

    /*LCD Drama handler*/
    if( Drama_Mode != DRAMA_MODE_SPLASH_SCREEN)
    {
        UpdateFlag.Status.MBIT.LCD_UpdateFlag = Drama_LCD_Handler();
    }
    else if(Drama_ActCnt == Drama_ActCmp)
    {
        UpdateFlag.Status.MBIT.LCD_UpdateFlag = 1;
        Drama_ActCnt = 0;
    }
    else
    {
        Drama_ActCnt ++;
    }

    Drama_LED_Handler();
    Drama_ARGB_Handler();
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @exception   
 * @note        
 *******************************************************************************
 */
int main(void)
{
    /*Device CSC initial */
    ChipInit();

    /*LED initial*/
    BSP_LED_Init();
    BSP_LED_Cmd(BSP_LED0, BSP_LED_OFF);
    BSP_LED_Cmd(BSP_LED1, BSP_LED_OFF);

    /*Button initial*/  
    BSP_Button_Init();

    /*ADC initial*/
    ADC_Init();

//    /*Variable Resistor initial*/
//    BSP_VR_Init();

    /*Temperature sensor initial*/
    BSP_TemperatureSensor_Init();

    /*LCD initial*/
    BSP_LCD_Init();

    /*Calendar initial*/
    Calendar_Init();

    /*Parameter initial*/
    UpdateFlag.Status.B  = 0x00;
    LCD_StatusUpdateFlag = 0;
    LCD_Status           = LCD_STATUS_ENABLE;
    Drama_Mode           = DRAMA_MODE_SPLASH_SCREEN;
    Drama_LCD_Mode       = LCD_DRAMA_SPLASH_SCREEN;
    Drama_ActCmp        = 200;
    Drama_ActCnt        = 0;
    Drama_ADC_Act       = 0;
    Drama_Init();

    /*Systick initial*/
    InitTick(24000000UL,0);     // 1ms 

    /*main loop*/
    while(1)
    {
        /*KEY0 Handler (LCD status to Blank status)*/
        if(BSP_Button_GetKeyFlag(LCD_BLANK_KEY) == 1)
        {
            BSP_Button_ClearKeyFlag(LCD_BLANK_KEY);
            Drama_Mode = DRAMA_MODE_NORMAL;

            if( LCD_Status == LCD_STATUS_BLANK)
            {
                LCD_Status = LCD_STATUS_ENABLE;

                BSP_LCD_OFF_Cmd(DISABLE);
                LCD_BlinkMode_Config(LCD, BlinkDisable);
            }
            else
            {
                LCD_Status = LCD_STATUS_BLANK;

                BSP_LCD_OFF_Cmd(ENABLE);
                LCD_BlinkMode_Config(LCD, BlinkDisable);
            }
        }

        /*KEY1 Handler (LCD status to Blinking status)*/
        if(BSP_Button_GetKeyFlag(LCD_BLINK_KEY)  == 1)
        {
            BSP_Button_ClearKeyFlag(LCD_BLINK_KEY);

            if( Drama_Mode == DRAMA_MODE_NORMAL)
            {
                if( LCD_Status != LCD_STATUS_BLANK)
                {
                    if( LCD_Status == LCD_STATUS_BLINK)
                    {
                        LCD_Status = LCD_STATUS_ENABLE;
                    }
                    else
                    {
                        LCD_Status = LCD_STATUS_ToBLINK;
                    }

                    LCD_StatusUpdateFlag = 1;
                    Drama_LEDMode_Config(LED_DRAMA_VAR);
                    Drama_ARGBMode_Config(ARGB_DRAMA_CALENDAR);
                }
            }
        }

        /*KEY2 Handler (LCD drama mode to Calendar mode)*/
        if(BSP_Button_GetKeyFlag(LCD_CALENDAR_KEY) == 1)
        {
            BSP_Button_ClearKeyFlag(LCD_CALENDAR_KEY);

            Drama_Mode = DRAMA_MODE_NORMAL;

            if( Drama_LCD_Mode != LCD_DRAMA_CALENDAR && LCD_Status != LCD_STATUS_BLANK)
            {
                Drama_LCD_Mode = LCD_DRAMA_CALENDAR;
                UpdateFlag.Status.MBIT.LCD_UpdateFlag = 1;

                Drama_LEDMode_Config(LED_DRAMA_VAR);
                Drama_ARGBMode_Config(ARGB_DRAMA_CALENDAR);
            }
        }

        /*KEY3 Handler (LCD drama mode to Measure mode)*/
        if(BSP_Button_GetKeyFlag(LCD_MEASURE_KEY)  == 1)
        {
            BSP_Button_ClearKeyFlag(LCD_MEASURE_KEY);
            Drama_Mode = DRAMA_MODE_NORMAL;

            if(LCD_Status != LCD_STATUS_BLANK)
            {
                Drama_LCD_Mode ++;
                
                if( Drama_LCD_Mode > LCD_DRAMA_DC_mA)
                {
                    Drama_LCD_Mode = LCD_DRAMA_TEMP;
                }

                UpdateFlag.Status.MBIT.LCD_UpdateFlag = 1;
                Drama_LEDMode_Config(LED_DRAMA_VAR);
                Drama_ARGBMode_Config(ARGB_DRAMA_CALENDAR);
            }
        }

        /*LCD status update*/
        if(LCD_StatusUpdateFlag == 1)
        {
            LCD_StatusUpdateFlag = 0;
            switch(LCD_Status)
            {
                case LCD_STATUS_ToBLINK:
                                        UpdateFlag.Status.MBIT.LCD_UpdateFlag = 0;
                                        LCD_Status = LCD_STATUS_BLINK;
                                        LCD_BlinkMode_Config(LCD, BlinkAll);
                                        break;

                case LCD_STATUS_ENABLE:
                                        LCD_BlinkMode_Config(LCD, BlinkDisable);
                                        UpdateFlag.Status.MBIT.LCD_UpdateFlag = 1;
                                        break;

                case LCD_STATUS_BLANK:
                                        LCD_BlinkMode_Config(LCD, BlinkDisable);
                                        break;

                case LCD_STATUS_BLINK:
                default:
                                        break;
            }
        }

        /*LCD update*/
        if(UpdateFlag.Status.MBIT.LCD_UpdateFlag == 1)
        {
            if(Drama_LCD_Update(Drama_LCD_Mode) == BSP_LCD_SUCCESS)
            {
                UpdateFlag.Status.MBIT.LCD_UpdateFlag = 0;
            }
        }
    }
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @exception   
 * @note        
 *******************************************************************************
 */
void Drama_LCD_SplashScreen_CpltCallback(void)
{
    Drama_Mode     = DRAMA_MODE_NORMAL;
    Drama_LCD_Mode = LCD_DRAMA_CALENDAR; 
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @exception   
 * @note        
 *******************************************************************************
 */
void Drama_LCD_SplashScreen_HalfCallback(void)
{
    Drama_LEDMode_Config(LED_DRAMA_VAR);
    Drama_ARGBMode_Config(ARGB_DRAMA_CALENDAR);
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @param[in]   
 * @exception   
 * @note        
 *******************************************************************************
 */
static void ADC_Init(void)
{
    ADC_InitTypeDef ADC_Base;

    /*ADC Default Initial.*/
    ADC_DeInit(ADC0);

    ADC_BaseStructure_Init(&ADC_Base);

    ADC_Base.ADCMainClockSelect   = ADC_CKADC; 
    ADC_Base.ADC_IntCK_Div        = ADC_IntDIV2;
    ADC_Base.ADC_CKPLL_Div        = ADC_PLLDIV6;
    ADC_Base.ADC_DataAlign        = ADC_RightJustified;
    ADC_Base.ADC_ResolutionSel    = ADC_12BitData;
    ADC_Base.ADC_DataOverrunEvent = ADC_DataOverWritten;

    ADC_Base_Init(ADC0, &ADC_Base);

    ADC_WaitDataReadOut(ADC0, DISABLE);

    ADC_Cmd(ADC0, ENABLE);

    ADC_SampleClockPhase_Select(ADC0, ADC_CK_PHASE0);

    //Disable PGA
    ADC_PGA_Cmd(ADC0, DISABLE);
    ADC_SetPGAGain(ADC0, 0);

    //
    ADC_TriggerSource_Select(ADC0, ADC_START);
    ADC_TriggerEdge_Select(ADC0, ADC_DisableTrg);
    ADC_ConversionMode_Select(ADC0, ADCMode);
    ADC_ContinueMode_Cmd(ADC0, DISABLE);
    ADC_MainConversionMode_Select(ADC0, ADC_OneShot);

    ADC_SetExtendSampling(ADC0, 30);

    ADC_SetConversionTime(ADC0, ADC_FastCONV);
    ADC_SetOperationCurrent(ADC0, ADC_BIAS_LVL3);
    ADC_SampleClockPhase_Select(ADC0, ADC_CK_PHASE2);

    ADC_SingleDifferentMode_Select(ADC0, ADC_SingleMode);
    ADC_SetOutputCodeFormat(ADC0, ADC_UnsignedFormat);

    //ADC calibration
    ADC_StartCalibration(ADC0, ENABLE);
}

