


/**
 ******************************************************************************
 *
 * @file        MG32_CAN_DRV.h
 * @brief       The code is CAN driver H file.
 *
 * @par         Project
 *              MG32
 * @version     V1.03
 * @date        2024/09/25
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


#ifndef _MG32_CAN_DRV_H

#define _MG32_CAN_DRV_H

/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32_CAN.h"
#include "MG32__Common_DRV.h"

/* Typedef ------------------------------------------------------------------*/


/*! @enum  CAN_Clock_TypeDef
    @brief CAN internal clock source option. 
*/ 
typedef enum
{
    CAN_Clock_PROC    = CAN_CLK_CK_SEL_proc_w,    /*!< CAN bit time clock source is from CK_CAN_PR*/
    CAN_Clock_NCO_P0  = CAN_CLK_CK_SEL_nco_p0_w,  /*!< CAN bit time clock source is from NCO_P0*/
    CAN_Clock_LS      = CAN_CLK_CK_SEL_ck_ls_w    /*!< CAN bit time clock source is from CK_LS*/
}CAN_Clock_TypeDef;


/*! @enum  CAN_SetOpMode_TypeDef
    @brief CAN operation mode option. 
*/ 
typedef enum
{
    CAN_Initial_Mode  = (CAN_CR0_INIT_EN_enable_w  | CAN_CR0_LP_EN_disable_w), /*!< CAN into "Initial Mode".*/ 
    CAN_Normal_Mode   = (CAN_CR0_INIT_EN_disable_w | CAN_CR0_LP_EN_disable_w), /*!< CAN into "Normal Mode".*/
    CAN_LowPower_Mode = (CAN_CR0_INIT_EN_disable_w | CAN_CR0_LP_EN_enable_w)   /*!< CAN into "Low Power Mdoe".*/
}CAN_SetOpMode_TypeDef;


/*! @enum  CAN_RxErrorCounterMode_TypeDef
    @brief CAN receive error counter mode option. 
*/ 
typedef enum
{
    CAN_RXErrorMode_Normal  =  CAN_CR3_RXERR_MDS_normal_b3,  /*!< Receive error counter increased if receive error counter next value less than 255.*/
    CAN_RXErrorMode_Passive =  CAN_CR3_RXERR_MDS_passive_b3  /*!< Receive error counter increased until fault confinement state is error passive.*/
}CAN_RxErrorCounterMode_TypeDef;


/*! @enum  CAN_TXPRI_TypeDef
    @brief CAN TX message buffer process priority mode option. 
*/ 
typedef enum
{
    CAN_TXPriority_ID  =  CAN_CR0_TX_PRI_id_w,  /*!< The message buffer is transmission by the priority of the message identifier.*/
    CAN_TXPriority_SEQ =  CAN_CR0_TX_PRI_seq_w  /*!< The message buffer is transmissoin by the request sequence order.*/
}CAN_TXPRI_TypeDef;


/*! @enum  CAN_TSTMode_TypeDef
    @brief CAN test mode option. 
*/ 
typedef enum
{
    CAN_TestMode_Disable             =  CAN_CR0_TST_MDS_disable_w,  /*!< Test mode disable.*/
    CAN_TestMode_Loopbackk           =  CAN_CR0_TST_MDS_lbm_w,      /*!< Test mode is "Loop Back Mode".*/
    CAN_TestMode_Silent              =  CAN_CR0_TST_MDS_sil_w,      /*!< Test mode is "Silent Mode".*/
    CAN_TestMode_LoopBackWithSilent  =  CAN_CR0_TST_MDS_lbs_w       /*!< Test mode is "Loop Back Combined with Silent Mode".*/
}CAN_TSTMode_TypeDef;


/*! @enum  CAN_RXFIFO_TypeDef
    @brief CAN receive message buffer type option. 
*/ 
typedef enum
{
    CAN_RXFIFO_Two           =  CAN_CR0_RBUF_SEL_two_w,  /*!< Two FIFO structure.*/
    CAN_RXFIFO_One           =  CAN_CR0_RBUF_SEL_one_w   /*!< One FIFO structure ( Two FIFOs are combined as one message FIFO).*/
}CAN_RXFIFO_TypeDef;


/*! @enum  CAN_ROVR_TypeDef
    @brief CAN receive message buffer overrun mode option. 
*/ 
typedef enum
{
    CAN_RXOverrun_Overwrite = CAN_CR0_ROVR_MDS_overwritten_w,  /*!< The last message of FIFO be overwritten by mew message.*/
    CAN_RXOverrun_Keep      = CAN_CR0_ROVR_MDS_keep_w          /*!< Lost new message.*/
}CAN_ROVR_TypeDef;


/*! @enum  CAN_LPWakeup_TypeDef
    @brief CAN wakeup mode option. 
*/ 
typedef enum
{
    CAN_LowPowerWakeup_OnlySoftware = CAN_CR0_WUP_MDS_soft_w,  /*!< Only software can leave low power mode.*/
    CAN_LowPowerWakeup_BusActive    = CAN_CR0_WUP_MDS_auto_w   /*!< Both software and bus active can leave low power mode.*/
}CAN_LPWakeup_TypeDef;

/*! @enum  CAN_RXOS_TypeDef
    @brief CAN RX data oversampling mode option. 
*/ 
typedef enum
{
    CAN_RXOversampling_One   = CAN_CR0_OS_MDS_one_w,   /*!< RX sample once time..*/
    CAN_RXOversampling_Three = CAN_CR0_OS_MDS_three_w  /*!< RX sample three time..*/
}CAN_RXOS_TypeDef;


/*! @enum  CAN_FilterIndex_TypeDef
    @brief CAN acceptance filter bank option. 
*/ 
typedef enum
{
    CAN_Filter0  = 0,  /*!< 1st bank filter of the CAN.*/
    CAN_Filter1  = 1,  /*!< 2nd bank filter of the CAN.*/
    CAN_Filter2  = 2,  /*!< 3rd bank filter of the CAN.*/
    CAN_Filter3  = 3,  /*!< 4th bank filter of the CAN.*/
    CAN_Filter4  = 4,  /*!< 5th bank filter of the CAN.*/
    CAN_Filter5  = 5,  /*!< 6th bank filter of the CAN.*/
    
}CAN_FilterIndex_TypeDef;


/*! @enum  CAN_FilterMode_TypeDef
    @brief CAN acceptance filter mode option. 
*/ 
typedef enum
{

    CAN_Filter_32bitMask  = ( CAN_AFC1_AF0_CFG_single_w | CAN_AFC1_AF0_MDS_maskmode_w),  /*!< 32 bit Mask mode.*/
    CAN_Filter_32bitList  = ( CAN_AFC1_AF0_CFG_single_w | CAN_AFC1_AF0_MDS_listmode_w),  /*!< 32 bit LIST mode.*/
    CAN_Filter_16bitMask  = ( CAN_AFC1_AF0_CFG_dual_w   | CAN_AFC1_AF0_MDS_maskmode_w),  /*!< 16 bit MASK mode.*/
    CAN_Filter_16bitList  = ( CAN_AFC1_AF0_CFG_dual_w   | CAN_AFC1_AF0_MDS_listmode_w)   /*!< 16 bit LIST mode.*/
}CAN_FilterMode_TypeDef;


/*! @enum  CAN_FilterFIFO_TypeeDef
    @brief CAN acceptance filter FIFO option.
*/ 
typedef enum
{
    CAN_Filter_FIFO0   = CAN_AFC0_AF0_FSEL_fifo0_w, /*!< The FIFO use FIFO 0.*/
    CAN_Filter_FIFO1   = CAN_AFC0_AF0_FSEL_fifo1_w, /*!< The FIFO use FIFO 1.*/
}CAN_FilterFIFO_TypeeDef;

/*! @enum  CAN_TXIndex_TypeDef
    @brief TX message buffer of the CANx option. 
*/ 
typedef enum
{
    CAN_Tx_Buf0  = 0,  /*!< TX message buffer 0 of the CANx.*/
    CAN_Tx_Buf1  = 1,  /*!< TX message buffer 1 of the CANx.*/
    CAN_Tx_Buf2  = 2,  /*!< TX message buffer 2 of the CANx.*/
}CAN_TXIndex_TypeDef;

/*! @enum  CAN_RXFIFOIndex_TypeDef
    @brief RX message FIFO of the CANx option. 
*/ 
typedef enum
{
    CAN_Rx_FIFO0 = 0,  /*!< RX message FIFO 0 of the CANx.*/
    CAN_Rx_FIFO1 = 1   /*!< RX message FIFO 1 of the CANx.*/
}CAN_RXFIFOIndex_TypeDef;





/*! @struct CAN_Bittime_TypeDef
    @brief  Bit time struct
*/
typedef struct
{
    union
    {
        uint32_t  W[2];
        struct
        {
            uint32_t TimeSeg1       :4;   /*!< Prop_Seg + Phase_Seg1.*/  
            uint32_t                :4;   
            uint32_t TimeSeg2       :3;   /*!< Phase_Seg2.*/
            uint32_t                :5;   
            uint32_t SyncJumpWidth  :2;   /*!< Synchronization jump width.*/
            uint32_t                :14;
            uint32_t                :16;
            uint32_t prescaler      :10;  /*!< programmable prescaler for time quantum.*/  
            uint32_t                :6;
        }MBIT;
    }CTR;
}CAN_Bittime_TypeDef;


/*! @struct CAN_TXBuf_TypeDef
    @brief  Transmit buffer data register struct.
*/
typedef struct
{
    union
    {
        uint32_t  W[4];
        struct
        {
            uint32_t  EID  : 18;    /*!< Identifier extension (ID-17 to ID-0).*/
            uint32_t  SID  : 11;    /*!< Base identifier (ID-28 to ID-18).*/
            uint32_t  RTR  :  1;    /*!< Remote transmission request bit.*/
            uint32_t  IDE  :  1;    /*!< Identifier extension bit.*/
            uint32_t       :  1;
            uint32_t  DLC  :  4;    /*!< Data length code.*/
            uint32_t       : 28;
            uint8_t   Data[8];      /*!< Data field.*/
        }Field;
    }Format;
}CAN_TXBuf_TypeDef;


/*! @struct CAN_RXBuf_TypeDef
    @brief  Receive FIFO data register struct.
*/
typedef struct
{
    union
    {
        uint32_t  W[4];
        struct
        {
            uint32_t EID  : 18;  /*!< Identifier extension (ID-17 to ID-0).*/
            uint32_t SID  : 11;  /*!< Base identifier (ID-28 to ID-18).*/
            uint32_t RTR  :  1;  /*!< Remote transmission request bit.*/
            uint32_t IDE  :  1;  /*!< Identifier extension bit.*/
            uint32_t      :  1;  
            uint32_t DLC  :  4;  /*!< Data length code.*/
            uint32_t      : 28;  /*!< Data field.*/
            uint8_t  Data[8];              
        }Field;
    }Format;
}CAN_RXBuf_TypeDef;


/*! @struct CAN_RXBuf_TypeDef
    @brief  Acceptance filter bank struct.
*/
typedef struct
{
    union
    {
        uint32_t W[2];
        uint16_t H[4];
        uint8_t  B[8];
        struct
        {
            struct
            {
                uint32_t            :1;   
                uint32_t  RTR       :1;   /*!< Compare ID's RTR (Remote transmission request bit).*/
                uint32_t  IDE       :1;   /*!< Compare ID's IDE (Identifier extension bit).*/
                uint32_t  EID       :18;  /*!< Compare ID's EID (Identifier extension ID-17 to ID-10).*/
                uint32_t  SID       :11;  /*!< Compare ID's SID (Base identifier ID-28 to ID-18).*/  
            }ID;         
            struct
            {
                uint32_t            :1;
                uint32_t  RTR       :1;   /*!< Mask ID's RTR (Remote transmission request bit)
                                               0: Don't care 
                                               1: Must match.*/
                uint32_t  IDE       :1;   /*!< Mask ID's IDE (Identifier extension bit)
                                               0: Don't care 
                                               1: Must match.*/
                uint32_t  EID       :18;  /*!< Mask ID's EID (Identifier extension ID-17 to ID-0)
                                               0: Don't care 
                                               1: Must match.*/
                uint32_t  SID       :11;  /*!< Mask ID's SID (Base identifier ID-28 to ID-18)
                                               0: Don't care 
                                               1: Must match.*/    
            }Mask;                        /*!< Acceptance filter mask configure struct.*/
        }Mask32;
        struct
        {
            struct
            {
                uint32_t            :1;
                uint32_t  RTR       :1;   /*!< List0 ID's RTR (Remote transmission request bit).*/
                uint32_t  IDE       :1;   /*!< List0 ID's IDE (Identifier extension bit).*/
                uint32_t  EID       :18;  /*!< List0 ID's EID (Identifier extension ID-17 to ID-10).*/
                uint32_t  SID       :11;  /*!< List0 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List0;           
            struct                  
            {                       
                uint32_t            :1;
                uint32_t  RTR       :1;   /*!< List1 ID's RTR (Remote transmission request bit).*/
                uint32_t  IDE       :1;   /*!< List1 ID's IDE (Identifier extension bit).*/
                uint32_t  EID       :18;  /*!< List1 ID's EID (Identifier extension ID-17 to ID-10).*/
                uint32_t  SID       :11;  /*!< List1 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List1;                       
        }List32;
        struct
        {
            struct
            {
                uint16_t  EID_17_15 :3;   /*!< Compare0 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t  RTR       :1;   /*!< Compare0 ID's RTR (Remote transmission request bit).*/
                uint16_t  IDE       :1;   /*!< Compare0 ID's EID (Identifier extension bit).*/
                uint16_t  SID       :11;  /*!< Compare0 ID's SID (Base identifier ID-28 to ID-18).*/    
            }ID0;                         
            struct
            {
                uint16_t  EID_17_15 :3;   /*!< Mask0 ID's EID (Identifier extension ID-17 to ID-15)
                                               0: Don't care 
                                               1: Must match.*/             
                uint16_t  RTR       :1;   /*!< Mask0 ID's RTR (Remote transmission request bit)
                                               0: Don't care 
                                               1: Must match.*/              
                uint16_t  IDE       :1;   /*!< Mask0 ID's IDE (Identifier extension bit)
                                               0: Don't care 
                                               1: Must match.*/               
                uint16_t  SID       :11;  /*!< Mask ID's SID (Base identifier ID-28 to ID-18)
                                               0: Don't care 
                                               1: Must match.*/      
            }Mask0;
            struct
            {
                uint16_t  EID_17_15 :3;   /*!< Compare1 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t  RTR       :1;   /*!< Compare1 ID's RTR (Remote transmission request bit).*/
                uint16_t  IDE       :1;   /*!< Compare1 ID's EID (Identifier extension bit).*/
                uint16_t  SID       :11;  /*!< Compare1 ID's SID (Base identifier ID-28 to ID-18).*/       
            }ID1;
            struct
            {
                uint16_t  EID_17_15 :3;   /*!< Mask1 ID's EID (Identifier extension ID-17 to ID-15)
                                               0: Don't care 
                                               1: Must match.*/             
                uint16_t  RTR       :1;   /*!< Mask1 ID's RTR (Remote transmission request bit)
                                               0: Don't care 
                                               1: Must match.*/              
                uint16_t  IDE       :1;   /*!< Mask1 ID's IDE (Identifier extension bit)
                                               0: Don't care 
                                               1: Must match.*/               
                uint16_t  SID       :11;  /*!< Mask1 ID's SID (Base identifier ID-28 to ID-18)
                                               0: Don't care 
                                               1: Must match.*/  
            }Mask1;
        }Mask16;      
        struct
        {
            struct
            {
                uint16_t EID_17_15 : 3;   /*!< List0 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t RTR       : 1;   /*!< List0 ID's RTR (Remote transmission request bit).*/
                uint16_t IDE       : 1;   /*!< List0 ID's IDE (Identifier extension bit).*/
                uint16_t SID       :11;   /*!< List0 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List0;
            struct
            {
                uint16_t EID_17_15 : 3;   /*!< List1 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t RTR       : 1;   /*!< List1 ID's RTR (Remote transmission request bit).*/
                uint16_t IDE       : 1;   /*!< List1 ID's IDE (Identifier extension bit).*/
                uint16_t SID       :11;   /*!< List1 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List1;
            struct
            {
                uint16_t EID_17_15 : 3;   /*!< List2 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t RTR       : 1;   /*!< List2 ID's RTR (Remote transmission request bit).*/
                uint16_t IDE       : 1;   /*!< List2 ID's IDE (Identifier extension bit).*/
                uint16_t SID       :11;   /*!< List2 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List2;           
            struct
            {
                uint16_t EID_17_15 : 3;   /*!< List3 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t RTR       : 1;   /*!< List3 ID's RTR (Remote transmission request bit).*/
                uint16_t IDE       : 1;   /*!< List3 ID's IDE (Identifier extension bit).*/
                uint16_t SID       :11;   /*!< List3 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List3;
        }List16;
    }AFnR;
}CAN_Filter_TypeDef;


/* Define ------------------------------------------------------------------*/

                        
#define CAN_RXFIFO_SHIFT     0xA0   /*!< CAN receive FIFO register address shift in CAN module register.*/
#define CAN_RXFIFOX_SHIFT    0x10   /*!< Each CAN receive FIFO related register size.*/

#define CAN0_RXFIFO0_Base    CAN0_Base + ( CAN_RXFIFO_SHIFT + (CAN_RXFIFOX_SHIFT * 0))   /*!< CAN0 receive FIFO 0 start address.*/
#define CAN0_RXFIFO1_Base    CAN0_Base + ( CAN_RXFIFO_SHIFT + (CAN_RXFIFOX_SHIFT * 1))   /*!< CAN0 receive FIFO 1 start address.*/

#define CAN0_RXFIFO0         ((CAN_RXBuf_TypeDef*)CAN0_RXFIFO0_Base) /*!< CAN receive FIFO 0 related register group.*/
#define CAN0_RXFIFO1         ((CAN_RXBuf_TypeDef*)CAN0_RXFIFO1_Base) /*!< CAN receive FIFO 0 related register group.*/

#define CAN_TXBUF_SHIFT      0xC0   /*!< CAN transmit buffer register address shift in CAN module register.*/
#define CAN_TXBUFX_SHIFT     0x10   /*!< Each CAN transmit buffer related register size.*/

#define CAN0_TXBUF0_Base     (CAN0_Base + ( CAN_TXBUF_SHIFT + (CAN_TXBUFX_SHIFT * 0)))   /*!< CAN0 transmit buffer 0 start address .*/
#define CAN0_TXBUF1_Base     (CAN0_Base + ( CAN_TXBUF_SHIFT + (CAN_TXBUFX_SHIFT * 1)))   /*!< CAN0 transmit buffer 1 start address .*/
#define CAN0_TXBUF2_Base     (CAN0_Base + ( CAN_TXBUF_SHIFT + (CAN_TXBUFX_SHIFT * 2)))   /*!< CAN0 transmit buffer 2 start address .*/

#define CAN0_TXBUF0          ((CAN_TXBuf_TypeDef*)CAN0_TXBUF0_Base)  /*!< CAN0 transmit buffer 0 related register group.*/
#define CAN0_TXBUF1          ((CAN_TXBuf_TypeDef*)CAN0_TXBUF1_Base)  /*!< CAN0 transmit buffer 1 related register group..*/
#define CAN0_TXBUF2          ((CAN_TXBuf_TypeDef*)CAN0_TXBUF2_Base)  /*!< CAN0 transmit buffer 2 related register group..*/

#define CAN_FILTER_SHIFT     0x30   /*!< Acceptance filter register address shift in CAN module register.*/
#define CAN_FILTERX_SHIFT    0x08   /*!< Each Acceptance filter related register size.*/

#define CAN0_FILTER0_Base    (CAN0_Base + (CAN_FILTER_SHIFT + (CAN_FILTERX_SHIFT * 0))) /*!< CAN0 acceptance filter bank 0 register start address.*/
#define CAN0_FILTER1_Base    (CAN0_Base + (CAN_FILTER_SHIFT + (CAN_FILTERX_SHIFT * 1))) /*!< CAN0 acceptance filter bank 1 register start address.*/
#define CAN0_FILTER2_Base    (CAN0_Base + (CAN_FILTER_SHIFT + (CAN_FILTERX_SHIFT * 2))) /*!< CAN0 acceptance filter bank 2 register start address.*/
#define CAN0_FILTER3_Base    (CAN0_Base + (CAN_FILTER_SHIFT + (CAN_FILTERX_SHIFT * 3))) /*!< CAN0 acceptance filter bank 3 register start address.*/
#define CAN0_FILTER4_Base    (CAN0_Base + (CAN_FILTER_SHIFT + (CAN_FILTERX_SHIFT * 4))) /*!< CAN0 acceptance filter bank 4 register start address.*/
#define CAN0_FILTER5_Base    (CAN0_Base + (CAN_FILTER_SHIFT + (CAN_FILTERX_SHIFT * 5))) /*!< CAN0 acceptance filter bank 5 register start address.*/

#define CAN0_FILTER0         ((CAN_Filter_TypeDef*)CAN0_FILTER0_Base) /*!< CAN0 acceptance filter bank 0 related register group.*/
#define CAN0_FILTER1         ((CAN_Filter_TypeDef*)CAN0_FILTER1_Base) /*!< CAN0 acceptance filter bank 1 related register group.*/
#define CAN0_FILTER2         ((CAN_Filter_TypeDef*)CAN0_FILTER2_Base) /*!< CAN0 acceptance filter bank 2 related register group.*/
#define CAN0_FILTER3         ((CAN_Filter_TypeDef*)CAN0_FILTER3_Base) /*!< CAN0 acceptance filter bank 3 related register group.*/
#define CAN0_FILTER4         ((CAN_Filter_TypeDef*)CAN0_FILTER4_Base) /*!< CAN0 acceptance filter bank 4 related register group.*/
#define CAN0_FILTER5         ((CAN_Filter_TypeDef*)CAN0_FILTER5_Base) /*!< CAN0 acceptance filter bank 5 related register group.*/



#define CAN_IT_BUS           CAN_STA_BUSF_mask_w     /*!< Bus-off interrupt.*/
#define CAN_IT_EW            CAN_STA_EWF_mask_w      /*!< Error warning interrupt.*/
#define CAN_IT_WUP           CAN_STA_WUPF_mask_w     /*!< Wakeup from CAN low power state interrupt.*/
#define CAN_IT_EP            CAN_STA_EPF_mask_w      /*!< Error passive interrupt.*/
#define CAN_IT_ALOS          CAN_STA_ALOSF_mask_w    /*!< Bus arbitration lost interrupt.*/
#define CAN_IT_BERR          CAN_STA_BERRF_mask_w    /*!< Bus error interrupt.*/
#define CAN_IT_RX0           CAN_STA_RX0F_mask_w     /*!< Data receive message FIFO-0 interrupt.*/
#define CAN_IT_RX1           CAN_STA_RX1F_mask_w     /*!< Data receive message FIFO-1 interrupt.*/
#define CAN_IT_RFUL0         CAN_STA_RFUL0F_mask_w   /*!< Receive message FIFO-0 full interrupt.*/
#define CAN_IT_RFUL1         CAN_STA_RFUL1F_mask_w   /*!< Receive message FIFO-1 full interrupt.*/
#define CAN_IT_ROVR0         CAN_STA_ROVR0F_mask_w   /*!< Receive message FIFO-0 overrun interrupt.*/
#define CAN_IT_ROVR1         CAN_STA_ROVR1F_mask_w   /*!< Receive message FIFO-1 overrun interrupt.*/
#define CAN_IT_RPEND0        CAN_STA_RPEND0F_mask_w  /*!< Receive message FIFO-0 remained pending interrupt.*/
#define CAN_IT_RPEND1        CAN_STA_RPEND1F_mask_w  /*!< Receive message FIFO-1 remained pending interrupt.*/
#define CAN_IT_TX0           CAN_STA_TX0F_mask_w     /*!< Data transmit interrupt flag for TX message buffer0.*/
#define CAN_IT_TX1           CAN_STA_TX1F_mask_w     /*!< Data transmit interrupt flag for TX message buffer1.*/
#define CAN_IT_TX2           CAN_STA_TX2F_mask_w     /*!< Data transmit interrupt flag for TX message buffer2.*/
                              
#define CAN_IT_ALL           0x07FFFFFF
                              
                              
#define CAN_INITIAL_MODE      ( CAN_STA_INIT_STA_init_w | CAN_STA_LP_STA_not_w)        /*!< The current operation mode is a initial mode.*/
#define CAN_NORMAL_MODE       ( CAN_STA_INIT_STA_not_w  | CAN_STA_LP_STA_not_w)        /*!< The current operation mode is a normal mode.*/
#define CAN_LOWPOWER_MODE     ( CAN_STA_INIT_STA_not_w  | CAN_STA_LP_STA_slp_w)        /*!< The current operation mode is a low power mode.*/
                                                                                       
#define CAN_ERROR_ACTIVE      ( CAN_STA_BUS_STA_act_w  | CAN_STA_EP_STA_nonpassive_w)  /*!< Error mode is error active state now.*/
#define CAN_ERROR_PASSIVE     ( CAN_STA_BUS_STA_act_w  | CAN_STA_EP_STA_passive_w)     /*!< Error mode is error passive state now.*/
#define CAN_BUS_OFF           ( CAN_STA_BUS_STA_boff_w | CAN_STA_EP_STA_nonpassive_w)  /*!< Error mode is Bus-off state now.*/
                                                                                       
#define CAN_WARNING_OVER      (CAN_STA_EW_STA_err_w)                                   /*!< TX error / RX error count is over the error warning limit.*/  
#define CAN_WARNING_UNDER     (CAN_STA_EW_STA_normal_w)                                /*!< TX error / RX error count is under the error warning limit.*/

/**
 * @name CAN_TXBufStatus
 *             
 */ 
///@{
#define CAN_TRANSMIT_STATE_BUSY      (CAN_STA2_TB0_STA_lock_w )                                 /*!< CAN transmit buffer transmit status is busy.*/
#define CAN_TRANSMIT_STATE_FAILURE   (CAN_STA2_TB0_STA_release_w)                               /*!< CAN transmit buffer transmit status is failure.*/
#define CAN_TRANSMIT_STATE_SUCCESS   (CAN_STA2_TB0_STA_release_w | CAN_STA2_TC0_STA_happened_w) /*!< CAN transmit buffer transmit status is success.*/
///@}

/**
 * @name CAN_RXFIFOStatus
 *             
 */ 
///@{       
#define CAN_RXFIFO_STATE_RXOVERRUN   (CAN_STA2_ROVR0_STA_mask_w | CAN_STA2_RFUL0_STA_mask_w)  /*!< Receive FIFO status is overrun.*/
#define CAN_RXFIFO_STATE_FULL        (CAN_STA2_RFUL0_STA_mask_w)                              /*!< Receive FIFO status is full.*/
#define CAN_RXFIFO_STATE_PENDING     0x00000001                                               /*!< Receive FIFO status is existence data but not full yet.*/
#define CAN_RXFIFO_STATE_EMPTY       0x00000000                                               /*!< Receive FIFO status is empty.*/ 

///@}

/**
 * @name CAN_TransmissionState
 *       
 */ 
///@{ 
#define CAN_TRANSMISSION_STATE_TX    (CAN_STA2_TX_STA_mask_w)        /*!< The CAN module is transmitting now.*/
#define CAN_TRANSMISSION_STATE_RX    (CAN_STA2_RX_STA_mask_w)        /*!< The CAN module is receiving now.*/
#define CAN_TRANSMISSION_STATE_IDLE  (0UL)                           /*!< The CAN module is idle.*/
///@}


/**
 * @name CAN_ArbitrationLostCode
 *             
 */ 
///@{
#define CAN_ARBLOST_ID28     0x00     /*!< Arbitratio lost in  1st bit of identifier (ID28).*/
#define CAN_ARBLOST_ID27     0x01     /*!< Arbitratio lost in  2nd bit of identifier (ID27).*/
#define CAN_ARBLOST_ID26     0x02     /*!< Arbitratio lost in  3rd bit of identifier (ID26).*/
#define CAN_ARBLOST_ID25     0x03     /*!< Arbitratio lost in  4th bit of identifier (ID25).*/
#define CAN_ARBLOST_ID24     0x04     /*!< Arbitratio lost in  5th bit of identifier (ID24).*/
#define CAN_ARBLOST_ID23     0x05     /*!< Arbitratio lost in  6th bit of identifier (ID23).*/
#define CAN_ARBLOST_ID22     0x06     /*!< Arbitratio lost in  7th bit of identifier (ID22).*/
#define CAN_ARBLOST_ID21     0x07     /*!< Arbitratio lost in  8th bit of identifier (ID21).*/
#define CAN_ARBLOST_ID20     0x08     /*!< Arbitratio lost in  9th bit of identifier (ID20).*/
#define CAN_ARBLOST_ID19     0x09     /*!< Arbitratio lost in 10th bit of identifier (ID19).*/
#define CAN_ARBLOST_ID18     0x0A     /*!< Arbitratio lost in 11th bit of identifier (ID18).*/
#define CAN_ARBLOST_SRTR     0x0B     /*!< Arbitratio lost in RTR bit of stardard format or SRR bit of extended format.*/
#define CAN_ARBLOST_IDE      0x0C     /*!< Arbitratio lost in IDE bit.*/
#define CAN_ARBLOST_ID17     0x0D     /*!< Arbitratio lost in 12th bit of identifier (ID17).*/
#define CAN_ARBLOST_ID16     0x0E     /*!< Arbitratio lost in 13th bit of identifier (ID16).*/
#define CAN_ARBLOST_ID15     0x0F     /*!< Arbitratio lost in 14th bit of identifier (ID15).*/
#define CAN_ARBLOST_ID14     0x10     /*!< Arbitratio lost in 15th bit of identifier (ID14).*/
#define CAN_ARBLOST_ID13     0x11     /*!< Arbitratio lost in 16th bit of identifier (ID13).*/
#define CAN_ARBLOST_ID12     0x12     /*!< Arbitratio lost in 17th bit of identifier (ID12).*/
#define CAN_ARBLOST_ID11     0x13     /*!< Arbitratio lost in 18th bit of identifier (ID11).*/
#define CAN_ARBLOST_ID10     0x14     /*!< Arbitratio lost in 19th bit of identifier (ID10).*/
#define CAN_ARBLOST_ID9      0x15     /*!< Arbitratio lost in 20th bit of identifier (ID9).*/
#define CAN_ARBLOST_ID8      0x16     /*!< Arbitratio lost in 21th bit of identifier (ID8).*/
#define CAN_ARBLOST_ID7      0x17     /*!< Arbitratio lost in 22th bit of identifier (ID7).*/
#define CAN_ARBLOST_ID6      0x18     /*!< Arbitratio lost in 23th bit of identifier (ID6).*/
#define CAN_ARBLOST_ID5      0x19     /*!< Arbitratio lost in 24th bit of identifier (ID5).*/
#define CAN_ARBLOST_ID4      0x1A     /*!< Arbitratio lost in 25th bit of identifier (ID4).*/
#define CAN_ARBLOST_ID3      0x1B     /*!< Arbitratio lost in 26th bit of identifier (ID3).*/
#define CAN_ARBLOST_ID2      0x1C     /*!< Arbitratio lost in 27th bit of identifier (ID2).*/
#define CAN_ARBLOST_ID1      0x1D     /*!< Arbitratio lost in 28th bit of identifier (ID1).*/
#define CAN_ARBLOST_ID0      0x1E     /*!< Arbitratio lost in 29th bit of identifier (ID0).*/
#define CAN_ARBLOST_RTR      0x1F     /*!< Arbitratio lost in RTR bit of extended format.*/     

///@}

/**
 * @name CAN_ErrorCode
 *             
 */ 
///@{
//Error Code
#define CAN_ERR_CODE_BIT              0x0100    /*!< Bit error.*/
#define CAN_ERR_CODE_FORM             0x0200    /*!< Form error.*/
#define CAN_ERR_CODE_STUFF            0x0300    /*!< Stuff error.*/
#define CAN_ERR_CODE_CRC              0x0400    /*!< CRC error.*/
#define CAN_ERR_CODE_OTHER            0x0700    /*!< The other error.*/
                                      
//Error Direction                                      
#define CAN_ERR_DIR_TX                0x0000    /*!< Error happen in transmit.*/
#define CAN_ERR_DIR_RX                0x0080    /*!< Error happen in receive.*/

//Error Segment Code
#define CAN_ERR_SCODE_ID28_ID21       0x0002    /*!< Error happen in ID28 ~ ID21.*/
#define CAN_ERR_SCODE_SOF             0x0003    /*!< Error happen in SOF.*/
#define CAN_ERR_SCODE_SRTR            0x0004    /*!< Error happen in RTR of stardard format or SRR of extended format.*/
#define CAN_ERR_SCODE_IDE             0x0005    /*!< Error happen in IDE.*/
#define CAN_ERR_SCODE_ID20_ID18       0x0006    /*!< Error happen in ID20 ~ ID18.*/
#define CAN_ERR_SCODE_ID17_ID13       0x0007    /*!< Error happen in ID17 ~ ID13.*/
#define CAN_ERR_SCODE_CRC             0x0008    /*!< Error happen in CRC.*/
#define CAN_ERR_SCODE_RESBIT0         0x0009    /*!< Error happen in r0 (reserved bit 0).*/
#define CAN_ERR_SCODE_DATAFIELD       0x000A    /*!< Error happen in data field.*/
#define CAN_ERR_SCODE_DLC             0x000B    /*!< Error happen in DLC.*/
#define CAN_ERR_SCODE_RTR             0x000C    /*!< Error happen in RTR of extended format.*/
#define CAN_ERR_SCODE_RESBIT1         0x000D    /*!< Error happen in r1 (reserved bit 1).*/
#define CAN_ERR_SCODE_ID4_ID0         0x000E    /*!< Error happen in ID4 ~ ID0.*/
#define CAN_ERR_SCODE_ID12_ID5        0x000F    /*!< Error happen in ID12 ~ ID5.*/
#define CAN_ERR_SCODE_AERROR          0x0011    /*!< Error happen in active error flag.*/
#define CAN_ERR_SCODE_INTERMISSION    0x0012    /*!< Error happen in intermission.*/
#define CAN_ERR_SCODE_TOLERATE        0x0013    /*!< Error happen in tolerate dominant bits.*/
#define CAN_ERR_SCODE_PERROR          0x0016    /*!< Error happen in Passive error flag.*/
#define CAN_ERR_SCODE_ERRORDELIMITER  0x0017    /*!< Error happen in error delimiter.*/
#define CAN_ERR_SCODE_CRCDELIMITER    0x0018    /*!< Error happen in CRC delimiter.*/
#define CAN_ERR_SCODE_ACK             0x0019    /*!< Error happen in ACK.*/
#define CAN_ERR_SCODE_EOF             0x001A    /*!< Error happen in EOF.*/
#define CAN_ERR_SCODE_ACKDELIMITER    0x001B    /*!< Error happen in ACK delimiter.*/
#define CAN_ERR_SCODE_OVERLOADFRAME   0x001C    /*!< Error happen in overload frame.*/
  
///@}




/* Exported Function  --------------------------------------------------------*/

/**
 * @name Interrupt & Event Related.
 * 
 */
///@{
void CAN_IT_Config( CAN_Struct* CANx, uint32_t CAN_IT, FunctionalState CAN_IT_State);
void CAN_ITEA_Cmd( CAN_Struct* CANx, FunctionalState CAN_ITEA_State);
void CAN_ClearITFlag( CAN_Struct* CANx, uint32_t CAN_ITFlag);
uint32_t CAN_GetITStatus( CAN_Struct* CANx);
uint32_t CAN_GetITAllFlagStatus( CAN_Struct* CANx);
DRV_Return CAN_GetITSingleFlagStatus( CAN_Struct* CANx, uint32_t CAN_ITFlag);

uint32_t CAN_GetOperationMode( CAN_Struct* CANx);
uint32_t CAN_GetFaultConfinementState( CAN_Struct* CANx);
uint32_t CAN_GetWarningStatus( CAN_Struct* CANx);

uint32_t CAN_GetTXBufferStatus( CAN_Struct* CANx, CAN_TXIndex_TypeDef CAN_TxBuf_Index);
uint8_t CAN_GetRXFIFOMessageNumber( CAN_Struct* CANx, CAN_RXFIFOIndex_TypeDef CAN_RXFIFO_Index);
uint32_t CAN_GetRXFIFOStatus( CAN_Struct* CANx, CAN_RXFIFOIndex_TypeDef CAN_RXFIFO_Index);
uint32_t CAN_GetTransmissionStatus( CAN_Struct* CANx);

///@}

/**
 * @name Mode Control Related.
 * 
 */
///@{
void CAN_Cmd( CAN_Struct* CANx, FunctionalState  CAN_EN);
void CAN_OperationMode_Select( CAN_Struct* CANx, CAN_SetOpMode_TypeDef CAN_OPMode);
void CAN_TXPriorityMode_Select( CAN_Struct* CANx, CAN_TXPRI_TypeDef CAN_TXPriority_Mode);
void CAN_TestMode_Select(CAN_Struct* CANx, CAN_TSTMode_TypeDef CAN_TestMode);
void CAN_RXFIFOType_Select(CAN_Struct* CANx, CAN_RXFIFO_TypeDef CAN_RBUF_SEL);
void CAN_RXOverrunMode_Select( CAN_Struct* CANx, CAN_ROVR_TypeDef CAN_RXOverrunMode);
void CAN_AutoRetransmit_Cmd( CAN_Struct* CANx, FunctionalState CAN_AutoTX_EN);
void CAN_FDTolerant_Cmd( CAN_Struct* CANx, FunctionalState CAN_FDTolerant_EN);
void CAN_FDTolerantEdgeFilter_Cmd( CAN_Struct* CANx, FunctionalState CAN_FDTolerant_EdgeFL_EN);
void CAN_LowpowerWakeupMode_Select( CAN_Struct* CANx, CAN_LPWakeup_TypeDef CAN_LPWakeup_Mode);
void CAN_RXOversamplingMode_Select( CAN_Struct* CANx, CAN_RXOS_TypeDef CAN_RXOS_Mode);
void CAN_SelfReceptionMode_Cmd( CAN_Struct* CANx, FunctionalState CAN_SRR_EN);
///@}

/**
 * @name Bit-time Related.
 * 
 */
///@{
void CAN_ClockSource_Select(CAN_Struct* CANx, CAN_Clock_TypeDef CANx_ClockSource);
void CAN_BitTime_Config( CAN_Struct* CANx , CAN_Bittime_TypeDef* CANx_Bittime);
///@}

/**
 * @name Acceptance Filter Related.
 * 
 */
///@{
void CAN_AcceptanceFilter_Cmd( CAN_Struct* CANx, CAN_FilterIndex_TypeDef CANx_FL_Index, FunctionalState CANx_FL_Cmd);
void CAN_AcceptanceFilterMode_Config( CAN_Struct* CANx, CAN_FilterIndex_TypeDef CANx_FL_Index, CAN_FilterMode_TypeDef CANx_FL_Mode);
void CAN_AcceptanceFilterFIFO_Select( CAN_Struct* CANx, CAN_FilterIndex_TypeDef CANx_FL_Index, CAN_FilterFIFO_TypeeDef CANx_FL_FIFO);
void CAN_AccptanceFilter_Config( CAN_Struct* CANx, CAN_FilterIndex_TypeDef CANx_FL_Index, CAN_Filter_TypeDef* CAN_Filter_Config);
///@}

/**
 * @name Transmission Related.
 * 
 */
///@{
void CAN_SendFrame( CAN_Struct *CANx, CAN_TXIndex_TypeDef CAN_TXBuf_Index, CAN_TXBuf_TypeDef *CAN_TXMailbox);
void CAN_SingleShotSendFrame( CAN_Struct *CANx, CAN_TXIndex_TypeDef CAN_TXBuf_Index, CAN_TXBuf_TypeDef *CAN_TXMailbox);
void CAN_AbortTXRequest( CAN_Struct *CANx, CAN_TXIndex_TypeDef CAN_TXBuf_Index);
void CAN_ReceiveFrame( CAN_Struct* CANx, CAN_RXFIFOIndex_TypeDef CAN_RXFIFO_Index, CAN_RXBuf_TypeDef *CAN_RXMessage);
void CAN_ResetRXFIFO( CAN_Struct* CANx, CAN_RXFIFOIndex_TypeDef CAN_rRXFIFO_Index);
///@}

/**
 * @name Error Related.
 * 
 */
///@{
uint8_t CAN_GetTXErrorCounter( CAN_Struct* CANx);
uint8_t CAN_GetRXErrorCounter( CAN_Struct* CANx);

void CAN_RXErrorCounterMode_Select( CAN_Struct* CANx, CAN_RxErrorCounterMode_TypeDef CAN_RxErr_Mode);
void CAN_SetRXErrorCounter( CAN_Struct *CANx, uint8_t CAN_RXError_Counter);
void CAN_SetTXErrorCounter( CAN_Struct *CANx, uint8_t CAN_TXError_Counter);
void CAN_SetErrorWarningLimit( CAN_Struct *CANx, uint8_t CAN_EW_Limit);

uint16_t CAN_GetErrorCode( CAN_Struct* CANx);
uint8_t  CAN_GetArbitrationLostBit( CAN_Struct* CANx);
///@}

/**
 * @name Signal Related.
 * 
 */
///@{
void CAN_TXInverse_Cmd( CAN_Struct* CANx, FunctionalState CAN_TXInv_EN);
void CAN_RXInverse_Cmd( CAN_Struct* CANx, FunctionalState CAN_RXInv_EN);
void CAN_RxTxSwap_Cmd( CAN_Struct* CANx, FunctionalState CAN_RxTxSwap_EN);
///@}



#endif























