


/**
 ******************************************************************************
 *
 * @file        MG32_RGB_TLC5948_API.c
 * @brief       The TLC5948 LED Driver C file.
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2023/04/26
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32_RGB_TLC5948_API.h"

#if MG32_USBD_KEYBOARD_RGB_EN == API_RGB_TLC5948


/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define API_LED_SHIFT_BYTE             32

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static uint8_t  API_LED_UpdateDataBuf[3][API_LED_SHIFT_BYTE];
static void     API_RGB_UpdateControlTrigger(void);

static uint8_t  API_RGB1_SceneryTmp[API_RGB_TotalLine][API_LED_SHIFT_BYTE];                      
static uint8_t  API_RGB2_SceneryTmp[API_RGB_TotalLine][API_LED_SHIFT_BYTE];     
static uint8_t  API_RGB3_SceneryTmp[API_RGB_TotalLine][API_LED_SHIFT_BYTE]; 

static uint8_t  API_RGB1_Scenery[API_RGB_TotalLine][API_LED_SHIFT_BYTE];                      
static uint8_t  API_RGB2_Scenery[API_RGB_TotalLine][API_LED_SHIFT_BYTE];     
static uint8_t  API_RGB3_Scenery[API_RGB_TotalLine][API_LED_SHIFT_BYTE];     

/* Private function prototypes -----------------------------------------------*/
void API_RGBData_IRQHandler(void);


/**
 *******************************************************************************
 * @brief	    LED Driver Data buffer to default.
 * @details     
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
void API_RGBLedDriverData_DeInit(void)
{
    uint32_t RGBLDD_DeInitTmp;

    for(RGBLDD_DeInitTmp = 0; RGBLDD_DeInitTmp < API_LED_SHIFT_BYTE; RGBLDD_DeInitTmp++)
    {
        API_LED_UpdateDataBuf[0][RGBLDD_DeInitTmp] = 0;
        API_LED_UpdateDataBuf[1][RGBLDD_DeInitTmp] = 0;
        API_LED_UpdateDataBuf[2][RGBLDD_DeInitTmp] = 0;
    }
}

/**
 *******************************************************************************
 * @brief	    LED Driver control parameter initial.
 * @details     
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
void API_RGBLedControlParameter_Init(void)
{
    uint32_t RGBLDD_DeInitTmp;
    uint32_t RGBLDD_DeInitTmp2;
    
    //====================================================================================
    //RGB Data Initial
    for( RGBLDD_DeInitTmp = 0; RGBLDD_DeInitTmp < API_RGB_TotalLine ; RGBLDD_DeInitTmp++)
    {
        for( RGBLDD_DeInitTmp2 = 0 ; RGBLDD_DeInitTmp2 < API_LED_SHIFT_BYTE; RGBLDD_DeInitTmp2++)
        {
            API_RGB3_Scenery[RGBLDD_DeInitTmp][RGBLDD_DeInitTmp2] = 0x00; 
            API_RGB1_Scenery[RGBLDD_DeInitTmp][RGBLDD_DeInitTmp2] = 0x00;
            API_RGB2_Scenery[RGBLDD_DeInitTmp][RGBLDD_DeInitTmp2] = 0x00;    
                
            API_RGB1_SceneryTmp[RGBLDD_DeInitTmp][RGBLDD_DeInitTmp2] = 0;
            API_RGB2_SceneryTmp[RGBLDD_DeInitTmp][RGBLDD_DeInitTmp2] = 0;
            API_RGB3_SceneryTmp[RGBLDD_DeInitTmp][RGBLDD_DeInitTmp2] = 0;
        }
    }
}

/**
 *******************************************************************************
 * @brief	   Initial the other module for different LED driver.
 * @details     
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
void API_RGBLed_Init(void)
{
    DMA_BaseInitTypeDef     LED_DMA;
    
    /*DMA Initial ( DMA Channel0 )*/
    DMA_Cmd(ENABLE);  
    DMA_Channel_Cmd(DMAChannel0, ENABLE);
        
        
    DMA_BaseInitStructure_Init(&LED_DMA);
    LED_DMA.DMAChx = DMAChannel0;
    LED_DMA.DMALoopCmd           = DISABLE;
    LED_DMA.DestDINCSel          = ENABLE;  
    LED_DMA.SrcSymSel            = DMA_MEM_Read;
    LED_DMA.DestSymSel           = DMA_SPI0_TX;
    LED_DMA.BurstDataSize        = DMA_BurstSize_1Byte;
    LED_DMA.DMATransferNUM       = (API_LED_SHIFT_BYTE-1);
    DMA_Base_Init(&LED_DMA);    
    
    
    /*LED parameter initial*/
    API_RGB_CTR.RGB1_SceneryTmpBuf    = &API_RGB1_SceneryTmp[0][0];
    API_RGB_CTR.RGB2_SceneryTmpBuf    = &API_RGB2_SceneryTmp[0][0];
    API_RGB_CTR.RGB3_SceneryTmpBuf    = &API_RGB3_SceneryTmp[0][0];
    
}

/**
 *******************************************************************************
 * @brief	    
 * @details     
 * @return      
 * @exception   
 * @note        Need 110us in SPI Baudrate is 12Mhz
 *******************************************************************************
 */
void API_RGB_UpdateDataTrigger(void)
{


    API_RGB_CTR.RGB_UpdateStatus = ( API_RGB_UPDATE_GS_BLANK0 | API_RGB_UPDATE_GS_BLANK1 |
                                     API_RGB_UPDATE_GS        | API_RGB3_GS_DATA1        |
                                     API_RGB2_GS_DATA0        | API_RGB2_GS_DATA1        |   
                                     API_RGB1_GS_DATA0        | API_RGB1_GS_DATA1 );
    
    
    API_RGB_DATA_SIZE(API_RGB_GS_DATA0_BIT);
    
    DMAChannel0->NUM.W = ( API_LED_SHIFT_BYTE - 1);
    DMAChannel0->B.W = DMAChannel0->B.W | DMA_CH0B_CH0_SINC_mask_w;
    
    SPI_NSSOutputStatusControl_SW(SPI0, DRV_Low);
    
    
    SPI0->TDAT.H[0] = ( API_RGB_GS_SELECT | API_LED_UpdateDataBuf[2][0]);
    
}

/**
 *******************************************************************************
 * @brief	    
 * @details     
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
void API_RGBData_IRQHandler(void)
{
    API_RGB_CLEAR_UPDATEFALG();
    
    
    if((API_RGB_CTR.RGB_UpdateStatus & API_RGB_UPDATE_CONTROL ) == API_RGB_UPDATE_CONTROL)  
    {
        if( API_RGB_CTR.RGB_UpdateStatus & API_RGB_CONTROL_LATCH )
        {
            API_RGB_DATA_SIZE(API_RGB_CONTROL_RESERVE_BIT);
            
            API_RGB_DATA_SEND1BYTE( (API_RGB_UPDATE_CONTROL >> 24));
            
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus & (uint32_t)(~API_RGB_CONTROL_LATCH);
            return;
        }
        else if( API_RGB_CTR.RGB_UpdateStatus & API_RGB_CONTROL_RESERVE)
        {
            SPI0->CR0.W = SPI0->CR0.W | SPI_CR0_DMA_TXEN_mask_w;
            
            DMA->STA.W = DMA_FLAG_CH0_TCF;
            DMAChannel0->NUM.W = API_RGB_CONTROL_RESERVE_BYTE;
            API_RGB_DMA_SIZE( API_RGB_CONTROL_RESERVE_BYTE);
            DMAChannel0->B.W = DMAChannel0->B.W &(~DMA_CH0B_CH0_SINC_mask_w);
            DMAChannel0->SSA.W = (uint32_t)(&API_RGB_CTR.RGB_DUMMY);
            
            API_RGB_DMA_TRG();
            
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus & (uint32_t)(~API_RGB_CONTROL_RESERVE);
            return;
        }
        else if( API_RGB_CTR.RGB_UpdateStatus & API_RGB_CONTROL_FUNC1)
        {
            API_RGB_DATA_SIZE(API_RGB_CONTROL_FUNC1_BIT);
            
            API_RGB_DATA_SEND1BYTE( API_RGB_CTR.Function.B[2]);
            
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus & (uint32_t)(~API_RGB_CONTROL_FUNC1);
            return;
        }
        else if( API_RGB_CTR.RGB_UpdateStatus & API_RGB_CONTROL_FUNC0)
        {
            API_RGB_DATA_SIZE(API_RGB_CONTROL_FUNC0_BIT);

            API_RGB_DATA_SEND2BYTE( API_RGB_CTR.Function.H[0]);
        
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus & (uint32_t)(~API_RGB_CONTROL_FUNC0);
            return;
        }
        else if( API_RGB_CTR.RGB_UpdateStatus & API_RGB_CONTROL_BC)
        {
            API_RGB_DATA_SIZE(API_RGB_CONTROL_BC_BIT);

            API_RGB_DATA_SEND1BYTE( API_RGB_CTR.Brightness);
            
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus & (uint32_t)(~API_RGB_CONTROL_BC);
            return;
        }
        else if( API_RGB_CTR.RGB_UpdateStatus & API_RGB_CONTROL_DC)
        {
            SPI0->CR0.W = SPI0->CR0.W | SPI_CR0_DMA_TXEN_mask_w;
            
            DMA->STA.W = DMA_FLAG_CH0_TCF;
            DMAChannel0->NUM.W = API_RGB_CONTROL_DC_BYTE;
            DMAChannel0->B.W = DMAChannel0->B.W | DMA_CH0B_CH0_SINC_mask_w;
            DMAChannel0->SSA.W = (uint32_t)(&API_RGB_CTR.Dot[0]);

            API_RGB_DMA_TRG();
            
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus - 1;
            
            if( (API_RGB_CTR.RGB_UpdateStatus & API_RGB_CONTROL_CNTMASK) != 0 )
            {
                API_RGB_CTR.RGB_UpdateStatus =( API_RGB_CTR.RGB_UpdateStatus | 
                                                API_RGB_UPDATE_CONTROL       |  API_RGB_CONTROL_LATCH   | API_RGB_CONTROL_RESERVE | 
                                                API_RGB_CONTROL_FUNC0        |  API_RGB_CONTROL_FUNC1   | API_RGB_CONTROL_BC      | API_RGB_CONTROL_DC );
                return;
            }
            else if((API_RGB_CTR.RGB_UpdateStatus &  API_RGB_UPDATE_GS_BLANK0) == API_RGB_UPDATE_GS_BLANK0)
            {
                API_RGB_CTR.RGB_UpdateStatus = API_RGB_UPDATE_GS_BLANK0;
                
                return;
            }
            else if((API_RGB_CTR.RGB_UpdateStatus &  API_RGB_UPDATE_GS_BLANK1) == API_RGB_UPDATE_GS_BLANK1)
            {
                API_RGB_CTR.RGB_UpdateStatus = API_RGB_UPDATE_GS_BLANK1;
                
                return;
            }
            
            
            API_RGB_CTR.RGB_UpdateStatus = API_LED_IDLE;
            
            return;
        }
    }
    else if((API_RGB_CTR.RGB_UpdateStatus & API_RGB_UPDATE_GS ) == API_RGB_UPDATE_GS)  
    {
        if( API_RGB_CTR.RGB_UpdateStatus & API_RGB3_GS_DATA1)
        {
            SPI0->CR0.W = SPI0->CR0.W | SPI_CR0_DMA_TXEN_mask_w;
            
            API_RGB_DATA_SIZE(API_RGB_GS_DATA1_BIT);
            
            DMA->STA.W = DMA_FLAG_CH0_TCF;
            DMAChannel0->SSA.W = (uint32_t)(&API_LED_UpdateDataBuf[2][1]);
            DMAChannel0->A.W = DMAChannel0->A.W | DMA_CH0A_CH0_REQ_mask_w;
            
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus & (uint32_t)(~(API_RGB3_GS_DATA1));
            return;
        }
        else if( API_RGB_CTR.RGB_UpdateStatus & API_RGB2_GS_DATA0)
        {
            API_RGB_DATA_SIZE(API_RGB_GS_DATA0_BIT);
            
            SPI0->TDAT.H[0] = ( API_RGB_GS_SELECT | API_LED_UpdateDataBuf[1][0]);
            
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus & (uint32_t)(~API_RGB2_GS_DATA0);
            return;
        }
        else if( API_RGB_CTR.RGB_UpdateStatus & API_RGB2_GS_DATA1)
        {
            SPI0->CR0.W = SPI0->CR0.W | SPI_CR0_DMA_TXEN_mask_w;
            
            API_RGB_DATA_SIZE(API_RGB_GS_DATA1_BIT);
            DMA->STA.W = DMA_FLAG_CH0_TCF;
            DMAChannel0->SSA.W = (uint32_t)(&API_LED_UpdateDataBuf[1][1]);
            DMAChannel0->A.W = DMAChannel0->A.W | DMA_CH0A_CH0_REQ_mask_w;
            
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus & (uint32_t)(~API_RGB2_GS_DATA1);
            return;
        }
        else if( API_RGB_CTR.RGB_UpdateStatus & API_RGB1_GS_DATA0)
        {
            API_RGB_DATA_SIZE(API_RGB_GS_DATA0_BIT);
            
            SPI0->TDAT.H[0] = ( API_RGB_GS_SELECT | API_LED_UpdateDataBuf[0][0] );
            
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus & (uint32_t)(~API_RGB1_GS_DATA0);
            return;
        }
        else if( API_RGB_CTR.RGB_UpdateStatus & API_RGB1_GS_DATA1)
        {
            SPI0->CR0.W = SPI0->CR0.W | SPI_CR0_DMA_TXEN_mask_w;
            
            API_RGB_DATA_SIZE(API_RGB_GS_DATA1_BIT);
            DMA->STA.W = DMA_FLAG_CH0_TCF;
            DMAChannel0->SSA.W = (uint32_t)(&API_LED_UpdateDataBuf[0][1]);
            DMAChannel0->A.W = DMAChannel0->A.W | DMA_CH0A_CH0_REQ_mask_w;
            
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus & (uint32_t)(~API_RGB1_GS_DATA1);
            return;
        }
        else if( (API_RGB_CTR.RGB_UpdateStatus & ( API_RGB_UPDATE_GS_BLANK0 | API_RGB_UPDATE_GS_BLANK1)) == ( API_RGB_UPDATE_GS_BLANK0 | API_RGB_UPDATE_GS_BLANK1))
        {    
            SPI0->CR2.W = SPI0->CR2.W | SPI_CR2_NSS_SWO_mask_w;
            API_RGB_CTR.Function.MBIT.BLANK  = 1;
            API_RGB_UpdateControlTrigger();
            
            API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus | API_RGB_UPDATE_GS_BLANK0;
            
            return;
        }
        
        
        API_RGB_CTR.RGB_UpdateStatus = API_LED_IDLE;
        SPI0->CR2.W = SPI0->CR2.W | SPI_CR2_NSS_SWO_mask_w;
        
        return;
    }
    else if( API_RGB_CTR.RGB_UpdateStatus == API_RGB_UPDATE_GS_BLANK0)
    {
        SPI0->CR2.W = SPI0->CR2.W | SPI_CR2_NSS_SWO_mask_w;
        API_RGB_CTR.Function.MBIT.BLANK  = 0;
        API_RGB_UpdateControlTrigger();
        
        API_RGB_CTR.RGB_UpdateStatus = API_RGB_CTR.RGB_UpdateStatus | API_RGB_UPDATE_GS_BLANK1;
        
        return;
    }
    else if( API_RGB_CTR.RGB_UpdateStatus == API_RGB_UPDATE_GS_BLANK1)
    {
        
        API_AllRGB_OFF();
        
        SPI0->CR2.W = SPI0->CR2.W | SPI_CR2_NSS_SWO_mask_w;
        
        API_RGB_GCLK_TRG_PIN = 1;
        API_RGB_GCLK_TRG_PIN = 0;
        
        switch( API_RGB_CTR.RGB_LINE_ON)
        {
            case 0:
            case 1:
            case 10:                
                    GPIO_ClearPortBit( GPIOB, API_RGB_PWPIN_TABLE[API_RGB_CTR.RGB_LINE_ON]);
                    break;
            default:
                    GPIO_ClearPortBit( GPIOD, API_RGB_PWPIN_TABLE[API_RGB_CTR.RGB_LINE_ON]); 
                    break;
        }
    
        API_RGB_CTR.RGB_UpdateStatus = API_LED_IDLE;
        
        return;
    }
    
    
    SPI0->CR2.W = SPI0->CR2.W | SPI_CR2_NSS_SWO_mask_w;
    
}

/**
 *******************************************************************************
 * @brief	    
 * @details     
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
void API_RGB_UpdateNextLineData( FunctionalState RGB_UpdateNewData)
{
    uint8_t RGB_UpNLData_Tmp;
    
    if( RGB_UpdateNewData == ENABLE)
    {
        for( RGB_UpNLData_Tmp = 0; RGB_UpNLData_Tmp < API_LED_SHIFT_BYTE; RGB_UpNLData_Tmp++)
        {
            API_RGB1_Scenery[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp] = API_RGB1_SceneryTmp[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp];                                                    
            API_RGB2_Scenery[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp] = API_RGB2_SceneryTmp[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp];                                                 
            API_RGB3_Scenery[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp] = API_RGB3_SceneryTmp[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp]; 
            
            API_LED_UpdateDataBuf[0][RGB_UpNLData_Tmp] = API_RGB1_Scenery[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp];
            API_LED_UpdateDataBuf[1][RGB_UpNLData_Tmp] = API_RGB2_Scenery[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp];
            API_LED_UpdateDataBuf[2][RGB_UpNLData_Tmp] = API_RGB3_Scenery[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp];   
        }
    }
    else
    {
        for( RGB_UpNLData_Tmp = 0; RGB_UpNLData_Tmp < API_LED_SHIFT_BYTE; RGB_UpNLData_Tmp++)
        {
            API_LED_UpdateDataBuf[0][RGB_UpNLData_Tmp] = API_RGB1_Scenery[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp];
            API_LED_UpdateDataBuf[1][RGB_UpNLData_Tmp] = API_RGB2_Scenery[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp];
            API_LED_UpdateDataBuf[2][RGB_UpNLData_Tmp] = API_RGB3_Scenery[API_RGB_CTR.RGB_LINE_ON][RGB_UpNLData_Tmp];
        }  
    }
}

/**
 *******************************************************************************
 * @brief	    
 * @details     
 * @return      
 * @exception   
 * @note        
 *******************************************************************************
 */
void API_RGB_UpdateDateDramaTmp(void)
{
    uint8_t  API_RGB_UDramaLine = API_RGB_CTR.RGB_RefreshColumn - 6;
    uint8_t  API_RGB_UDramaTmp;
    
    uint16_t API_RGB_UDramaDataTmp;

    for( API_RGB_UDramaTmp = 0; API_RGB_UDramaTmp < API_RGB_KBTotalColume; API_RGB_UDramaTmp++)
    {
        
        API_RGB_UDramaDataTmp       = ( API_RGB_PWMData_Table[API_RGB_UDramaLine ][ API_RGB_UDramaTmp] & API_RGB_PWMDATA_MASK);
        
        
        switch(( API_RGB_PWMData_Table[API_RGB_UDramaLine ][ API_RGB_UDramaTmp] & API_RGB_PWMIC_MASK))
        {
            case API_RGB_PWMIC1:
                                API_RGB_CTR.RGB1_SceneryTmpBuf[API_RGB_UDramaDataTmp    ] = 0;
                                API_RGB_CTR.RGB1_SceneryTmpBuf[API_RGB_UDramaDataTmp + 1] = API_KBRGB_BColorScenery[API_RGB_UDramaLine][API_RGB_UDramaTmp];
                                API_RGB_CTR.RGB1_SceneryTmpBuf[API_RGB_UDramaDataTmp + 2] = 0;
                                API_RGB_CTR.RGB1_SceneryTmpBuf[API_RGB_UDramaDataTmp + 3] = API_KBRGB_RColorScenery[API_RGB_UDramaLine][API_RGB_UDramaTmp];
                                API_RGB_CTR.RGB1_SceneryTmpBuf[API_RGB_UDramaDataTmp + 4] = 0;
                                API_RGB_CTR.RGB1_SceneryTmpBuf[API_RGB_UDramaDataTmp + 5] = API_KBRGB_GColorScenery[API_RGB_UDramaLine][API_RGB_UDramaTmp];
                                break;
            case API_RGB_PWMIC2:
                                API_RGB_CTR.RGB2_SceneryTmpBuf[API_RGB_UDramaDataTmp    ] = 0;
                                API_RGB_CTR.RGB2_SceneryTmpBuf[API_RGB_UDramaDataTmp + 1] = API_KBRGB_BColorScenery[API_RGB_UDramaLine][API_RGB_UDramaTmp];
                                API_RGB_CTR.RGB2_SceneryTmpBuf[API_RGB_UDramaDataTmp + 2] = 0;
                                API_RGB_CTR.RGB2_SceneryTmpBuf[API_RGB_UDramaDataTmp + 3] = API_KBRGB_RColorScenery[API_RGB_UDramaLine][API_RGB_UDramaTmp];
                                API_RGB_CTR.RGB2_SceneryTmpBuf[API_RGB_UDramaDataTmp + 4] = 0;
                                API_RGB_CTR.RGB2_SceneryTmpBuf[API_RGB_UDramaDataTmp + 5] = API_KBRGB_GColorScenery[API_RGB_UDramaLine][API_RGB_UDramaTmp];
                                break;
            case API_RGB_PWMIC3:
                                API_RGB_CTR.RGB3_SceneryTmpBuf[API_RGB_UDramaDataTmp    ] = 0;
                                API_RGB_CTR.RGB3_SceneryTmpBuf[API_RGB_UDramaDataTmp + 1] = API_KBRGB_BColorScenery[API_RGB_UDramaLine][API_RGB_UDramaTmp];
                                API_RGB_CTR.RGB3_SceneryTmpBuf[API_RGB_UDramaDataTmp + 2] = 0;
                                API_RGB_CTR.RGB3_SceneryTmpBuf[API_RGB_UDramaDataTmp + 3] = API_KBRGB_RColorScenery[API_RGB_UDramaLine][API_RGB_UDramaTmp];
                                API_RGB_CTR.RGB3_SceneryTmpBuf[API_RGB_UDramaDataTmp + 4] = 0;
                                API_RGB_CTR.RGB3_SceneryTmpBuf[API_RGB_UDramaDataTmp + 5] = API_KBRGB_GColorScenery[API_RGB_UDramaLine][API_RGB_UDramaTmp];
                                break;
            default:
                                break;
        }
    }
}


/**
 *******************************************************************************
 * @brief	    
 * @details     
 * @return      
 * @exception   
 * @note        Need 110us in SPI Baudrate is 12Mhz
 *******************************************************************************
 */
static void API_RGB_UpdateControlTrigger(void)
{
    API_RGB_CTR.RGB_UpdateStatus = ( API_RGB_UPDATE_CONTROL  | API_RGB_CONTROL_RESERVE | 
                                     API_RGB_CONTROL_FUNC0   | API_RGB_CONTROL_FUNC1   | API_RGB_CONTROL_BC | API_RGB_CONTROL_DC | 
                                     API_RGB_DRIVER_TOTAL);


    API_RGB_DATA_SIZE(API_RGB_CONTROL_RESERVE_BIT);

    
    API_RGB_CTR.RGB_DUMMY = 0xFF;

    SPI0->CR2.W = SPI0->CR2.W & (~SPI_CR2_NSS_SWO_mask_w);
    
    SPI0->TDAT.B[0] = (API_RGB_UPDATE_CONTROL >> 24);
}


#endif


