
/**
 ******************************************************************************
 *
 * @file        BSP_Init.c
 * @brief       This is BSP initial C file.
 
 * @par         Project
 *              MG32
 * @version     V1.04
 * @date        2023/07/03
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 
/* Includes ------------------------------------------------------------------*/
//Module
#include "MG32_GPIO_DRV.h"
#include "MG32_EXIC_DRV.h"
#include "MG32_IWDT_DRV.h"
#include "MG32_URT_DRV.h"
#include "MG32_CSC_Init.h"
#include "MG32.h"

//TH222x Control
#include "TH222x_Global.h"

//BSP
#if BSP_2_ARGB == 1
    #include "BSP_2_ARGB_WS2812.h"
#endif  
#if BSP_3_RGB == 1  
    #include "BSP_3_RGB.h"
#endif
#if BSP_7_STEP_MOTOR == 1 
    #include "BSP_7_StepMotor.h" 
#endif
#if BSP_8_VAR == 1
    #include "BSP_8_VariableResistor.h"
#endif
#if BSP_9_ROTARY_ENCODER == 1
    #include "BSP_9_RotaryEncoder.h"
#endif
#if BSP_10_BUZZER == 1
    #include "BSP_10_Buzzer.h"
#endif
#if BSP_11_RC_SERVO_MOTOR == 1
    #include "BSP_11_RCServoMotor.h"
#endif
#if BSP_12_BLE == 1
    #if BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_HC06
        #include "BSP_12_BLE_HC06.h"
    #endif
    #if BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_HC42
        #include "BSP_12_BLE_HC42.h"
    #endif
#endif
#if BSP_15_4X4_KEYBOARD == 1
    #include "BSP_15_4x4Keyboard.h"
#endif
#if BSP_17_2COLOR_DOTMATRIX_LED == 1
    #include "BSP_17_2ColorDotMatrixLED.h"
#endif
#if BSP_18_7SEGMENT_DISPLAY == 1
    #include "BSP_18_7Segment.h" 
#endif
#if BSP_19_LCD == 1
    #include "BSP_19_LCD.h"
#endif

/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
// ----------------------------------------------------------------------------
// GPIO
#define TH223x_LEDRYG_GPIOX     GPIOE
#define TH223x_LEDRYG_IOM       IOME
                                
#define TH223x_LEDR_PXPin       PX_Pin_13
#define TH223x_LEDY_PXPin       PX_Pin_14
#define TH223x_LEDG_PXPin       PX_Pin_15
                                
#define TH223x_LEDR_PINX        PINE(13)
#define TH223x_LEDY_PINX        PINE(14)
#define TH223x_LEDG_PINX        PINE(15)
                                
#define TH223x_SW34_GPIOX       GPIOB
#define TH223x_SW34_IOM         IOMB
                                
#define TH223x_SW3_PXPin        PX_Pin_10
#define TH223x_SW4_PXPin        PX_Pin_11
                                
#define TH223x_SW3_PINX         PINB(10)
#define TH223x_SW4_PINX         PINB(11)
                                
#define TH223x_SW3_PIN          PB10
#define TH223x_SW4_PIN          PB11

// ----------------------------------------------------------------------------
// EXIC
#define TH223x_SW34_EXIC_PX     EXIC_PB

#define TH223x_SW34_EXIC_IT     EXIC_PB_IT

#define TH223x_SW3_EXIC_TRGSPIN EXIC_TRGS_PIN10       
#define TH223x_SW4_EXIC_TRGSPIN EXIC_TRGS_PIN11

#define TH223x_SW3_EXIC_PINX    EXIC_PX_PIN10
#define TH223x_SW4_EXIC_PINX    EXIC_PX_PIN11

#define TH223x_SW34_IRQ         EXINT1_IRQn

// ----------------------------------------------------------------------------
// System Tick
#define SYSTICK_CLK             CONF_CK_AHB_FREQ

/* Private typedef -----------------------------------------------------------*/

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
TH222xCTR_TypeDef  TH222x_CTR;

//#if BSP_12_BLE == 1 && BSP_2_ARGB == 1
//    uint8_t TH222x_BLE_ARGDelay;
//#endif

/* Private function prototypes -----------------------------------------------*/
void SysTick_Handler(void);
void EXINT1_IRQHandler(void);
void URT4x_IRQHandler(void);

static uint32_t TH222x_GetSysTickCount(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/


/**
 ******************************************************************************
 * @brief       SysTick IRQ handler routine. (trigger unit is 500us)
 * @details     Trigger module event through by DramaAct.
 * @return      None
 ******************************************************************************
 */
void SysTick_Handler(void)
{
    // ------------------------------------------------------------------------
    TH222x_CTR.SysTickCount = TH222x_CTR.SysTickCount + 1;  // SysTickCount ++;

    // ------------------------------------------------------------------------
    // (500us) Trigger TH222x_2ColorMatrixLED_UpdateFlag to '1' when BSP_17_2COLOR_DOTMATRIX_LED=1
    // ------------------------------------------------------------------------
    #if BSP_17_2COLOR_DOTMATRIX_LED == 1
        TH222x_2ColorMatrixLED_UpdateFlag = 1;
    #endif
    
    // ------------------------------------------------------------------------
    // (Unit 500us) to trigger TH222x module's UpdateFlag
    // ------------------------------------------------------------------------
    switch(TH222x_CTR.DramaAct)
    {
        // --------------------------------------------------------------------
        // case 0: Config TH222x_CTR.UpdateFlag,  TH222x_RGB_UpdateFlag, 
        //         TH222x_ARGB_UpdateFlag, TH222x_StepMotor_UpdateFlag to '1'.
        // --------------------------------------------------------------------
        case 0:
                TH222x_CTR.UpdateFlag = 1;
        
                #if BSP_2_ARGB == 1
                        TH222x_ARGB_UpdateFlag = 1;
                #endif
        
                #if BSP_3_RGB == 1 && BSP_2_ARGB == 1
                    TH222x_RGB_UpdateFlag  = TH222x_ARGB_UpdateFlag;
                #elif BSP_3_RGB == 1
                    TH222x_RGB_UpdateFlag  = 1;    
                #endif
        
                #if BSP_7_STEP_MOTOR == 1
                    TH222x_StepMotor_UpdateFlag = 1;
                #endif
                break;
        // --------------------------------------------------------------------
        // case 2: Config TH222x_VR_UpdateFlag,  TH222x_RotaryEncoder_UpdateFlag to '1'.
        // --------------------------------------------------------------------
        case 2: // 1ms        
                #if BSP_8_VAR == 1
                    TH222x_VR_UpdateFlag = 1;
                #endif
                #if BSP_9_ROTARY_ENCODER == 1
                    TH222x_RotaryEncoder_UpdateFlag  = 1;
                #endif
                
                break;
        // --------------------------------------------------------------------
        // case 2: Config TH222x_RCServoMotor_UpdateFlag to '1'.
        // --------------------------------------------------------------------
        case 4: // 2ms
                #if BSP_11_RC_SERVO_MOTOR == 1
                    TH222x_RCServoMotor_UpdateFlag = 1;
                #endif
                break;
        // --------------------------------------------------------------------
        // case 6: Config TH222x_4x4Keyboard_UpdateFlag, BSP_18_7SEGMENT_DISPLAY to '1'.
        // --------------------------------------------------------------------
        case 6: // 3ms
                #if BSP_15_4X4_KEYBOARD == 1
                    TH222x_4x4Keyboard_UpdateFlag = 1;      // User press 4x4 keyboard then
                                                            // it will output to BLE module (BSP_12_BLE=1).
                #endif
                #if BSP_18_7SEGMENT_DISPLAY == 1
                    TH222x_7Segment_UpdateFlag = 1;
                #endif 
                break;
        // --------------------------------------------------------------------
        // case 9: Config TH222x_BLE_UpdateFlag to '1'.
        //         Clear TH222x_CTR.DramaAct.
        // --------------------------------------------------------------------
        case 9: // 4.5ms    
                #if BSP_12_BLE == 1
                    TH222x_BLE_UpdateFlag = 1;
                #endif
                TH222x_CTR.DramaAct = 0;
                return;
        default:
                break;            
    }
    
    // ------------------------------------------------------------------------
    // Increase DramaAct
    // ------------------------------------------------------------------------
    TH222x_CTR.DramaAct = TH222x_CTR.DramaAct + 1;

}

/**
 *******************************************************************************
 * @brief	   SW3 & SW4 IRQHandler function.
 * @details    When press SW3 button then set SW3_UpdateFlag.
 *             Or press SW4 button then set SW4_UpdateFlag.
 * @return     None
 * @note       No
 *******************************************************************************
 */
void EXINT1_IRQHandler(void)
{
    uint32_t EXINT1_IRQHandlerTmp;
    
    
    EXINT1_IRQHandlerTmp = EXIC_GetPxAllTriggerEventFlagStatus(EXIC_PB);
    
    // ------------------------------------------------------------------------
    // if press TH223x's SW3 button
    // ------------------------------------------------------------------------
    if(EXINT1_IRQHandlerTmp & TH223x_SW3_EXIC_PINX)
    {
        // Clear EXIC's flag and config SW3_UpdateFlag event
        EXIC_ClearPxTriggerEventFlag(EXIC_PB, TH223x_SW3_EXIC_PINX);
        TH222x_CTR.SW3_UpdateFlag = 1;
    }
    
    // ------------------------------------------------------------------------
    // if press TH223x's SW4 button
    // ------------------------------------------------------------------------
    if(EXINT1_IRQHandlerTmp & TH223x_SW4_EXIC_PINX)
    {
        // Clear EXIC's flag and config SW4_UpdateFlag event
        EXIC_ClearPxTriggerEventFlag(EXIC_PB, TH223x_SW4_EXIC_PINX);
        TH222x_CTR.SW4_UpdateFlag = 1;
    }
}

/**
// *******************************************************************************
// * @brief	 DEBUG & BLE URT IRQHandler function.  
// * @details     
// * @return      
// * @note       
// *******************************************************************************
// */
#if BSP_12_BLE == 1
void URT4x_IRQHandler(void)
{
    uint8_t URT4x_IDTmp;

    URT4x_IDTmp = __DRV_EXIC_GET_ID22_SOURCE();

    //=====================================================
    //BLE
    //URT4
        #if BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_HC06 || BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_HC42
            if( (URT4x_IDTmp & 0x01) == 0x01)
            {
                BSP_BLE_IRQHandler();
            }
        #endif
        //URT5
        if( (URT4x_IDTmp & 0x02) == 0x02)
        {
            TH222x_BLE_URTIRQHandler();
        }
}
#endif

/**
 *******************************************************************************
 * @brief       Delay X ms
 * @details     Use SysTick_Handler to get current counter.
 * @param[in]   Delay : Needs delay time. (unit 500us)
 * @return      None
 *******************************************************************************
 */
void TH222x_Delay(__IO uint32_t Delay)
{
    uint32_t tickstart = 0;
    uint32_t ticknow = 0;
    uint32_t MID_Delay_wait = 0;
  
    // ------------------------------------------------------------------------
    // Get current tick time
    // ------------------------------------------------------------------------
    tickstart        = TH222x_GetSysTickCount();
    
    // ------------------------------------------------------------------------
    // Add a period to guarantee minimum wait 
    // ------------------------------------------------------------------------
    MID_Delay_wait   = Delay;
    if (MID_Delay_wait < 0xFFFFFFFF)
    {
       MID_Delay_wait++;
    }
  
    // ------------------------------------------------------------------------
    // until timeout
    // ------------------------------------------------------------------------
    do{
        ticknow = TH222x_GetSysTickCount();
    } while((ticknow- tickstart) < MID_Delay_wait);
    
}

/**
 *******************************************************************************
 * @brief       Deinitial module control parameters	of TH222x board.  
 * @details     Deinitial CTR/SW3/SW4/ARGB/LED ... /LCD parameters.
 * @return      None
 *******************************************************************************
 */
void TH222x_Parameter_DeInit(void)
{
    // ------------------------------------------------------------------------
    // Systick parameter initial
    // ------------------------------------------------------------------------
    TH222x_CTR.SysTickCount       = 0;
    
    // ------------------------------------------------------------------------
    // TH222x common parameter initial
    // ------------------------------------------------------------------------
    TH222x_CTR.UpdateFlag          = 0;                     // Configure UpdateFlag to '1' in SysTick_Handler
    TH222x_CTR.DramaAct            = TH222x_CTR.DramaAct + 1;// Trigger TH222x module UpdateFlag through DramaAct.
    
    // ------------------------------------------------------------------------
    // TH223x Button software parameter initial  
    // ------------------------------------------------------------------------
    TH222x_CTR.SW3_UpdateFlag      = 0;                     // SW3(PB10) 0: No update event.  1: Has update event.
    TH222x_CTR.SW3_TRGlag          = 0;                     // Clear SW3 press event
    TH222x_CTR.SW3_DramaTime       = 0;                     // Clear SW3 Time counter
                               
    // ------------------------------------------------------------------------
    TH222x_CTR.SW4_UpdateFlag      = 0;                     // SW4(PB11) 0: No update event.  1: Has update event.
    TH222x_CTR.SW4_TRGlag          = 0;                     // Clear SW4 press event
    TH222x_CTR.SW4_DramaTime       = 0;                     // Clear SW4 Time counter
                               
    // ------------------------------------------------------------------------
    TH222x_CTR.SW_DramaTimeMax     = 7;                     // SW_DramaTimeDefault=4 (0~SW_DramaTimeMax=7)
    TH222x_CTR.SW_DramaTimeDefault = 4;
    
    // ------------------------------------------------------------------------
    // TH223x LED software parameter initial
    // ------------------------------------------------------------------------
    TH222x_CTR.LED_CountMax        = 200;
    
    TH222x_CTR.GLED_Count          = 0;
    TH222x_CTR.RLED_Count          = 0;
    TH222x_CTR.YLED_Count          = 0;
    
    TH222x_CTR.GLED_Lock           = 1;
    TH222x_CTR.RLED_Lock           = 1;
    TH222x_CTR.YLED_Lock           = 1;
    
    /**/
//    #if BSP_12_BLE == 1 & BSP_2_ARGB == 1
//        TH222x_BLE_ARGDelay = 0;
//    #endif

    // ------------------------------------------------------------------------
    // TH222x functions parameter initial. 
    // ------------------------------------------------------------------------
    #if BSP_0_DEBUG == 1
        TH222x_DEBUGParameter_Init();
    #endif
    // ------------------------------------------------------------------------
    // Deinitial ARGB control parameters
    // ------------------------------------------------------------------------
    #if BSP_2_ARGB == 1
        TH222x_ARGBParameter_DeInit();
    #endif
    // ------------------------------------------------------------------------
    // Deinitial RGB LED control parameters
    // ------------------------------------------------------------------------
    #if BSP_3_RGB == 1
        TH222x_RGBParameter_DeInit();
    #endif
    // ------------------------------------------------------------------------
    // Deinitial Step Motor control parameters
    // ------------------------------------------------------------------------
    #if BSP_7_STEP_MOTOR == 1
        TH222x_StepMotorParameter_DeInit();
    #endif
    // ------------------------------------------------------------------------
    // Deinitial variable Resistor control parameters
    // ------------------------------------------------------------------------
    #if BSP_8_VAR == 1
        TH222x_VariableResistorParameter_DeInit();
    #endif
    // ------------------------------------------------------------------------
    // Deinitial QEI-Rotary encoder control parameters
    // ------------------------------------------------------------------------
    #if BSP_9_ROTARY_ENCODER == 1
        TH222x_RotaryEncoderParameter_DeInit();
    #endif
    // ------------------------------------------------------------------------
    // Deinitial Buzzer control parameters
    // ------------------------------------------------------------------------
    #if BSP_10_BUZZER == 1
        TH222x_BuzzerParameter_DeInit();
    #endif
    // ------------------------------------------------------------------------
    // Deinitial RC Servo Motor control parameters
    // ------------------------------------------------------------------------
    #if BSP_11_RC_SERVO_MOTOR == 1
        TH222x_RCServoMotorParameter_DeInit();
    #endif
    // ------------------------------------------------------------------------
    // BLE does not support GCC
    // ------------------------------------------------------------------------
    #if BSP_12_BLE == 1
        TH222x_BLEParameter_DeInit();
    #endif
    // ------------------------------------------------------------------------
    // Deinitial 4x4 keyboard control parameters
    // ------------------------------------------------------------------------
    #if BSP_15_4X4_KEYBOARD == 1
        TH222x_4x4KeyboardParameter_DeInit();
    #endif
    // ------------------------------------------------------------------------
    // Deinitial 2 colors dot matrix LED control parameters
    // ------------------------------------------------------------------------
    #if BSP_17_2COLOR_DOTMATRIX_LED == 1
        TH222x_2ColorDotMatrixParameter_DeInit();
    #endif
    // ------------------------------------------------------------------------
    // Deinitial 4x7-SEG display control parameters
    // ------------------------------------------------------------------------
    #if BSP_18_7SEGMENT_DISPLAY == 1
        TH222x_7SegmentParameter_DeInit();
    #endif
    // ------------------------------------------------------------------------
    // Deinitial LCD control parameters
    // ------------------------------------------------------------------------
    #if BSP_19_LCD == 1
        TH222x_LCDParameter_DeInit();
    #endif
    
}

/**
 *******************************************************************************
 * @brief       TH223x initial routine	   
 * @details     1. Disable IWDT.
 *              2. Initial SW3/SW4 pin with EXIC interrupt.
 *              3. Initial LEDs pin I/O mode.
 *              4. Initial SysTick.
 *              5. Initial BSP test module.
 *              6. Delay a time.
 * @return      None
 *******************************************************************************
 */
void TH222x_Init(void)
{
    PIN_InitTypeDef    TH223x_PIN;
    EXIC_TRGSTypeDef   TH223x_EIXC;
    
    // ------------------------------------------------------------------------
    // Disable IWDT (for BSP_ARGB_CpltCallback)
    // ------------------------------------------------------------------------
    UnProtectModuleReg(IWDTprotect);
    IWDT_Cmd(DISABLE);
    ProtectModuleReg(IWDTprotect);
    
    // ------------------------------------------------------------------------
    // TH223A SW Control Pin and EXIC Initial. 
    // ------------------------------------------------------------------------
    // GPIO Initial (SW3(PB10), SW4(PB11) Configs Open-drain with Pull-up)    
    GPIO_SetPortBit(TH223x_SW34_GPIOX, (TH223x_SW3_PXPin | TH223x_SW4_PXPin ));
    
    TH223x_PIN.PINX_Mode               = PINX_Mode_OpenDrain_O;
    TH223x_PIN.PINX_PUResistant        = PINX_PUResistant_Enable;
    TH223x_PIN.PINX_Speed              = PINX_Speed_Low;
    TH223x_PIN.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    TH223x_PIN.PINX_FilterDivider      = PINX_FilterDivider_16;         // Digital input function filter is GPIO Clock Freq / 16.
    TH223x_PIN.PINX_Inverse            = PINX_Inverse_Disable;
    TH223x_PIN.PINX_Alternate_Function = 0;
    
    GPIO_PinMode_Config(TH223x_SW3_PINX,&TH223x_PIN);
    GPIO_PinMode_Config(TH223x_SW4_PINX,&TH223x_PIN);
    
    // ------------------------------------------------------------------------
    // EXIC Initial (SW3(PB10), SW4(PB11))
    // ------------------------------------------------------------------------
    TH223x_EIXC.EXIC_Pin       = TH223x_SW3_EXIC_TRGSPIN | TH223x_SW4_EXIC_TRGSPIN;
    TH223x_EIXC.EXIC_TRGS_Mode = Edge;
    
    EXIC_PxTriggerMode_Select(TH223x_SW34_EXIC_PX,&TH223x_EIXC);
    
    //
    EXIC_ClearPxTriggerEventFlag(TH223x_SW34_EXIC_PX,(TH223x_SW3_EXIC_PINX | TH223x_SW4_EXIC_PINX));
    EXIC_PxTriggerOrMask_Select(TH223x_SW34_EXIC_PX,(TH223x_SW3_EXIC_PINX | TH223x_SW4_EXIC_PINX));
    
    // Enable interrupt 
    EXIC_PxTriggerITEA_Cmd(TH223x_SW34_EXIC_IT, ENABLE);
    NVIC_EnableIRQ(TH223x_SW34_IRQ);
    
    // ------------------------------------------------------------------------
    // TH223A LED Pin Initial. (I/O mode is PPO)
    // ------------------------------------------------------------------------
    GPIO_SetPortBit(TH223x_LEDRYG_GPIOX, (TH223x_LEDR_PXPin | TH223x_LEDY_PXPin | TH223x_LEDG_PXPin));

    TH223x_PIN.PINX_Mode               = PINX_Mode_PushPull_O;
    TH223x_PIN.PINX_PUResistant        = PINX_PUResistant_Disable;
    TH223x_PIN.PINX_Speed              = PINX_Speed_Low;
    TH223x_PIN.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    TH223x_PIN.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    TH223x_PIN.PINX_Inverse            = PINX_Inverse_Disable;
    TH223x_PIN.PINX_Alternate_Function = 0;
    
    GPIO_PinMode_Config(TH223x_LEDR_PINX,&TH223x_PIN);
    GPIO_PinMode_Config(TH223x_LEDY_PINX,&TH223x_PIN);
    GPIO_PinMode_Config(TH223x_LEDG_PINX,&TH223x_PIN);

    // Turn off LEDs
    TH223x_LED_RED_OFF();
    TH223x_LED_YELLOW_OFF();
    TH223x_LED_GREEN_OFF();
    
    // ------------------------------------------------------------------------
    // System Tick Initial.(The Project Time base)
    // ------------------------------------------------------------------------
    InitTick(SYSTICK_CLK/2,0);         // 500us
    
    
    // ------------------------------------------------------------------------
    // Initial URT0 to show debug information.
    // ------------------------------------------------------------------------
    #if BSP_0_DEBUG                 == 1
        TH222x_DEBUG_Init();
    #endif
    // ------------------------------------------------------------------------
    // Initial ARGB LED (WS2812) through by URT0 module
    // ------------------------------------------------------------------------
    #if BSP_2_ARGB                  == 1
        BSP_ARGB_Init();
    #endif        
    // ------------------------------------------------------------------------
    // Initial TM36 output PWM to control RGB LEDs.                                
    // ------------------------------------------------------------------------
    #if BSP_3_RGB                   == 1    
        BSP_RGB_Init();
    #endif               
    // ------------------------------------------------------------------------
    // Initial GPIO control step motor.                     
    // ------------------------------------------------------------------------
    #if BSP_7_STEP_MOTOR            == 1
        BSP_StepMotor_Init();     
    #endif
    // ------------------------------------------------------------------------
    // Initial ADC sample variable resistor                                        
    // ------------------------------------------------------------------------
    #if BSP_8_VAR                   == 1
        BSP_VR_Init();        
    #endif
    // ------------------------------------------------------------------------
    // Initial TM26 QEI functionality for rotary encoder.            
    // ------------------------------------------------------------------------
    #if BSP_9_ROTARY_ENCODER        == 1
        BSP_RotaryEncoder_Init();        
    #endif
    // ------------------------------------------------------------------------
    // Initial GPIO to control buzzer.                                                
    // ------------------------------------------------------------------------
    #if BSP_10_BUZZER               == 1
        BSP_Buzzer_Init();        
    #endif
    // ------------------------------------------------------------------------
    // Initial TM20 to control RC servo motor.                     
    // ------------------------------------------------------------------------
    #if BSP_11_RC_SERVO_MOTOR       == 1
        BSP_RCServoMotor_Init();        
    #endif
    // ------------------------------------------------------------------------
    // The BLE library does not support GCC.
    // ------------------------------------------------------------------------
    #if BSP_12_BLE                   == 1
        TH222x_BLE_Init();
    #endif

    // ------------------------------------------------------------------------
    // Initial GPIO for 4x4 Keyboard state.                                 
    // ------------------------------------------------------------------------
    #if BSP_15_4X4_KEYBOARD         == 1
        BSP_4X4Keyboard_Init();
    #endif
    // ------------------------------------------------------------------------
    // Initial GPIO for 8x8 dot matrix LED.                    
    // ------------------------------------------------------------------------
    #if BSP_17_2COLOR_DOTMATRIX_LED == 1
        BSP_2ColorDotMatrixLED_Init();
    #endif
    // ------------------------------------------------------------------------
    // Initial GPIO for show 4x7-SEG display.                                
    // ------------------------------------------------------------------------
    #if BSP_18_7SEGMENT_DISPLAY     == 1
        BSP_7Segment_Init();        
    #endif
    // ------------------------------------------------------------------------
    // Initial GPIO+Timer to display on LCD.                                
    // ------------------------------------------------------------------------
    #if BSP_19_LCD                  == 1
        BSP_LCM_Init();        
    #endif
    
    // ------------------------------------------------------------------------
    // Delay 1ms for stablie
    // ------------------------------------------------------------------------
    TH222x_Delay(1);
    
}

/**
 *******************************************************************************
 * @brief	    Display Flash access result.  
 * @details     Success controls R LED. Fail controls G LED.
 * @return      None
 * @note        Ne
 *******************************************************************************
 */ 
void TH222x_DisplayTestResult(void)
{
    uint8_t BSP_TestResultTmp;
    
    // ------------------------------------------------------------------------
    // with TH222x_CTR.TestFlashResult when BSP_13_SPIFLASH=1
    BSP_TestResultTmp = 0;
    
    #if BSP_13_SPIFLASH == 1
        BSP_TestResultTmp = BSP_TestResultTmp + TH222x_CTR.TestFlashResult;
    #endif
    #if BSP_14_EEPROM == 1
        BSP_TestResultTmp = BSP_TestResultTmp + TH222x_CTR.TestEEPROMResult;
    #endif
    
    // ------------------------------------------------------------------------
    // Control R/G/B LED (Turn off)
    TH222x_CTR.RLED_Count = TH222x_CTR.LED_CountMax;
    TH222x_CTR.GLED_Count = TH222x_CTR.LED_CountMax;
    TH222x_CTR.YLED_Count = TH222x_CTR.LED_CountMax;
    
    // ------------------------------------------------------------------------
    // TH222x_SUCCESS controls G LED, the other condition controls R LED.
    if(BSP_TestResultTmp != TH222x_SUCCESS)
    {
        TH222x_CTR.RLED_Lock  = 1;							// Turn on R-LED when fail
        TH222x_CTR.RLED_Count = 0;
    }
    else
    {
        TH222x_CTR.GLED_Lock  = 1;							// Turn on G-LED when success
        TH222x_CTR.GLED_Count = 0;
    }
    
}

/**
 *******************************************************************************
 * @brief		TH222x_main control
 * @details     1. Only has UpdateFlag event.
 * 				2. SW3 control step motor + Buzzer.
 * 				3. SW4 control LCD display (with SPI FLASH or EEPROM)
 * 				4. Rotary Encoder/VR1,VR2 controls the G/R/Y LEDs with conditions.
 * @param[in]  
 * @return      
 * @note       
 *******************************************************************************
 */ 
void TH222x_main(void)
{
    #if BSP_7_STEP_MOTOR == 1
        uint8_t TH222x_mainTmp;
    #endif
    
    
    // ------------------------------------------------------------------------
    // SysTick IRQ handles TH222x_CTR.UpdateFlag event.
    // ------------------------------------------------------------------------
    if(TH222x_CTR.UpdateFlag == 0)
    {
        return;
        
    }

    // ------------------------------------------------------------------------
    TH222x_CTR.UpdateFlag = 0;
    
    

    // ------------------------------------------------------------------------
    // TH223A SW3 Button Detect. (Control Step Motor + Buzzer)
    // Condition is SW3_TRGlag=1.
    // ------------------------------------------------------------------------
    if(TH222x_CTR.SW3_TRGlag == 1)
    {
    	// --------------------------------------------------------------------
        // SW3 Drama Time :
    	//	1. TH223x_SW3_PIN(PB10) = H : SW3_DramaTime--
    	//	2. TH223x_SW3_PIN(PB10) = L : SW3_DramaTime++
    	// --------------------------------------------------------------------
    	// SW3_DramaTime'z size is 8bit.
        if(TH223x_SW3_PIN == 1)             // PB10=1(High)
            TH222x_CTR.SW3_DramaTime--;
        else
            TH222x_CTR.SW3_DramaTime++;
        
        // --------------------------------------------------------------------
        // when SW3_DranaTime overflow or underflow
        if(TH222x_CTR.SW3_DramaTime == 0)
        {
            TH222x_CTR.SW3_TRGlag     = 0;
            TH222x_CTR.SW3_UpdateFlag = 0;
        }
        // --------------------------------------------------------------------
        // SW3_DramaTime > 7 (SW_DramaTimeMax)
        else if(TH222x_CTR.SW3_DramaTime > TH222x_CTR.SW_DramaTimeMax) 
        {
            TH222x_CTR.SW3_TRGlag     = 0;
            TH222x_CTR.SW3_UpdateFlag = 0;
    
            // ----------------------------------------------------------------
            // Buzzer ON.
            // ----------------------------------------------------------------
            #if BSP_10_BUZZER == 1
                TH222x_Buzzer_UpdateFlag = 1;               // Buzzer turn on event
                TH222x_Buzzer_ONFlag     = 1;               // Turn On Buzzer.
                TH222x_Buzzer_ONCount    = BSP_BUZZER_TIME; // Turn On Time counter 
            #endif
            // ----------------------------------------------------------------
            // Step Motor Status Control.
            // ----------------------------------------------------------------
            #if BSP_7_STEP_MOTOR == 1
                TH222x_mainTmp = TH222x_StepMotor_GetStatus();
                if((uint8_t)TH222x_mainTmp == STEPMOTOR_ANTICLOCKWISE)
                {
                    TH222x_StepMotor_SetStatus(STEPMOTOR_STOP);
                }
                else
                {
                    TH222x_StepMotor_SetStatus(((uint8_t)TH222x_mainTmp+1));
                }
            
            #endif
        }
        
    }
    // ------------------------------------------------------------------------
    // (SW3_TRGlag, SW3_UpdateFlag) 
    // SW3_TRGlag=0 then check if SW3_UpdateFlag=1
    // ------------------------------------------------------------------------
    else if(TH222x_CTR.SW3_UpdateFlag == 1)
    {
        TH222x_CTR.SW3_TRGlag    = 1;                       // Only here it will configure SW3_TRGlag to 1.
        TH222x_CTR.SW3_DramaTime = TH222x_CTR.SW_DramaTimeDefault;  // SW_DramaTimeDefault is 4. 
    }



    // ------------------------------------------------------------------------
    // TH223A SW4 Button Detect. (Control Display with SPIFLASH or EEPROM)
    // ------------------------------------------------------------------------
    if(TH222x_CTR.SW4_TRGlag == 1)
    {
        // ------------------------------------------------------------------------
    	// SW4 Drama Time
    	//	1. TH223x_SW4_PIN(PB11) = H : SW4_DramaTime--
    	//	2. TH223x_SW4_PIN(PB11) = L : SW4_DramaTime++
        // SW4_DramaTime'z size is 8bit.
        // ------------------------------------------------------------------------
    	if(TH223x_SW4_PIN == 1)                             // PB11=1(High)
            TH222x_CTR.SW4_DramaTime--;
        else
            TH222x_CTR.SW4_DramaTime++;
        
        // ------------------------------------------------------------------------
        // when SW4_DranaTime overflow or underflow
        if(TH222x_CTR.SW4_DramaTime == 0)
        {
            TH222x_CTR.SW4_TRGlag     = 0;
            TH222x_CTR.SW4_UpdateFlag = 0;
        }
        else if(TH222x_CTR.SW4_DramaTime > TH222x_CTR.SW_DramaTimeMax) 
        {
            TH222x_CTR.SW4_TRGlag     = 0;
            TH222x_CTR.SW4_UpdateFlag = 0;
            // ----------------------------------------------------------------
            // Display Flash access result.
            // 	1. Access Flash | EEPROM (Success) -> Turn on G-LED
            // 	2. Access Flash | EEPROM (Fail) -> Turn on R-LED
            // ----------------------------------------------------------------
            #if (BSP_13_SPIFLASH==1) || (BSP_14_EEPROM==1)
                TH222x_DisplayTestResult();
            #endif
        }
    }
    // ------------------------------------------------------------------------
    // SW4_TRGlag=0 then check if SW4_UpdateFlag=1
    // ------------------------------------------------------------------------
    else if(TH222x_CTR.SW4_UpdateFlag == 1)
    {
        TH222x_CTR.SW4_TRGlag    = 1;
        TH222x_CTR.SW4_DramaTime = TH222x_CTR.SW_DramaTimeDefault;
    }
    
    
    
    // ------------------------------------------------------------------------
    // TH223A G LED Control
    // ------------------------------------------------------------------------
    if(TH222x_CTR.GLED_Count < TH222x_CTR.LED_CountMax)
    {
        TH223x_LED_GREEN_ON();                              // Turn on LED
        
        TH222x_CTR.GLED_Count = TH222x_CTR.GLED_Count + 1;  // Turn on timer counter
    }
    else if(TH222x_CTR.GLED_Count == TH222x_CTR.LED_CountMax)// timeout ?
    {
        TH223x_LED_GREEN_OFF();                             // turn off LED
        TH222x_CTR.GLED_Lock = 0;                           // Nothing
        TH222x_CTR.GLED_Count = TH222x_CTR.GLED_Count + 1;  // Stop timer counter    
    }
    
    // ------------------------------------------------------------------------
    // TH223A R LED Control
    // ------------------------------------------------------------------------
    if(TH222x_CTR.RLED_Count < TH222x_CTR.LED_CountMax)
    {
        TH223x_LED_RED_ON();
        
        TH222x_CTR.RLED_Count = TH222x_CTR.RLED_Count + 1;
    }
    else if(TH222x_CTR.RLED_Count == TH222x_CTR.LED_CountMax)
    {
        TH223x_LED_RED_OFF();
        TH222x_CTR.RLED_Lock = 0;
        TH222x_CTR.RLED_Count = TH222x_CTR.RLED_Count + 1;
    }
    
    // ------------------------------------------------------------------------
    // TH223A Y LED Control
    // ------------------------------------------------------------------------
    if(TH222x_CTR.YLED_Count < TH222x_CTR.LED_CountMax)
    {
        TH223x_LED_YELLOW_ON();
        
        TH222x_CTR.YLED_Count = TH222x_CTR.YLED_Count + 1;
    }
    else if(TH222x_CTR.YLED_Count == TH222x_CTR.LED_CountMax)
    {
        TH223x_LED_YELLOW_OFF();
        TH222x_CTR.YLED_Lock = 0;
        TH222x_CTR.YLED_Count = TH222x_CTR.YLED_Count + 1;
    }
    


    // ------------------------------------------------------------------------
    // Buzzer Control
    // ------------------------------------------------------------------------
    #if BSP_10_BUZZER == 1
    
        if(TH222x_Buzzer_ONFlag == 1)                       // Turn on ?
        {
            if(TH222x_Buzzer_ONCount == 0)                  // timeout ?
            {
                TH222x_Buzzer_UpdateFlag = 1;               // Update Buzzer event
                TH222x_Buzzer_ONFlag     = 0;               // Turn Off Buzzer.
            }
            else                                            // (--) Buzzen turn on time counter. 
            {
                TH222x_Buzzer_ONCount = TH222x_Buzzer_ONCount - 1;
            }
            
        }
        
    #endif
}
 
/**
 *******************************************************************************
 * @brief       Get SystemTick counter value.
 * @return      None
 * @note        No
 *******************************************************************************
 */
static uint32_t TH222x_GetSysTickCount(void) 
{
    return(TH222x_CTR.SysTickCount);
} 
