


/**
 ******************************************************************************
 *
 * @file        TH222x_13_SPIFlash.c
 * @brief       This is to test TH222x SPI Flash C file.
                
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2023/02/07
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32_IWDT_DRV.h"
#include "TH222x_Global.h"
#include "BSP_13_SPIFlash.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static FlashCTR_TypeDef TH222x_Flash;

static uint8_t TH222x_SPI_ProgramFlashBuffer[10] = {0x05, 0x30, 0x06, 0x15, 0x5A, 0xA5, 0x11, 0x20, 0x09 , 0x12};
static uint8_t TH222x_SPI_ReadFlashBuffer[10];

/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/



/**
 *******************************************************************************
 * @brief       TH222x SPI Flash main function. 
 * @details     1. Disable IWDT.
 *              2. Initial SPI module.
 *              3. Check Flash ID. (ID=0x1728C2)
 *              4. Erase Flash + Read it to check. 
 *              5. Program Flash + Read it to check. 
 * @return      None
 *******************************************************************************
 */
uint8_t TH222x_SPIFlash(void)
{
    uint32_t TH222x_SPIFlashTmp;
    
    // ------------------------------------------------------------------------
    // Disable IWDT (for BSP_ARGB_CpltCallback)
    // ------------------------------------------------------------------------
    UnProtectModuleReg(IWDTprotect);
    IWDT_Cmd(DISABLE);
    ProtectModuleReg(IWDTprotect);
    
    // ------------------------------------------------------------------------
    // Initial SPI
    // ------------------------------------------------------------------------
    BSP_SPIFlash_Init();
    
    // ------------------------------------------------------------------------
    // Check Flash ID
    // ------------------------------------------------------------------------
    if(BSP_SPIFlash_ReadFlashRDID()!= BSP_13_SPIFLASH_RDID)
    {
        return(TH222x_FAILURE);
    }
    
    // ------------------------------------------------------------------------
    // Test Erase Flash
    // ------------------------------------------------------------------------
    BSP_SPIFlash_Erase(Flash_Erase_Sector, 1);                       // Erase sector 1 (Address = 0x1000 ~ 0x1FFF)
        
    // ------------------------------------------------------------------------
    // Read Erase area to check if empty?
    // ------------------------------------------------------------------------
    TH222x_Flash.Address      = 0x1040;                              // Define address of flash program. 
    TH222x_Flash.Buffer       = &TH222x_SPI_ReadFlashBuffer[0];      // Define storage address of receive data from flash.
    TH222x_Flash.Total_Length = 10;                                  // Define size of read data from flash.
    
    BSP_SPIFlash_Read(TH222x_Flash,Flash_Read);                      // SPI read flash
    
    // check erase zone data
    for(TH222x_SPIFlashTmp=0; TH222x_SPIFlashTmp<10; TH222x_SPIFlashTmp++)
    {
        // if the data is not equal to 0xFF, it means the erase failed.
        if(TH222x_SPI_ReadFlashBuffer[TH222x_SPIFlashTmp] != 0xFF)  
        {
            // Return failure if not empty.
            return(TH222x_FAILURE);
        }
    }
    
    // ------------------------------------------------------------------------
    // Test Program Flash 
    // ------------------------------------------------------------------------
    TH222x_Flash.Address      = 0x1040;                              // Define address of flash program. 
    TH222x_Flash.Buffer       = &TH222x_SPI_ProgramFlashBuffer[0];   // Define datas of flash program.
    TH222x_Flash.Total_Length = 10;                                  // Define size of flash program.
    
    BPS_SPIFlash_Program(TH222x_Flash, Flash_PageProgram);           // SPI program

    // ------------------------------------------------------------------------
    // Read the program area to check the data.
    // ------------------------------------------------------------------------
    TH222x_Flash.Address      = 0x1040;                              // Define address of flash program. 
    TH222x_Flash.Buffer       = &TH222x_SPI_ReadFlashBuffer[0];      // Define storage address of receive data from flash.
    TH222x_Flash.Total_Length = 10;                                  // Define size of read data from flash.
    
    BSP_SPIFlash_Read(TH222x_Flash, Flash_Read);                     // SPI read flash 
    
    // check data
    for(TH222x_SPIFlashTmp=0; TH222x_SPIFlashTmp<10; TH222x_SPIFlashTmp++)
    {
        // Compare read data with source data.
        if(TH222x_SPI_ReadFlashBuffer[TH222x_SPIFlashTmp] != TH222x_SPI_ProgramFlashBuffer[TH222x_SPIFlashTmp])
        {
            // Return failure if not equal.
            return(TH222x_FAILURE);
        }
    }
    
    // Pass state
    return(TH222x_SUCCESS);
}











