
/**
 ******************************************************************************
 *
 * @file        LCD.c
 * @brief       LCD C file.
 
 * @par         Project
 *              USB_LCD Development
 * @version     V1.01
 * @date        2022/09/23
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include <stdlib.h>
#include "LCD.h"
#include "Font.h"

/* Wizard menu ---------------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/
Display_HandleTypeDef LCD1st;

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/

#define EMB_CR1_MAD_SWAP_mask_w                     ((uint32_t)0x00000200)  /*!< Bit Mask of 32bit */
#define EMB_CR1_MAD_SWAP_mask_h0                    ((uint16_t)0x0200)      /*!< Bit Mask of 16bit */
#define EMB_CR1_MAD_SWAP_mask_b1                    ((uint8_t )0x02)        /*!< Bit Mask of 8bit */
#define EMB_CR1_MAD_SWAP_disable_w                  ((uint32_t)0x00000000)  /*!< Bit Value =(0):Disable of 32bit */
#define EMB_CR1_MAD_SWAP_enable_w                   ((uint32_t)0x00000200)  /*!< Bit Value =(1):Enable of 32bit */

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

/* Private function prototypes -----------------------------------------------*/
void delay_us(uint32_t DelayCount);
uint16_t DataReverse(uint16_t InData);
uint8_t DataReverse8(uint8_t InData);
void SPI_LCD_Reset(uint16_t Time);
uint16_t SPI_LCD_ReadData8(void);
uint8_t I80_LCD_ReadData8(void);
void LCD_NT35310_SPI_SelectRegister(uint8_t Register);
void LCD_NT35310_SPI_WriteData(uint16_t Data);
void LCD_NT35310_SPI_Write8BitsData(uint8_t Data);
uint16_t LCD_NT35310_SPI_ReadData(void);
uint8_t LCD_NT35310_SPI_Read8BitsData(void);
void LCD_NT35310_Initial(Display_HandleTypeDef *DispX);
void LCD_NT35510_SPI_SelectRegister(uint8_t Register);
void LCD_NT35510_SPI_WriteData(uint16_t Data);
void LCD_NT35510_SPI_Write8BitsData(uint8_t Data);
uint16_t LCD_NT35510_SPI_ReadData(void);
uint8_t LCD_NT35510_SPI_Read8BitsData(void);
void LCD_NT35510_SPI_WriteRegister(uint16_t Register, uint16_t Value);
void LCD_NT35510_Initial(Display_HandleTypeDef *DispX);
void LCD_ILI93xx_SPI_SelectRegister(uint8_t Register);
void LCD_ILI93xx_SPI_WriteData(uint16_t Data);
void LCD_ILI93xx_SPI_Write8BitsData(uint8_t Data);
uint16_t LCD_ILI93xx_SPI_ReadData(void);
uint8_t LCD_ILI93xx_SPI_Read8BitsData(void);
void LCD_ILI93xx_SPI_WriteRegister(uint16_t Register, uint16_t Value);
void LCD_ILI9320_Initial(Display_HandleTypeDef *DispX);
void LCD_ILI9325_Initial(Display_HandleTypeDef *DispX);
void LCD_ILI9328_Initial(Display_HandleTypeDef *DispX);
void LCD_ILI9331_Initial(Display_HandleTypeDef *DispX);
void LCD_ILI9341_SPI_SelectRegister(uint8_t Register);
void LCD_ILI9341_SPI_WriteData(uint16_t Data);
void LCD_ILI9341_SPI_Write8BitsData(uint8_t Data);
uint16_t LCD_ILI9341_SPI_ReadData(void);
uint8_t LCD_ILI9341_SPI_Read8BitsData(void);
void LCD_ILI9341_Initial(Display_HandleTypeDef *DispX);
uint16_t LCD_BGR2RGB(uint16_t c);
void LCD_ButtonDone(Display_HandleTypeDef *DispX, uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t Color);
void LCD_Color_Fill_DMA(Display_HandleTypeDef *DispX, uint16_t sx, uint16_t sy, uint16_t width, uint16_t height, uint16_t *color);
void LCD_Color_FillX(Display_HandleTypeDef *DispX, uint16_t sx, uint16_t sy, uint16_t ex, uint16_t ey, uint16_t *color);
uint32_t LCD_Pow(uint8_t m,uint8_t n);
void LCD_ColorBar_1(Display_HandleTypeDef *DispX);
void LCD_ColorBar_2(Display_HandleTypeDef *DispX);
void LCD_GrayBar_1(Display_HandleTypeDef *DispX);

//////////////////////////////////////////////////////////////////////////////////
//本程序只供學習使用，未經作者許可，不得用於其它任何用途
//ALIENTEK戰艦STM32開發板
//2.4寸/2.8寸/3.5寸/4.3寸 TFT液晶驅動
//支持驅動IC型號包括:ILI9341/ILI9325/RM68042/RM68021/ILI9320/ILI9328/LGDP4531/LGDP4535/
//                  SPFD5408/SSD1289/1505/B505/C505/NT35310/NT35510等
//正點原子@ALIENTEK
//技術論壇:www.openedv.com
//修改日期:2014/7/10
//版本：V2.6
//版權所有，盜版必究。
//Copyright(C) 廣州市星翼電子科技有限公司 2009-2019
//All rights reserved
//********************************************************************************
//V1.2修改說明
//支持了SPFD5408的驅動,另外把液晶ID直接打印成HEX格式.方便查看LCD驅動IC.
//V1.3
//加入了快速IO的支持
//修改了背光控制的極性（適用於V1.8及以後的開發板版本）
//對於1.8版本之前(不包括1.8)的液晶模塊,請修改LCD_Init函數的LCD_LED=1;為LCD_LED=1;
//V1.4
//修改了LCD_ShowChar函數，使用畫點功能畫字符。
//加入了橫豎屏顯示的支持
//V1.5 20110730
//1,修改了B505液晶讀顏色有誤的bug.
//2,修改了快速IO及橫豎屏的設置方式.
//V1.6 20111116
//1,加入對LGDP4535液晶的驅動支持
//V1.7 20120713
//1,增加LCD_RD_DATA函數
//2,增加對ILI9341的支持
//3,增加ILI9325的獨立驅動代碼
//4,增加LCD_Scan_Dir函數(慎重使用)
//6,另外修改了部分原來的函數,以適應9341的操作
//V1.8 20120905
//1,加入LCD重要參數設置結構體lcddev
//2,加入LCD_Display_Dir函數,支持在線橫豎屏切換
//V1.9 20120911
//1,新增RM68042驅動（ID:6804），但是6804不支持橫屏顯示！！原因：改變掃描方式，
//導致6804坐標設置失效，試過很多方法都不行，暫時無解。
//V2.0 20120924
//在不硬件覆位的情況下,ILI9341的ID讀取會被誤讀成9300,修改LCD_Init,將無法識別
//的情況（讀到ID為9300/非法ID）,強制指定驅動IC為ILI9341，執行9341的初始化。
//V2.1 20120930
//修正ILI9325讀顏色的bug。
//V2.2 20121007
//修正LCD_Scan_Dir的bug。
//V2.3 20130120
//新增6804支持橫屏顯示
//V2.4 20131120
//1,新增NT35310（ID:5310）驅動器的支持
//2,新增LCD_Set_Window函數,用於設置窗口,對快速填充,比較有用,但是該函數在橫屏時,不支持6804.
//V2.5 20140211
//1,新增NT35510（ID:5510）驅動器的支持
//V2.6 20140710
//1,修正LCD_Scan_Dir橫屏時設置的掃描方式顯示不全的bug.
//////////////////////////////////////////////////////////////////////////////////


//////LCD的畫筆顏色和背景色
////uint16_t POINT_COLOR=0xFFFF;    //畫筆顏色
////uint16_t BACK_COLOR=0x0000;     //背景色 

//////管理LCD重要參數
//////默認為豎屏
////_lcd_dev lcddev;

/**
 *******************************************************************************
 * @brief       This function provides accurate delay (in microseconds).
 * @param       uDelay specifies the delay time length, in microseconds.
 *      
 * @return      None
 *******************************************************************************
 */
void delay_us(uint32_t DelayCount)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(DelayCount));
}

/**
 *******************************************************************************
 * @brief       Data Reverse 16-Bits
 * @param       InData Input Data.
 *      
 * @return      uint16_t Reverse Data 
 *******************************************************************************
 */
uint16_t DataReverse(uint16_t InData)
{
    uint16_t t, t2;

    t2 = InData;
    t = 0;
    t |= (t2 & 0x0001); // Bit0

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit1

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit2

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit3

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit4

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit5

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit6

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit7

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit8

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit9

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit10

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit11

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit12

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit13

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit14

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x0001); // Bit15

    return t;
}

/**
 *******************************************************************************
 * @brief       Data Reverse 16-Bits
 * @param       InData Input Data.
 *      
 * @return      uint16_t Reverse Data 
 *******************************************************************************
 */
uint8_t DataReverse8(uint8_t InData)
{
    uint8_t t, t2;

    t2 = InData;
    t = 0;
    t |= (t2 & 0x01); // Bit0

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x01); // Bit1

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x01); // Bit2

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x01); // Bit3

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x01); // Bit4

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x01); // Bit5

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x01); // Bit6

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x01); // Bit7

    t <<= 1;
    t2 >>= 1;
    t |= (t2 & 0x01); // Bit8

    return t;
}

void SPI_LCD_Reset(uint16_t Time)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Time));
    
    __SPI_LCD_RESET_ENABLE();
    MID_Delay(1);
    __SPI_LCD_RESET_DISABLE();
    MID_Delay(50);
}

void SPI_LCD_BackLight(uint16_t Duty)
{
    if(Duty == 0x0000)
        __SPI_LCD_BACKLIGHT_OFF();

    if(Duty == 0xFFFF)
        __SPI_LCD_BACKLIGHT_ON();
}

void SPI_LCD_DataReverse(FunctionalState State)
{
    if(State == DISABLE)
        __NOP();
    else
        __NOP();
}

void SPI_LCD_SelectRegister(uint16_t Register)
{
    __LCD_RS_CLR();
    //SPI_WriteByte(SPI1, regval&0x00FF);
    __LCD_CS_CLR();  //LCD_CS=0
    URT0->TDAT.B[0] = (uint8_t)Register;
    while((URT0->STA.B[0] & URT_STA_TCF_mask_b0) == 0);
    URT0->STA.W = URT_STA_TCF_mask_w;
    __LCD_CS_SET();  //LCD_CS=1
}

void SPI_LCD_WriteData(uint16_t Data)
{
    __LCD_RS_SET();

////// HalfWord
////    //SPI_WriteByte(SPI1,regval&0x00FF);
////    LCD_CS_LOW();  //LCD_CS=0
////    URT0->TDAT.H[0] = data;
////    while((URT0->STA.B[0] & URT_STA_TCF_mask_b0) == 0);
////    URT_ClearITFlag(URTn,URT_IT_TC);
////    LCD_CS_HIGH();  //LCD_CS=1

// Byte
    //SPI_WriteByte(SPI1,regval&0x00FF);
    __LCD_CS_CLR();  //LCD_CS=0
    URT0->TDAT.B[0] = (uint8_t)(Data >> 8);
    while((URT0->STA.B[0] & URT_STA_TCF_mask_b0) == 0);
    URT0->STA.W = URT_STA_TCF_mask_w;

    URT0->TDAT.B[0] = (uint8_t)Data;
    while((URT0->STA.B[0] & URT_STA_TCF_mask_b0) == 0);
    URT0->STA.W = URT_STA_TCF_mask_w;
    __LCD_CS_SET();  //LCD_CS=1
}

void SPI_LCD_WriteData8(uint8_t Data)   //
{
    __LCD_RS_SET();
    //SPI_WriteByte(SPI1,regval&0x00FF);
    __LCD_CS_CLR();  //LCD_CS=0
    URT0->TDAT.B[0] = Data;
    while((URT0->STA.B[0] & URT_STA_TCF_mask_b0) == 0);
    URT0->STA.W = URT_STA_TCF_mask_w;
    __LCD_CS_SET();  //LCD_CS=1
}

uint16_t SPI_LCD_ReadData(void)
{
    __LCD_RS_SET();

////// HalfWord
////    //SPI_WriteByte(SPI1,regval&0x00FF);
////    LCD_CS_LOW();  //LCD_CS=0
////    URT0->TDAT.H[0] = data;
////    while((URT0->STA.B[0] & URT_STA_TCF_mask_b0) == 0);
////    URT_ClearITFlag(URTn,URT_IT_TC);
////    LCD_CS_HIGH();  //LCD_CS=1

// Byte
    //SPI_WriteByte(SPI1,regval&0x00FF);
    __LCD_CS_CLR();  //LCD_CS=0
    URT0->TDAT.B[0] = (uint8_t)(0xFF >> 8);
    while((URT0->STA.B[0] & URT_STA_TCF_mask_b0) == 0);
    URT0->STA.W = URT_STA_TCF_mask_w;
    //LCD_CS_HIGH();  //LCD_CS=1

    //SPI_WriteByte(SPI1,regval&0x00FF);
    URT0->TDAT.B[0] = (uint8_t)0xFF;
    while((URT0->STA.B[0] & URT_STA_TCF_mask_b0) == 0);
    URT0->STA.W = URT_STA_TCF_mask_w;
    __LCD_CS_SET();  //LCD_CS=1

    return 0;
}

uint16_t SPI_LCD_ReadData8(void)
{
    __LCD_RS_SET();

  // Byte
    //SPI_WriteByte(SPI1,regval&0x00FF);
    __LCD_CS_CLR();  //LCD_CS=0
    URT0->TDAT.B[0] = (uint8_t)(0xFF >> 8);
    while((URT0->STA.B[0] & URT_STA_TCF_mask_b0) == 0);
    URT0->STA.W = URT_STA_TCF_mask_w;
    __LCD_CS_SET();  //LCD_CS=1

    return 0;
}

void SPI_LCD_WriteRegister(uint16_t Register, uint16_t Value)
{
    __LCD_RS_SET();
    //SPI_WriteByte(SPI1,regval&0x00FF);
    __LCD_CS_CLR();  //LCD_CS=0
    URT0->TDAT.B[0] = (uint8_t)Register;
    while((URT0->STA.B[0] & URT_STA_TCF_mask_b0) == 0);
    URT0->STA.W = URT_STA_TCF_mask_w;
    __LCD_CS_SET();  //LCD_CS=1

    __LCD_RS_SET();
        //SPI_WriteByte(SPI1,regval&0x00FF);
    __LCD_CS_CLR();  //LCD_CS=0
    URT0->TDAT.B[0] = (uint8_t)Value;
    while((URT0->STA.B[0] & URT_STA_TCF_mask_b0) == 0);
    URT0->STA.W = URT_STA_TCF_mask_w;
    __LCD_CS_SET();  //LCD_CS=1
}

/**
 *******************************************************************************
 * @brief       LCD Reset
 * @return      uint16_t Data Value
 *******************************************************************************
 */
void I80_LCD_Reset(uint16_t Duty)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Duty));
    
    __I80_LCD_RESET_ENABLE();
    MID_Delay(1);
    __I80_LCD_RESET_DISABLE();
    MID_Delay(50);
}

/**
 *******************************************************************************
 * @brief       LCD Back Light ON
 * @return      uint16_t Data Value
 *******************************************************************************
 */
void I80_LCD_BackLight(uint16_t Duty)
{
    if(Duty == 0x0000)
        __I80_LCD_BACKLIGHT_OFF();

    if(Duty == 0xFFFF)
        __I80_LCD_BACKLIGHT_ON();
}

void I80_LCD_DataReverse(FunctionalState State)
{
    if(State == DISABLE)
        EMB->CR1.W &= ~EMB_CR1_MAD_SWAP_enable_w;
    else
        EMB->CR1.W |= EMB_CR1_MAD_SWAP_enable_w;
}

/**
 *******************************************************************************
 * @brief       Select LCD Module Control Register 寫寄存器函數
 * @param       Register LCD Module Control Register
 * @return      None 
 *******************************************************************************
 */
void I80_LCD_SelectRegister(uint16_t Register)
{
    //__IO uint16_t * lAddr = (uint16_t *)0x60000000;
    __LCD_CS_CLR();
    __LCD_RS_CLR();//寫地址
    __DATAOUT(Register);
    //* lAddr = Register;
    __LCD_WR_CLR();
    __LCD_WR_SET();
    __LCD_CS_SET();
}

/**
 *******************************************************************************
 * @brief       Write Data to LCD Module Register 寫數據至寄存器
 * @param       LCD_Reg LCD Module Control Register 寄存器編號
 * @param       LCD_RegValue Write Value            要寫入的值
 * @return      None 
 *******************************************************************************
 */
void I80_LCD_WriteRegister(uint16_t Register, uint16_t Value)
{
    //__IO uint16_t * lAddr = (uint16_t *)0x60000000;
    //DispX->SelectReg(LCD_Reg);
    __LCD_CS_CLR();
    __LCD_RS_CLR();//寫地址
    __DATAOUT(Register);
    //* lAddr = Register;
    __LCD_WR_CLR();
    __LCD_WR_SET();
    //__LCD_CS_SET();

    //LCD_WR_iData(LCD_RegValue);
    //__LCD_CS_CLR();
    __LCD_RS_SET();
    __DATAOUT(Value);
    //* lAddr = Value;
    __LCD_WR_CLR();
    __LCD_WR_SET();
    __LCD_CS_SET();
}

/**
 *******************************************************************************
 * @brief       Read Data from LCD Module Register 讀寄存器
 * @param       LCD_Reg LCD Module Control Register 寄存器編號
 * @param       LCD_RegValue Write Data
 * @return      uint16_t 讀到的值
 *******************************************************************************
 */
uint16_t I80_LCD_ReadRegister(uint16_t Register)
{
    //__IO uint16_t * lAddr = (uint16_t *)0x60000000;
    uint16_t lTemp;
    //DispX->SelectReg(LCD_Reg);  //寫入要讀的寄存器號
    __LCD_CS_CLR();

    __LCD_RS_CLR();//寫地址
    __DATAOUT(Register);
    //* lAddr = Register;
    __LCD_WR_CLR();
    __LCD_WR_SET();

    __LCD_RS_SET();
    //讀取數據(讀寄存器時,並不需要讀2次)
    __LCD_RD_CLR();
    //if(lcddev.id == 0X8989) delay_us(2);//FOR 8989,延時2us
    lTemp = __DATAIN();
    //lTemp = * lAddr
    __LCD_RD_SET();

    __LCD_CS_SET();

  #if defined(MG32_1ST)
    if((EMB->CR1.B[1] & EMB_CR1_MAD_SWAP_mask_b1) == 0)
        return lTemp;
    else
        return DataReverse(lTemp);
  #else
        return lTemp;
  #endif
}

/**
 *******************************************************************************
 * @brief       Write LCD Module Data 寫數據函數
 * @param       Data Write Data 寄存器值
 * @return      None 
 *******************************************************************************
 */
void I80_LCD_WriteData(uint16_t Data)
{
    //__IO uint16_t * lAddr = (uint16_t *)0x60000000;
    __LCD_CS_CLR();
    __LCD_RS_SET();
    __DATAOUT(Data);
    //*lAddr = Data;
    __LCD_WR_CLR();
    __LCD_WR_SET();
    __LCD_CS_SET();
}

/**
 *******************************************************************************
 * @brief       Write LCD Module Data 寫數據函數
 * @param       Data Write Data 寄存器值
 * @return      None 
 *******************************************************************************
 */
void I80_LCD_WriteData8(uint8_t Data)
{
    //__IO uint8_t * lAddr = (uint8_t *)0x60000000;
    __LCD_CS_CLR();
    __LCD_RS_SET();
    __DATAOUT(Data);
    //* lAddr = Data;
    __LCD_WR_CLR();
    __LCD_WR_SET();
    __LCD_CS_SET();
}

/**
 *******************************************************************************
 * @brief       Read LCD Module Data 讀LCD數據
 * @return      uint16_t Data Value
 *******************************************************************************
 */
uint16_t I80_LCD_ReadData(void)
{
    //__IO uint16_t * lAddr = (uint16_t *)0x60000000;
    __IO uint16_t t;
    __LCD_CS_CLR();
    __LCD_RS_SET();
                //讀取數據(讀寄存器時,並不需要讀2次)
    __LCD_RD_CLR();
    //if(lcddev.id == 0X8989)delay_us(2);//FOR 8989,延時2us
    t = __DATAIN();
    //t = *lAddr;
    __LCD_RD_SET();
    __LCD_CS_SET();

  #if defined(MG32_1ST)
    if((EMB->CR1.B[1] & EMB_CR1_MAD_SWAP_mask_b1) == 0)
        return t;
    else
        return DataReverse(t);
  #else
    return t;
  #endif
}

/**
 *******************************************************************************
 * @brief       Read LCD Module Data 讀LCD數據
 * @return      uint16_t Data Value
 *******************************************************************************
 */
uint8_t I80_LCD_ReadData8(void)
{
    //__IO uint8_t * lAddr = (uint8_t *)0x60000000;
    uint8_t t;
    __LCD_CS_CLR();
    __LCD_RS_SET();
                //讀取數據(讀寄存器時,並不需要讀2次)
    __LCD_RD_CLR();
    //if(lcddev.id == 0X8989)delay_us(2);//FOR 8989,延時2us
    t = (uint8_t)__DATAIN();
    //t = *lAddr;
    __LCD_RD_SET();
    __LCD_CS_SET();

  #if defined(MG32_1ST)
    if((EMB->CR1.B[1] & EMB_CR1_MAD_SWAP_mask_b1) == 0)
        return t;
    else
        return DataReverse(t);
  #else
    return t;
  #endif
}

/////**
//// *******************************************************************************
//// * @brief       Write Data to LCD Module GRAM
//// * @param       RGB_Code Color Data Value 顏色值
//// * @return      None
//// *******************************************************************************
//// */
////void LCD_WriteData_Prepare(void)
////{
////    DispX->SelectReg(lcddev.wramcmd);
////}

/////**
//// *******************************************************************************
//// * @brief       Select LCD Module Control Register to GRAM
//// * @return      None
//// *******************************************************************************
//// */
////void LCD_WriteData(uint16_t RGB_Code)
////{
////    LCD_WR_iData(RGB_Code);//寫十六位GRAM
////}

/**
 ******************************************************************************
 * @brief Read Register
 * @return None
 ******************************************************************************
 */

/////**
//// *******************************************************************************
//// * @brief       1505 Chip Inital.
//// * @return      None
//// *******************************************************************************
//// */
////void LCD_ChipInit_1505(Display_HandleTypeDef *DispX)
////{
////    // second release on 3/5  ,luminance is acceptable,water wave appear during camera preview
////    DispX->WriteRegister(0x07,0x0000);
////    MID_Delay(50); 
////    DispX->WriteRegister(0x12,0x011C);//0x011A   why need to set several times?
////    DispX->WriteRegister(0xA4,0x0001);//NVM
////    DispX->WriteRegister(0x08,0x000F);
////    DispX->WriteRegister(0x0A,0x0008);
////    DispX->WriteRegister(0x0D,0x0008);
////    //伽馬校正
////    DispX->WriteRegister(0x30,0x0707);
////    DispX->WriteRegister(0x31,0x0007); //0x0707
////    DispX->WriteRegister(0x32,0x0603); 
////    DispX->WriteRegister(0x33,0x0700); 
////    DispX->WriteRegister(0x34,0x0202); 
////    DispX->WriteRegister(0x35,0x0002); //?0x0606
////    DispX->WriteRegister(0x36,0x1F0F);
////    DispX->WriteRegister(0x37,0x0707); //0x0f0f  0x0105
////    DispX->WriteRegister(0x38,0x0000); 
////    DispX->WriteRegister(0x39,0x0000); 
////    DispX->WriteRegister(0x3A,0x0707); 
////    DispX->WriteRegister(0x3B,0x0000); //0x0303
////    DispX->WriteRegister(0x3C,0x0007); //?0x0707
////    DispX->WriteRegister(0x3D,0x0000); //0x1313//0x1f08
////    MID_Delay(50);
////    DispX->WriteRegister(0x07,0x0001);
////    DispX->WriteRegister(0x17,0x0001);      //開啟電源
////    MID_Delay(50);
////    //電源配置
////    DispX->WriteRegister(0x10,0x17A0); 
////    DispX->WriteRegister(0x11,0x0217);//reference voltage VC[2:0]   Vciout = 1.00*Vcivl
////    DispX->WriteRegister(0x12,0x011E);//0x011c  //Vreg1out = Vcilvl*1.80   is it the same as Vgama1out ?
////    DispX->WriteRegister(0x13,0x0F00);//VDV[4:0]-->VCOM Amplitude VcomL = VcomH - Vcom Ampl
////    DispX->WriteRegister(0x2A,0x0000);
////    DispX->WriteRegister(0x29,0x000A);//0x0001F  Vcomh = VCM1[4:0]*Vreg1out    gate source voltage??
////    DispX->WriteRegister(0x12,0x013E);// 0x013C  power supply on
////    //Coordinates Control//
////    DispX->WriteRegister(0x50,0x0000);//0x0e00
////    DispX->WriteRegister(0x51,0x00EF); 
////    DispX->WriteRegister(0x52,0x0000); 
////    DispX->WriteRegister(0x53,0x013F); 
////    //Pannel Image Control//
////    DispX->WriteRegister(0x60,0x2700); 
////    DispX->WriteRegister(0x61,0x0001); 
////    DispX->WriteRegister(0x6A,0x0000); 
////    DispX->WriteRegister(0x80,0x0000); 
////    //Partial Image Control//
////    DispX->WriteRegister(0x81,0x0000); 
////    DispX->WriteRegister(0x82,0x0000); 
////    DispX->WriteRegister(0x83,0x0000); 
////    DispX->WriteRegister(0x84,0x0000); 
////    DispX->WriteRegister(0x85,0x0000); 
////    //Panel Interface Control//
////    DispX->WriteRegister(0x90,0x0013);//0x0010 frenqucy
////    DispX->WriteRegister(0x92,0x0300); 
////    DispX->WriteRegister(0x93,0x0005); 
////    DispX->WriteRegister(0x95,0x0000); 
////    DispX->WriteRegister(0x97,0x0000); 
////    DispX->WriteRegister(0x98,0x0000); 
////
////    DispX->WriteRegister(0x01,0x0100); 
////    DispX->WriteRegister(0x02,0x0700); 
////    DispX->WriteRegister(0x03,0x1038);//掃描方向 上->下  左->右 
////    DispX->WriteRegister(0x04,0x0000); 
////    DispX->WriteRegister(0x0C,0x0000); 
////    DispX->WriteRegister(0x0F,0x0000); 
////    DispX->WriteRegister(0x20,0x0000); 
////    DispX->WriteRegister(0x21,0x0000); 
////    DispX->WriteRegister(0x07,0x0021); 
////    MID_Delay(20);
////    DispX->WriteRegister(0x07,0x0061); 
////    MID_Delay(20);
////    DispX->WriteRegister(0x07,0x0173); 
////    MID_Delay(20);
////}

/////**
//// *******************************************************************************
//// * @brief       5408 Chip Inital.
//// * @return      None
//// *******************************************************************************
//// */
////void LCD_ChipInit_5408(Display_HandleTypeDef *DispX)
////{
////    DispX->WriteRegister(0x01,0x0100);
////    DispX->WriteRegister(0x02,0x0700);//LCD Driving Waveform Contral 
////    DispX->WriteRegister(0x03,0x1030);//Entry Mode設置
////    //指針從左至右自上而下的自動增模式
////    //Normal Mode(Window Mode disable)
////    //RGB格式
////    //16位數據2次傳輸的8總線設置
////    DispX->WriteRegister(0x04,0x0000); //Scalling Control register
////    DispX->WriteRegister(0x08,0x0207); //Display Control 2
////    DispX->WriteRegister(0x09,0x0000); //Display Control 3
////    DispX->WriteRegister(0x0A,0x0000); //Frame Cycle Control
////    DispX->WriteRegister(0x0C,0x0000); //External Display Interface Control 1 
////    DispX->WriteRegister(0x0D,0x0000); //Frame Maker Position
////    DispX->WriteRegister(0x0F,0x0000); //External Display Interface Control 2 
////    MID_Delay(20);
////    //TFT 液晶彩色圖像顯示方法14
////    DispX->WriteRegister(0x10,0x16B0); //0x14B0 //Power Control 1
////    DispX->WriteRegister(0x11,0x0001); //0x0007 //Power Control 2
////    DispX->WriteRegister(0x17,0x0001); //0x0000 //Power Control 3
////    DispX->WriteRegister(0x12,0x0138); //0x013B //Power Control 4
////    DispX->WriteRegister(0x13,0x0800); //0x0800 //Power Control 5
////    DispX->WriteRegister(0x29,0x0009); //NVM read data 2
////    DispX->WriteRegister(0x2a,0x0009); //NVM read data 3
////    DispX->WriteRegister(0xa4,0x0000);
////    DispX->WriteRegister(0x50,0x0000); //設置操作窗口的X軸開始列
////    DispX->WriteRegister(0x51,0x00EF); //設置操作窗口的X軸結束列
////    DispX->WriteRegister(0x52,0x0000); //設置操作窗口的Y軸開始行
////    DispX->WriteRegister(0x53,0x013F); //設置操作窗口的Y軸結束行
////    DispX->WriteRegister(0x60,0x2700); //Driver Output Control
////    //設置屏幕的點數以及掃描的起始行
////    DispX->WriteRegister(0x61,0x0001); //Driver Output Control
////    DispX->WriteRegister(0x6A,0x0000); //Vertical Scroll Control
////    DispX->WriteRegister(0x80,0x0000); //Display Position - Partial Display 1
////    DispX->WriteRegister(0x81,0x0000); //RAM Address Start - Partial Display 1
////    DispX->WriteRegister(0x82,0x0000); //RAM address End - Partial Display 1
////    DispX->WriteRegister(0x83,0x0000); //Display Position - Partial Display 2
////    DispX->WriteRegister(0x84,0x0000); //RAM Address Start - Partial Display 2
////    DispX->WriteRegister(0x85,0x0000); //RAM address End - Partail Display2
////    DispX->WriteRegister(0x90,0x0013); //Frame Cycle Control
////    DispX->WriteRegister(0x92,0x0000);  //Panel Interface Control 2
////    DispX->WriteRegister(0x93,0x0003); //Panel Interface control 3
////    DispX->WriteRegister(0x95,0x0110);  //Frame Cycle Control
////    DispX->WriteRegister(0x07,0x0173);  
////    MID_Delay(50);
////}

/////**
//// *******************************************************************************
//// * @brief       SC6804A Chip Inital.
//// * @return      None
//// *******************************************************************************
//// */
////void LCD_ChipInit_6804(Display_HandleTypeDef *DispX)
////{
////    DispX->SelectReg(0X11);
////    MID_Delay(20);
////    DispX->SelectReg(0XD0);//VCI1  VCL  VGH  VGL DDVDH VREG1OUT power amplitude setting
////    DispX->WriteData(0X07); 
////    DispX->WriteData(0X42); 
////    DispX->WriteData(0X1D); 
////    DispX->SelectReg(0XD1);//VCOMH VCOM_AC amplitude setting
////    DispX->WriteData(0X00);
////    DispX->WriteData(0X1a);
////    DispX->WriteData(0X09); 
////    DispX->SelectReg(0XD2);//Operational Amplifier Circuit Constant Current Adjust , charge pump frequency setting
////    DispX->WriteData(0X01);
////    DispX->WriteData(0X22);
////    DispX->SelectReg(0XC0);//REV SM GS 
////    DispX->WriteData(0X10);
////    DispX->WriteData(0X3B);
////    DispX->WriteData(0X00);
////    DispX->WriteData(0X02);
////    DispX->WriteData(0X11);
////    DispX->SelectReg(0XC5);// Frame rate setting = 72HZ  when setting 0x03
////    DispX->WriteData(0X03);
////    DispX->SelectReg(0XC8);//Gamma setting
////    DispX->WriteData(0X00);
////    DispX->WriteData(0X25);
////    DispX->WriteData(0X21);
////    DispX->WriteData(0X05);
////    DispX->WriteData(0X00);
////    DispX->WriteData(0X0a);
////    DispX->WriteData(0X65);
////    DispX->WriteData(0X25);
////    DispX->WriteData(0X77);
////    DispX->WriteData(0X50);
////    DispX->WriteData(0X0f);
////    DispX->WriteData(0X00);
////    DispX->SelectReg(0XF8);
////    DispX->WriteData(0X01);
////    DispX->SelectReg(0XFE);
////    DispX->WriteData(0X00);
////    DispX->WriteData(0X02);
////    DispX->SelectReg(0X20);//Exit invert mode
////    DispX->SelectReg(0X36);
////    DispX->WriteData(0X08);//原來是a
////    DispX->SelectReg(0X3A);
////    DispX->WriteData(0X55);//16位模式
////    DispX->SelectReg(0X2B);
////    DispX->WriteData(0X00);
////    DispX->WriteData(0X00);
////    DispX->WriteData(0X01);
////    DispX->WriteData(0X3F);
////    DispX->SelectReg(0X2A);
////    DispX->WriteData(0X00);
////    DispX->WriteData(0X00);
////    DispX->WriteData(0X01);
////    DispX->WriteData(0XDF);
////    MID_Delay(120);
////    DispX->SelectReg(0X29);
////}


/////**
//// *******************************************************************************
//// * @brief       B505 Chip Inital.
//// * @return      None
//// *******************************************************************************
//// */
////void LCD_ChipInit_B505(Display_HandleTypeDef *DispX)
////{
////    DispX->WriteRegister(0x0000,0x0000);
////    DispX->WriteRegister(0x0000,0x0000);
////    DispX->WriteRegister(0x0000,0x0000);
////    DispX->WriteRegister(0x0000,0x0000);

////    DispX->WriteRegister(0x00a4,0x0001);
////    MID_Delay(20);
////    DispX->WriteRegister(0x0060,0x2700);
////    DispX->WriteRegister(0x0008,0x0202);
////
////    DispX->WriteRegister(0x0030,0x0214);
////    DispX->WriteRegister(0x0031,0x3715);
////    DispX->WriteRegister(0x0032,0x0604);
////    DispX->WriteRegister(0x0033,0x0e16);
////    DispX->WriteRegister(0x0034,0x2211);
////    DispX->WriteRegister(0x0035,0x1500);
////    DispX->WriteRegister(0x0036,0x8507);
////    DispX->WriteRegister(0x0037,0x1407);
////    DispX->WriteRegister(0x0038,0x1403);
////    DispX->WriteRegister(0x0039,0x0020);
////
////    DispX->WriteRegister(0x0090,0x001a);
////    DispX->WriteRegister(0x0010,0x0000);
////    DispX->WriteRegister(0x0011,0x0007);
////    DispX->WriteRegister(0x0012,0x0000);
////    DispX->WriteRegister(0x0013,0x0000);
////    MID_Delay(20);
////    
////    DispX->WriteRegister(0x0010,0x0730);
////    DispX->WriteRegister(0x0011,0x0137);
////    MID_Delay(20);
////
////    DispX->WriteRegister(0x0012,0x01b8);
////    MID_Delay(20);
////    
////    DispX->WriteRegister(0x0013,0x0f00);
////    DispX->WriteRegister(0x002a,0x0080);
////    DispX->WriteRegister(0x0029,0x0048);
////    MID_Delay(20);
////
////    DispX->WriteRegister(0x0001,0x0100);
////    DispX->WriteRegister(0x0002,0x0700);
////    DispX->WriteRegister(0x0003,0x1038);//掃描方向 上->下  左->右 
////    DispX->WriteRegister(0x0008,0x0202);
////    DispX->WriteRegister(0x000a,0x0000);
////    DispX->WriteRegister(0x000c,0x0000);
////    DispX->WriteRegister(0x000d,0x0000);
////    DispX->WriteRegister(0x000e,0x0030);
////    DispX->WriteRegister(0x0050,0x0000);
////    DispX->WriteRegister(0x0051,0x00ef);
////    DispX->WriteRegister(0x0052,0x0000);
////    DispX->WriteRegister(0x0053,0x013f);
////    DispX->WriteRegister(0x0060,0x2700);
////    DispX->WriteRegister(0x0061,0x0001);
////    DispX->WriteRegister(0x006a,0x0000);
////    //DispX->WriteRegister(0x0080,0x0000);
////    //DispX->WriteRegister(0x0081,0x0000);
////    DispX->WriteRegister(0x0090,0X0011);
////    DispX->WriteRegister(0x0092,0x0600);
////    DispX->WriteRegister(0x0093,0x0402);
////    DispX->WriteRegister(0x0094,0x0002);
////    MID_Delay(20);
////
////    DispX->WriteRegister(0x0007,0x0001);
////    MID_Delay(20);
////    DispX->WriteRegister(0x0007,0x0061);
////    DispX->WriteRegister(0x0007,0x0173);
////
////    DispX->WriteRegister(0x0020,0x0000);
////    DispX->WriteRegister(0x0021,0x0000);
////    DispX->WriteRegister(0x00,0x22);
////}

/////**
//// *******************************************************************************
//// * @brief       C505 Chip Inital.
//// * @return      None
//// *******************************************************************************
//// */
////void LCD_ChipInit_C505(Display_HandleTypeDef *DispX)
////{
////    DispX->WriteRegister(0x0000,0x0000);
////    DispX->WriteRegister(0x0000,0x0000);
////    MID_Delay(20);
////    DispX->WriteRegister(0x0000,0x0000);
////    DispX->WriteRegister(0x0000,0x0000);
////    DispX->WriteRegister(0x0000,0x0000);
////    DispX->WriteRegister(0x0000,0x0000);
////        DispX->WriteRegister(0x00a4,0x0001);
////    MID_Delay(20);
////    DispX->WriteRegister(0x0060,0x2700);
////    DispX->WriteRegister(0x0008,0x0806);
////    
////    DispX->WriteRegister(0x0030,0x0703);//gamma setting
////    DispX->WriteRegister(0x0031,0x0001);
////    DispX->WriteRegister(0x0032,0x0004);
////    DispX->WriteRegister(0x0033,0x0102);
////    DispX->WriteRegister(0x0034,0x0300);
////    DispX->WriteRegister(0x0035,0x0103);
////    DispX->WriteRegister(0x0036,0x001F);
////    DispX->WriteRegister(0x0037,0x0703);
////    DispX->WriteRegister(0x0038,0x0001);
////    DispX->WriteRegister(0x0039,0x0004);
////    
////    DispX->WriteRegister(0x0090, 0x0015);   //80Hz
////    DispX->WriteRegister(0x0010, 0X0410);   //BT,AP
////    DispX->WriteRegister(0x0011,0x0247);    //DC1,DC0,VC
////    DispX->WriteRegister(0x0012, 0x01BC);
////    DispX->WriteRegister(0x0013, 0x0e00);
////    MID_Delay(120);
////    DispX->WriteRegister(0x0001, 0x0100);
////    DispX->WriteRegister(0x0002, 0x0200);
////    DispX->WriteRegister(0x0003, 0x1030);
////    
////    DispX->WriteRegister(0x000A, 0x0008);
////    DispX->WriteRegister(0x000C, 0x0000);
////    
////    DispX->WriteRegister(0x000E, 0x0020);
////    DispX->WriteRegister(0x000F, 0x0000);
////    DispX->WriteRegister(0x0020, 0x0000);   //H Start
////    DispX->WriteRegister(0x0021, 0x0000);   //V Start
////    DispX->WriteRegister(0x002A,0x003D);    //vcom2
////    MID_Delay(20);
////    DispX->WriteRegister(0x0029, 0x002d);
////    DispX->WriteRegister(0x0050, 0x0000);
////    DispX->WriteRegister(0x0051, 0xD0EF);
////    DispX->WriteRegister(0x0052, 0x0000);
////    DispX->WriteRegister(0x0053, 0x013F);
////    DispX->WriteRegister(0x0061, 0x0000);
////    DispX->WriteRegister(0x006A, 0x0000);
////    DispX->WriteRegister(0x0092,0x0300); 

////    DispX->WriteRegister(0x0093, 0x0005);
////    DispX->WriteRegister(0x0007, 0x0100);
////}

/////**
//// *******************************************************************************
//// * @brief       8989 Chip Inital.
//// * @return      None
//// *******************************************************************************
//// */
////void LCD_ChipInit_8989(Display_HandleTypeDef *DispX)
////{
////    DispX->WriteRegister(0x00,0x0001);//打開晶振
////    DispX->WriteRegister(0x03,0xA8A4);//0xA8A4
////    DispX->WriteRegister(0x0C,0x0000);    
////    DispX->WriteRegister(0x0D,0x080C);   
////    DispX->WriteRegister(0x0E,0x2B00);    
////    DispX->WriteRegister(0x1E,0x00B0);    
////    DispX->WriteRegister(0x01,0x2B3F);//驅動輸出控制320*240  0x6B3F
////    DispX->WriteRegister(0x02,0x0600);
////    DispX->WriteRegister(0x10,0x0000);  
////    DispX->WriteRegister(0x11,0x6078); //定義數據格式  16位色     橫屏 0x6058
////    DispX->WriteRegister(0x05,0x0000);  
////    DispX->WriteRegister(0x06,0x0000);  
////    DispX->WriteRegister(0x16,0xEF1C);  
////    DispX->WriteRegister(0x17,0x0003);  
////    DispX->WriteRegister(0x07,0x0233); //0x0233
////    DispX->WriteRegister(0x0B,0x0000);  
////    DispX->WriteRegister(0x0F,0x0000); //掃描開始地址
////    DispX->WriteRegister(0x41,0x0000);  
////    DispX->WriteRegister(0x42,0x0000);  
////    DispX->WriteRegister(0x48,0x0000);  
////    DispX->WriteRegister(0x49,0x013F);  
////    DispX->WriteRegister(0x4A,0x0000);  
////    DispX->WriteRegister(0x4B,0x0000);  
////    DispX->WriteRegister(0x44,0xEF00);  
////    DispX->WriteRegister(0x45,0x0000);  
////    DispX->WriteRegister(0x46,0x013F);  
////    DispX->WriteRegister(0x30,0x0707);  
////    DispX->WriteRegister(0x31,0x0204);  
////    DispX->WriteRegister(0x32,0x0204);  
////    DispX->WriteRegister(0x33,0x0502);  
////    DispX->WriteRegister(0x34,0x0507);  
////    DispX->WriteRegister(0x35,0x0204);  
////    DispX->WriteRegister(0x36,0x0204);  
////    DispX->WriteRegister(0x37,0x0502);  
////    DispX->WriteRegister(0x3A,0x0302);  
////    DispX->WriteRegister(0x3B,0x0302);  
////    DispX->WriteRegister(0x23,0x0000);  
////    DispX->WriteRegister(0x24,0x0000);  
////    DispX->WriteRegister(0x25,0x8000);  
////    DispX->WriteRegister(0x4f,0);        //行首址0
////    DispX->WriteRegister(0x4e,0);        //列首址0
////}

/////**
//// *******************************************************************************
//// * @brief       4531 Chip Inital.
//// * @return      None
//// *******************************************************************************
//// */
////void LCD_ChipInit_4531(Display_HandleTypeDef *DispX)
////{
////    DispX->WriteRegister(0X00,0X0001);   
////    MID_Delay(10);   
////    DispX->WriteRegister(0X10,0X1628);   
////    DispX->WriteRegister(0X12,0X000e);//0x0006    
////    DispX->WriteRegister(0X13,0X0A39);   
////    MID_Delay(10);   
////    DispX->WriteRegister(0X11,0X0040);   
////    DispX->WriteRegister(0X15,0X0050);   
////    MID_Delay(10);   
////    DispX->WriteRegister(0X12,0X001e);//16    
////    MID_Delay(10);   
////    DispX->WriteRegister(0X10,0X1620);   
////    DispX->WriteRegister(0X13,0X2A39);   
////    MID_Delay(10);   
////    DispX->WriteRegister(0X01,0X0100);   
////    DispX->WriteRegister(0X02,0X0300);   
////    DispX->WriteRegister(0X03,0X1038);//改變方向的   
////    DispX->WriteRegister(0X08,0X0202);   
////    DispX->WriteRegister(0X0A,0X0008);   
////    DispX->WriteRegister(0X30,0X0000);   
////    DispX->WriteRegister(0X31,0X0402);   
////    DispX->WriteRegister(0X32,0X0106);   
////    DispX->WriteRegister(0X33,0X0503);   
////    DispX->WriteRegister(0X34,0X0104);   
////    DispX->WriteRegister(0X35,0X0301);   
////    DispX->WriteRegister(0X36,0X0707);   
////    DispX->WriteRegister(0X37,0X0305);   
////    DispX->WriteRegister(0X38,0X0208);   
////    DispX->WriteRegister(0X39,0X0F0B);   
////    DispX->WriteRegister(0X41,0X0002);   
////    DispX->WriteRegister(0X60,0X2700);   
////    DispX->WriteRegister(0X61,0X0001);   
////    DispX->WriteRegister(0X90,0X0210);   
////    DispX->WriteRegister(0X92,0X010A);   
////    DispX->WriteRegister(0X93,0X0004);   
////    DispX->WriteRegister(0XA0,0X0100);   
////    DispX->WriteRegister(0X07,0X0001);   
////    DispX->WriteRegister(0X07,0X0021);   
////    DispX->WriteRegister(0X07,0X0023);   
////    DispX->WriteRegister(0X07,0X0033);   
////    DispX->WriteRegister(0X07,0X0133);   
////    DispX->WriteRegister(0XA0,0X0000); 
////}

/////**
//// *******************************************************************************
//// * @brief       4535 Chip Inital.
//// * @return      None
//// *******************************************************************************
//// */
////void LCD_ChipInit_4535(Display_HandleTypeDef *DispX)
////{
////    DispX->WriteRegister(0X15,0X0030);   
////    DispX->WriteRegister(0X9A,0X0010);   
////    DispX->WriteRegister(0X11,0X0020);   
////    DispX->WriteRegister(0X10,0X3428);   
////    DispX->WriteRegister(0X12,0X0002);//16    
////    DispX->WriteRegister(0X13,0X1038);   
////    MID_Delay(40);   
////    DispX->WriteRegister(0X12,0X0012);//16    
////    MID_Delay(40);   
////    DispX->WriteRegister(0X10,0X3420);   
////    DispX->WriteRegister(0X13,0X3038);   
////    MID_Delay(70);   
////    DispX->WriteRegister(0X30,0X0000);   
////    DispX->WriteRegister(0X31,0X0402);   
////    DispX->WriteRegister(0X32,0X0307);   
////    DispX->WriteRegister(0X33,0X0304);   
////    DispX->WriteRegister(0X34,0X0004);   
////    DispX->WriteRegister(0X35,0X0401);   
////    DispX->WriteRegister(0X36,0X0707);   
////    DispX->WriteRegister(0X37,0X0305);   
////    DispX->WriteRegister(0X38,0X0610);   
////    DispX->WriteRegister(0X39,0X0610); 
////
////    DispX->WriteRegister(0X01,0X0100);   
////    DispX->WriteRegister(0X02,0X0300);   
////    DispX->WriteRegister(0X03,0X1030);//改變方向的   
////    DispX->WriteRegister(0X08,0X0808);   
////    DispX->WriteRegister(0X0A,0X0008);   
////    DispX->WriteRegister(0X60,0X2700);   
////    DispX->WriteRegister(0X61,0X0001);   
////    DispX->WriteRegister(0X90,0X013E);   
////    DispX->WriteRegister(0X92,0X0100);   
////    DispX->WriteRegister(0X93,0X0100);   
////    DispX->WriteRegister(0XA0,0X3000);   
////    DispX->WriteRegister(0XA3,0X0010);   
////    DispX->WriteRegister(0X07,0X0001);   
////    DispX->WriteRegister(0X07,0X0021);   
////    DispX->WriteRegister(0X07,0X0023);   
////    DispX->WriteRegister(0X07,0X0033);   
////    DispX->WriteRegister(0X07,0X0133);   
////}

/**
 *******************************************************************************
 * @brief       NT35310 Chip Inital.
 * @return      None
 *******************************************************************************
 */
void LCD_NT35310_SPI_SelectRegister(uint8_t Register)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Register));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_NT35310_SPI_WriteData(uint16_t Data)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Data));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_NT35310_SPI_Write8BitsData(uint8_t Data)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Data));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
uint16_t LCD_NT35310_SPI_ReadData(void)
{
    return 0;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
uint8_t LCD_NT35310_SPI_Read8BitsData(void)
{
    return 0;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_NT35310_Initial(Display_HandleTypeDef *DispX)
{
    DispX->SelectReg(0xED);
    DispX->WriteData(0x01);
    DispX->WriteData(0xFE);

    DispX->SelectReg(0xEE);
    DispX->WriteData(0xDE);
    DispX->WriteData(0x21);

    DispX->SelectReg(0xF1);
    DispX->WriteData(0x01);
    DispX->SelectReg(0xDF);
    DispX->WriteData(0x10);

    //VCOMvoltage//
    DispX->SelectReg(0xC4);
    DispX->WriteData(0x8F);     //5f

    DispX->SelectReg(0xC6);
    DispX->WriteData(0x00);
    DispX->WriteData(0xE2);
    DispX->WriteData(0xE2);
    DispX->WriteData(0xE2);
    DispX->SelectReg(0xBF);
    DispX->WriteData(0xAA);

    DispX->SelectReg(0xB0);
    DispX->WriteData(0x0D);
    DispX->WriteData(0x00);
    DispX->WriteData(0x0D);
    DispX->WriteData(0x00);
    DispX->WriteData(0x11);
    DispX->WriteData(0x00);
    DispX->WriteData(0x19);
    DispX->WriteData(0x00);
    DispX->WriteData(0x21);
    DispX->WriteData(0x00);
    DispX->WriteData(0x2D);
    DispX->WriteData(0x00);
    DispX->WriteData(0x3D);
    DispX->WriteData(0x00);
    DispX->WriteData(0x5D);
    DispX->WriteData(0x00);
    DispX->WriteData(0x5D);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xB1);
    DispX->WriteData(0x80);
    DispX->WriteData(0x00);
    DispX->WriteData(0x8B);
    DispX->WriteData(0x00);
    DispX->WriteData(0x96);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xB2);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x02);
    DispX->WriteData(0x00);
    DispX->WriteData(0x03);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xB3);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xB4);
    DispX->WriteData(0x8B);
    DispX->WriteData(0x00);
    DispX->WriteData(0x96);
    DispX->WriteData(0x00);
    DispX->WriteData(0xA1);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xB5);
    DispX->WriteData(0x02);
    DispX->WriteData(0x00);
    DispX->WriteData(0x03);
    DispX->WriteData(0x00);
    DispX->WriteData(0x04);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xB6);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xB7);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x3F);
    DispX->WriteData(0x00);
    DispX->WriteData(0x5E);
    DispX->WriteData(0x00);
    DispX->WriteData(0x64);
    DispX->WriteData(0x00);
    DispX->WriteData(0x8C);
    DispX->WriteData(0x00);
    DispX->WriteData(0xAC);
    DispX->WriteData(0x00);
    DispX->WriteData(0xDC);
    DispX->WriteData(0x00);
    DispX->WriteData(0x70);
    DispX->WriteData(0x00);
    DispX->WriteData(0x90);
    DispX->WriteData(0x00);
    DispX->WriteData(0xEB);
    DispX->WriteData(0x00);
    DispX->WriteData(0xDC);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xB8);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xBA);
    DispX->WriteData(0x24);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xC1);
    DispX->WriteData(0x20);
    DispX->WriteData(0x00);
    DispX->WriteData(0x54);
    DispX->WriteData(0x00);
    DispX->WriteData(0xFF);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xC2);
    DispX->WriteData(0x0A);
    DispX->WriteData(0x00);
    DispX->WriteData(0x04);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xC3);
    DispX->WriteData(0x3C);
    DispX->WriteData(0x00);
    DispX->WriteData(0x3A);
    DispX->WriteData(0x00);
    DispX->WriteData(0x39);
    DispX->WriteData(0x00);
    DispX->WriteData(0x37);
    DispX->WriteData(0x00);
    DispX->WriteData(0x3C);
    DispX->WriteData(0x00);
    DispX->WriteData(0x36);
    DispX->WriteData(0x00);
    DispX->WriteData(0x32);
    DispX->WriteData(0x00);
    DispX->WriteData(0x2F);
    DispX->WriteData(0x00);
    DispX->WriteData(0x2C);
    DispX->WriteData(0x00);
    DispX->WriteData(0x29);
    DispX->WriteData(0x00);
    DispX->WriteData(0x26);
    DispX->WriteData(0x00);
    DispX->WriteData(0x24);
    DispX->WriteData(0x00);
    DispX->WriteData(0x24);
    DispX->WriteData(0x00);
    DispX->WriteData(0x23);
    DispX->WriteData(0x00);
    DispX->WriteData(0x3C);
    DispX->WriteData(0x00);
    DispX->WriteData(0x36);
    DispX->WriteData(0x00);
    DispX->WriteData(0x32);
    DispX->WriteData(0x00);
    DispX->WriteData(0x2F);
    DispX->WriteData(0x00);
    DispX->WriteData(0x2C);
    DispX->WriteData(0x00);
    DispX->WriteData(0x29);
    DispX->WriteData(0x00);
    DispX->WriteData(0x26);
    DispX->WriteData(0x00);
    DispX->WriteData(0x24);
    DispX->WriteData(0x00);
    DispX->WriteData(0x24);
    DispX->WriteData(0x00);
    DispX->WriteData(0x23);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xC4);
    DispX->WriteData(0x62);
    DispX->WriteData(0x00);
    DispX->WriteData(0x05);
    DispX->WriteData(0x00);
    DispX->WriteData(0x84);
    DispX->WriteData(0x00);
    DispX->WriteData(0xF0);
    DispX->WriteData(0x00);
    DispX->WriteData(0x18);
    DispX->WriteData(0x00);
    DispX->WriteData(0xA4);
    DispX->WriteData(0x00);
    DispX->WriteData(0x18);
    DispX->WriteData(0x00);
    DispX->WriteData(0x50);
    DispX->WriteData(0x00);
    DispX->WriteData(0x0C);
    DispX->WriteData(0x00);
    DispX->WriteData(0x17);
    DispX->WriteData(0x00);
    DispX->WriteData(0x95);
    DispX->WriteData(0x00);
    DispX->WriteData(0xF3);
    DispX->WriteData(0x00);
    DispX->WriteData(0xE6);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xC5);
    DispX->WriteData(0x32);
    DispX->WriteData(0x00);
    DispX->WriteData(0x44);
    DispX->WriteData(0x00);
    DispX->WriteData(0x65);
    DispX->WriteData(0x00);
    DispX->WriteData(0x76);
    DispX->WriteData(0x00);
    DispX->WriteData(0x88);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xC6);
    DispX->WriteData(0x20);
    DispX->WriteData(0x00);
    DispX->WriteData(0x17);
    DispX->WriteData(0x00);
    DispX->WriteData(0x01);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xC7);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xC8);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xC9);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xE0);
    DispX->WriteData(0x16);
    DispX->WriteData(0x00);
    DispX->WriteData(0x1C);
    DispX->WriteData(0x00);
    DispX->WriteData(0x21);
    DispX->WriteData(0x00);
    DispX->WriteData(0x36);
    DispX->WriteData(0x00);
    DispX->WriteData(0x46);
    DispX->WriteData(0x00);
    DispX->WriteData(0x52);
    DispX->WriteData(0x00);
    DispX->WriteData(0x64);
    DispX->WriteData(0x00);
    DispX->WriteData(0x7A);
    DispX->WriteData(0x00);
    DispX->WriteData(0x8B);
    DispX->WriteData(0x00);
    DispX->WriteData(0x99);
    DispX->WriteData(0x00);
    DispX->WriteData(0xA8);
    DispX->WriteData(0x00);
    DispX->WriteData(0xB9);
    DispX->WriteData(0x00);
    DispX->WriteData(0xC4);
    DispX->WriteData(0x00);
    DispX->WriteData(0xCA);
    DispX->WriteData(0x00);
    DispX->WriteData(0xD2);
    DispX->WriteData(0x00);
    DispX->WriteData(0xD9);
    DispX->WriteData(0x00);
    DispX->WriteData(0xE0);
    DispX->WriteData(0x00);
    DispX->WriteData(0xF3);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xE1);
    DispX->WriteData(0x16);
    DispX->WriteData(0x00);
    DispX->WriteData(0x1C);
    DispX->WriteData(0x00);
    DispX->WriteData(0x22);
    DispX->WriteData(0x00);
    DispX->WriteData(0x36);
    DispX->WriteData(0x00);
    DispX->WriteData(0x45);
    DispX->WriteData(0x00);
    DispX->WriteData(0x52);
    DispX->WriteData(0x00);
    DispX->WriteData(0x64);
    DispX->WriteData(0x00);
    DispX->WriteData(0x7A);
    DispX->WriteData(0x00);
    DispX->WriteData(0x8B);
    DispX->WriteData(0x00);
    DispX->WriteData(0x99);
    DispX->WriteData(0x00);
    DispX->WriteData(0xA8);
    DispX->WriteData(0x00);
    DispX->WriteData(0xB9);
    DispX->WriteData(0x00);
    DispX->WriteData(0xC4);
    DispX->WriteData(0x00);
    DispX->WriteData(0xCA);
    DispX->WriteData(0x00);
    DispX->WriteData(0xD2);
    DispX->WriteData(0x00);
    DispX->WriteData(0xD8);
    DispX->WriteData(0x00);
    DispX->WriteData(0xE0);
    DispX->WriteData(0x00);
    DispX->WriteData(0xF3);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xE2);
    DispX->WriteData(0x05);
    DispX->WriteData(0x00);
    DispX->WriteData(0x0B);
    DispX->WriteData(0x00);
    DispX->WriteData(0x1B);
    DispX->WriteData(0x00);
    DispX->WriteData(0x34);
    DispX->WriteData(0x00);
    DispX->WriteData(0x44);
    DispX->WriteData(0x00);
    DispX->WriteData(0x4F);
    DispX->WriteData(0x00);
    DispX->WriteData(0x61);
    DispX->WriteData(0x00);
    DispX->WriteData(0x79);
    DispX->WriteData(0x00);
    DispX->WriteData(0x88);
    DispX->WriteData(0x00);
    DispX->WriteData(0x97);
    DispX->WriteData(0x00);
    DispX->WriteData(0xA6);
    DispX->WriteData(0x00);
    DispX->WriteData(0xB7);
    DispX->WriteData(0x00);
    DispX->WriteData(0xC2);
    DispX->WriteData(0x00);
    DispX->WriteData(0xC7);
    DispX->WriteData(0x00);
    DispX->WriteData(0xD1);
    DispX->WriteData(0x00);
    DispX->WriteData(0xD6);
    DispX->WriteData(0x00);
    DispX->WriteData(0xDD);
    DispX->WriteData(0x00);
    DispX->WriteData(0xF3);
    DispX->WriteData(0x00);
    DispX->SelectReg(0xE3);
    DispX->WriteData(0x05);
    DispX->WriteData(0x00);
    DispX->WriteData(0xA);
    DispX->WriteData(0x00);
    DispX->WriteData(0x1C);
    DispX->WriteData(0x00);
    DispX->WriteData(0x33);
    DispX->WriteData(0x00);
    DispX->WriteData(0x44);
    DispX->WriteData(0x00);
    DispX->WriteData(0x50);
    DispX->WriteData(0x00);
    DispX->WriteData(0x62);
    DispX->WriteData(0x00);
    DispX->WriteData(0x78);
    DispX->WriteData(0x00);
    DispX->WriteData(0x88);
    DispX->WriteData(0x00);
    DispX->WriteData(0x97);
    DispX->WriteData(0x00);
    DispX->WriteData(0xA6);
    DispX->WriteData(0x00);
    DispX->WriteData(0xB7);
    DispX->WriteData(0x00);
    DispX->WriteData(0xC2);
    DispX->WriteData(0x00);
    DispX->WriteData(0xC7);
    DispX->WriteData(0x00);
    DispX->WriteData(0xD1);
    DispX->WriteData(0x00);
    DispX->WriteData(0xD5);
    DispX->WriteData(0x00);
    DispX->WriteData(0xDD);
    DispX->WriteData(0x00);
    DispX->WriteData(0xF3);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xE4);
    DispX->WriteData(0x01);
    DispX->WriteData(0x00);
    DispX->WriteData(0x01);
    DispX->WriteData(0x00);
    DispX->WriteData(0x02);
    DispX->WriteData(0x00);
    DispX->WriteData(0x2A);
    DispX->WriteData(0x00);
    DispX->WriteData(0x3C);
    DispX->WriteData(0x00);
    DispX->WriteData(0x4B);
    DispX->WriteData(0x00);
    DispX->WriteData(0x5D);
    DispX->WriteData(0x00);
    DispX->WriteData(0x74);
    DispX->WriteData(0x00);
    DispX->WriteData(0x84);
    DispX->WriteData(0x00);
    DispX->WriteData(0x93);
    DispX->WriteData(0x00);
    DispX->WriteData(0xA2);
    DispX->WriteData(0x00);
    DispX->WriteData(0xB3);
    DispX->WriteData(0x00);
    DispX->WriteData(0xBE);
    DispX->WriteData(0x00);
    DispX->WriteData(0xC4);
    DispX->WriteData(0x00);
    DispX->WriteData(0xCD);
    DispX->WriteData(0x00);
    DispX->WriteData(0xD3);
    DispX->WriteData(0x00);
    DispX->WriteData(0xDD);
    DispX->WriteData(0x00);
    DispX->WriteData(0xF3);
    DispX->WriteData(0x00);
    DispX->SelectReg(0xE5);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x02);
    DispX->WriteData(0x00);
    DispX->WriteData(0x29);
    DispX->WriteData(0x00);
    DispX->WriteData(0x3C);
    DispX->WriteData(0x00);
    DispX->WriteData(0x4B);
    DispX->WriteData(0x00);
    DispX->WriteData(0x5D);
    DispX->WriteData(0x00);
    DispX->WriteData(0x74);
    DispX->WriteData(0x00);
    DispX->WriteData(0x84);
    DispX->WriteData(0x00);
    DispX->WriteData(0x93);
    DispX->WriteData(0x00);
    DispX->WriteData(0xA2);
    DispX->WriteData(0x00);
    DispX->WriteData(0xB3);
    DispX->WriteData(0x00);
    DispX->WriteData(0xBE);
    DispX->WriteData(0x00);
    DispX->WriteData(0xC4);
    DispX->WriteData(0x00);
    DispX->WriteData(0xCD);
    DispX->WriteData(0x00);
    DispX->WriteData(0xD3);
    DispX->WriteData(0x00);
    DispX->WriteData(0xDC);
    DispX->WriteData(0x00);
    DispX->WriteData(0xF3);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xE6);
    DispX->WriteData(0x11);
    DispX->WriteData(0x00);
    DispX->WriteData(0x34);
    DispX->WriteData(0x00);
    DispX->WriteData(0x56);
    DispX->WriteData(0x00);
    DispX->WriteData(0x76);
    DispX->WriteData(0x00);
    DispX->WriteData(0x77);
    DispX->WriteData(0x00);
    DispX->WriteData(0x66);
    DispX->WriteData(0x00);
    DispX->WriteData(0x88);
    DispX->WriteData(0x00);
    DispX->WriteData(0x99);
    DispX->WriteData(0x00);
    DispX->WriteData(0xBB);
    DispX->WriteData(0x00);
    DispX->WriteData(0x99);
    DispX->WriteData(0x00);
    DispX->WriteData(0x66);
    DispX->WriteData(0x00);
    DispX->WriteData(0x55);
    DispX->WriteData(0x00);
    DispX->WriteData(0x55);
    DispX->WriteData(0x00);
    DispX->WriteData(0x45);
    DispX->WriteData(0x00);
    DispX->WriteData(0x43);
    DispX->WriteData(0x00);
    DispX->WriteData(0x44);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xE7);
    DispX->WriteData(0x32);
    DispX->WriteData(0x00);
    DispX->WriteData(0x55);
    DispX->WriteData(0x00);
    DispX->WriteData(0x76);
    DispX->WriteData(0x00);
    DispX->WriteData(0x66);
    DispX->WriteData(0x00);
    DispX->WriteData(0x67);
    DispX->WriteData(0x00);
    DispX->WriteData(0x67);
    DispX->WriteData(0x00);
    DispX->WriteData(0x87);
    DispX->WriteData(0x00);
    DispX->WriteData(0x99);
    DispX->WriteData(0x00);
    DispX->WriteData(0xBB);
    DispX->WriteData(0x00);
    DispX->WriteData(0x99);
    DispX->WriteData(0x00);
    DispX->WriteData(0x77);
    DispX->WriteData(0x00);
    DispX->WriteData(0x44);
    DispX->WriteData(0x00);
    DispX->WriteData(0x56);
    DispX->WriteData(0x00);
    DispX->WriteData(0x23); 
    DispX->WriteData(0x00);
    DispX->WriteData(0x33);
    DispX->WriteData(0x00);
    DispX->WriteData(0x45);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xE8);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x99);
    DispX->WriteData(0x00);
    DispX->WriteData(0x87);
    DispX->WriteData(0x00);
    DispX->WriteData(0x88);
    DispX->WriteData(0x00);
    DispX->WriteData(0x77);
    DispX->WriteData(0x00);
    DispX->WriteData(0x66);
    DispX->WriteData(0x00);
    DispX->WriteData(0x88);
    DispX->WriteData(0x00);
    DispX->WriteData(0xAA);
    DispX->WriteData(0x00);
    DispX->WriteData(0xBB);
    DispX->WriteData(0x00);
    DispX->WriteData(0x99);
    DispX->WriteData(0x00);
    DispX->WriteData(0x66);
    DispX->WriteData(0x00);
    DispX->WriteData(0x55);
    DispX->WriteData(0x00);
    DispX->WriteData(0x55);
    DispX->WriteData(0x00);
    DispX->WriteData(0x44);
    DispX->WriteData(0x00);
    DispX->WriteData(0x44);
    DispX->WriteData(0x00);
    DispX->WriteData(0x55);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xE9);
    DispX->WriteData(0xAA);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);

    DispX->SelectReg(0x00);
    DispX->WriteData(0xAA);

    DispX->SelectReg(0xCF);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xF0);
    DispX->WriteData(0x00);
    DispX->WriteData(0x50);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xF3);
    DispX->WriteData(0x00);

    DispX->SelectReg(0xF9);
    DispX->WriteData(0x06);
    DispX->WriteData(0x10);
    DispX->WriteData(0x29);
    DispX->WriteData(0x00);

    DispX->SelectReg(0x3A);
    DispX->WriteData(0x55);  //66

    DispX->SelectReg(0x11);
    MID_Delay(100);
    DispX->SelectReg(0x29);
    DispX->SelectReg(0x35);
    DispX->WriteData(0x00);

    DispX->SelectReg(0x51);
    DispX->WriteData(0xFF);
    DispX->SelectReg(0x53);
    DispX->WriteData(0x2C);
    DispX->SelectReg(0x55);
    DispX->WriteData(0x82);
    DispX->SelectReg(0x2c); 
}

/**
 *******************************************************************************
 * @brief       NT35510 Chip Inital.
 * @return      None
 *******************************************************************************
 */
void LCD_NT35510_SPI_SelectRegister(uint8_t Register)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Register));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_NT35510_SPI_WriteData(uint16_t Data)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Data));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_NT35510_SPI_Write8BitsData(uint8_t Data)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Data));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
uint16_t LCD_NT35510_SPI_ReadData(void)
{
    return 0;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
uint8_t LCD_NT35510_SPI_Read8BitsData(void)
{
    return 0;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_NT35510_SPI_WriteRegister(uint16_t Register, uint16_t Value)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Register));
    ((void)(Value));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_NT35510_Initial(Display_HandleTypeDef *DispX)
{
    DispX->WriteRegister(0xF000,0x55);
    DispX->WriteRegister(0xF001,0xAA);
    DispX->WriteRegister(0xF002,0x52);
    DispX->WriteRegister(0xF003,0x08);
    DispX->WriteRegister(0xF004,0x01);
    //AVDD Set AVDD 5.2V
    DispX->WriteRegister(0xB000,0x0D);
    DispX->WriteRegister(0xB001,0x0D);
    DispX->WriteRegister(0xB002,0x0D);
    //AVDD ratio
    DispX->WriteRegister(0xB600,0x34);
    DispX->WriteRegister(0xB601,0x34);
    DispX->WriteRegister(0xB602,0x34);
    //AVEE -5.2V
    DispX->WriteRegister(0xB100,0x0D);
    DispX->WriteRegister(0xB101,0x0D);
    DispX->WriteRegister(0xB102,0x0D);
    //AVEE ratio
    DispX->WriteRegister(0xB700,0x34);
    DispX->WriteRegister(0xB701,0x34);
    DispX->WriteRegister(0xB702,0x34);
    //VCL -2.5V
    DispX->WriteRegister(0xB200,0x00);
    DispX->WriteRegister(0xB201,0x00);
    DispX->WriteRegister(0xB202,0x00);
    //VCL ratio
    DispX->WriteRegister(0xB800,0x24);
    DispX->WriteRegister(0xB801,0x24);
    DispX->WriteRegister(0xB802,0x24);
    //VGH 15V (Free pump)
    DispX->WriteRegister(0xBF00,0x01);
    DispX->WriteRegister(0xB300,0x0F);
    DispX->WriteRegister(0xB301,0x0F);
    DispX->WriteRegister(0xB302,0x0F);
    //VGH ratio
    DispX->WriteRegister(0xB900,0x34);
    DispX->WriteRegister(0xB901,0x34);
    DispX->WriteRegister(0xB902,0x34);
    //VGL_REG -10V
    DispX->WriteRegister(0xB500,0x08);
    DispX->WriteRegister(0xB501,0x08);
    DispX->WriteRegister(0xB502,0x08);
    DispX->WriteRegister(0xC200,0x03);
    //VGLX ratio
    DispX->WriteRegister(0xBA00,0x24);
    DispX->WriteRegister(0xBA01,0x24);
    DispX->WriteRegister(0xBA02,0x24);
    //VGMP/VGSP 4.5V/0V
    DispX->WriteRegister(0xBC00,0x00);
    DispX->WriteRegister(0xBC01,0x78);
    DispX->WriteRegister(0xBC02,0x00);
    //VGMN/VGSN -4.5V/0V
    DispX->WriteRegister(0xBD00,0x00);
    DispX->WriteRegister(0xBD01,0x78);
    DispX->WriteRegister(0xBD02,0x00);
    //VCOM
    DispX->WriteRegister(0xBE00,0x00);
    DispX->WriteRegister(0xBE01,0x64);
    //Gamma Setting
    DispX->WriteRegister(0xD100,0x00);
    DispX->WriteRegister(0xD101,0x33);
    DispX->WriteRegister(0xD102,0x00);
    DispX->WriteRegister(0xD103,0x34);
    DispX->WriteRegister(0xD104,0x00);
    DispX->WriteRegister(0xD105,0x3A);
    DispX->WriteRegister(0xD106,0x00);
    DispX->WriteRegister(0xD107,0x4A);
    DispX->WriteRegister(0xD108,0x00);
    DispX->WriteRegister(0xD109,0x5C);
    DispX->WriteRegister(0xD10A,0x00);
    DispX->WriteRegister(0xD10B,0x81);
    DispX->WriteRegister(0xD10C,0x00);
    DispX->WriteRegister(0xD10D,0xA6);
    DispX->WriteRegister(0xD10E,0x00);
    DispX->WriteRegister(0xD10F,0xE5);
    DispX->WriteRegister(0xD110,0x01);
    DispX->WriteRegister(0xD111,0x13);
    DispX->WriteRegister(0xD112,0x01);
    DispX->WriteRegister(0xD113,0x54);
    DispX->WriteRegister(0xD114,0x01);
    DispX->WriteRegister(0xD115,0x82);
    DispX->WriteRegister(0xD116,0x01);
    DispX->WriteRegister(0xD117,0xCA);
    DispX->WriteRegister(0xD118,0x02);
    DispX->WriteRegister(0xD119,0x00);
    DispX->WriteRegister(0xD11A,0x02);
    DispX->WriteRegister(0xD11B,0x01);
    DispX->WriteRegister(0xD11C,0x02);
    DispX->WriteRegister(0xD11D,0x34);
    DispX->WriteRegister(0xD11E,0x02);
    DispX->WriteRegister(0xD11F,0x67);
    DispX->WriteRegister(0xD120,0x02);
    DispX->WriteRegister(0xD121,0x84);
    DispX->WriteRegister(0xD122,0x02);
    DispX->WriteRegister(0xD123,0xA4);
    DispX->WriteRegister(0xD124,0x02);
    DispX->WriteRegister(0xD125,0xB7);
    DispX->WriteRegister(0xD126,0x02);
    DispX->WriteRegister(0xD127,0xCF);
    DispX->WriteRegister(0xD128,0x02);
    DispX->WriteRegister(0xD129,0xDE);
    DispX->WriteRegister(0xD12A,0x02);
    DispX->WriteRegister(0xD12B,0xF2);
    DispX->WriteRegister(0xD12C,0x02);
    DispX->WriteRegister(0xD12D,0xFE);
    DispX->WriteRegister(0xD12E,0x03);
    DispX->WriteRegister(0xD12F,0x10);
    DispX->WriteRegister(0xD130,0x03);
    DispX->WriteRegister(0xD131,0x33);
    DispX->WriteRegister(0xD132,0x03);
    DispX->WriteRegister(0xD133,0x6D);
    DispX->WriteRegister(0xD200,0x00);
    DispX->WriteRegister(0xD201,0x33);
    DispX->WriteRegister(0xD202,0x00);
    DispX->WriteRegister(0xD203,0x34);
    DispX->WriteRegister(0xD204,0x00);
    DispX->WriteRegister(0xD205,0x3A);
    DispX->WriteRegister(0xD206,0x00);
    DispX->WriteRegister(0xD207,0x4A);
    DispX->WriteRegister(0xD208,0x00);
    DispX->WriteRegister(0xD209,0x5C);
    DispX->WriteRegister(0xD20A,0x00);

    DispX->WriteRegister(0xD20B,0x81);
    DispX->WriteRegister(0xD20C,0x00);
    DispX->WriteRegister(0xD20D,0xA6);
    DispX->WriteRegister(0xD20E,0x00);
    DispX->WriteRegister(0xD20F,0xE5);
    DispX->WriteRegister(0xD210,0x01);
    DispX->WriteRegister(0xD211,0x13);
    DispX->WriteRegister(0xD212,0x01);
    DispX->WriteRegister(0xD213,0x54);
    DispX->WriteRegister(0xD214,0x01);
    DispX->WriteRegister(0xD215,0x82);
    DispX->WriteRegister(0xD216,0x01);
    DispX->WriteRegister(0xD217,0xCA);
    DispX->WriteRegister(0xD218,0x02);
    DispX->WriteRegister(0xD219,0x00);
    DispX->WriteRegister(0xD21A,0x02);
    DispX->WriteRegister(0xD21B,0x01);
    DispX->WriteRegister(0xD21C,0x02);
    DispX->WriteRegister(0xD21D,0x34);
    DispX->WriteRegister(0xD21E,0x02);
    DispX->WriteRegister(0xD21F,0x67);
    DispX->WriteRegister(0xD220,0x02);
    DispX->WriteRegister(0xD221,0x84);
    DispX->WriteRegister(0xD222,0x02);
    DispX->WriteRegister(0xD223,0xA4);
    DispX->WriteRegister(0xD224,0x02);
    DispX->WriteRegister(0xD225,0xB7);
    DispX->WriteRegister(0xD226,0x02);
    DispX->WriteRegister(0xD227,0xCF);
    DispX->WriteRegister(0xD228,0x02);
    DispX->WriteRegister(0xD229,0xDE);
    DispX->WriteRegister(0xD22A,0x02);
    DispX->WriteRegister(0xD22B,0xF2);
    DispX->WriteRegister(0xD22C,0x02);
    DispX->WriteRegister(0xD22D,0xFE);
    DispX->WriteRegister(0xD22E,0x03);
    DispX->WriteRegister(0xD22F,0x10);
    DispX->WriteRegister(0xD230,0x03);
    DispX->WriteRegister(0xD231,0x33);
    DispX->WriteRegister(0xD232,0x03);
    DispX->WriteRegister(0xD233,0x6D);
    DispX->WriteRegister(0xD300,0x00);
    DispX->WriteRegister(0xD301,0x33);
    DispX->WriteRegister(0xD302,0x00);
    DispX->WriteRegister(0xD303,0x34);
    DispX->WriteRegister(0xD304,0x00);
    DispX->WriteRegister(0xD305,0x3A);
    DispX->WriteRegister(0xD306,0x00);
    DispX->WriteRegister(0xD307,0x4A);
    DispX->WriteRegister(0xD308,0x00);
    DispX->WriteRegister(0xD309,0x5C);
    DispX->WriteRegister(0xD30A,0x00);

    DispX->WriteRegister(0xD30B,0x81);
    DispX->WriteRegister(0xD30C,0x00);
    DispX->WriteRegister(0xD30D,0xA6);
    DispX->WriteRegister(0xD30E,0x00);
    DispX->WriteRegister(0xD30F,0xE5);
    DispX->WriteRegister(0xD310,0x01);
    DispX->WriteRegister(0xD311,0x13);
    DispX->WriteRegister(0xD312,0x01);
    DispX->WriteRegister(0xD313,0x54);
    DispX->WriteRegister(0xD314,0x01);
    DispX->WriteRegister(0xD315,0x82);
    DispX->WriteRegister(0xD316,0x01);
    DispX->WriteRegister(0xD317,0xCA);
    DispX->WriteRegister(0xD318,0x02);
    DispX->WriteRegister(0xD319,0x00);
    DispX->WriteRegister(0xD31A,0x02);
    DispX->WriteRegister(0xD31B,0x01);
    DispX->WriteRegister(0xD31C,0x02);
    DispX->WriteRegister(0xD31D,0x34);
    DispX->WriteRegister(0xD31E,0x02);
    DispX->WriteRegister(0xD31F,0x67);
    DispX->WriteRegister(0xD320,0x02);
    DispX->WriteRegister(0xD321,0x84);
    DispX->WriteRegister(0xD322,0x02);
    DispX->WriteRegister(0xD323,0xA4);
    DispX->WriteRegister(0xD324,0x02);
    DispX->WriteRegister(0xD325,0xB7);
    DispX->WriteRegister(0xD326,0x02);
    DispX->WriteRegister(0xD327,0xCF);
    DispX->WriteRegister(0xD328,0x02);
    DispX->WriteRegister(0xD329,0xDE);
    DispX->WriteRegister(0xD32A,0x02);
    DispX->WriteRegister(0xD32B,0xF2);
    DispX->WriteRegister(0xD32C,0x02);
    DispX->WriteRegister(0xD32D,0xFE);
    DispX->WriteRegister(0xD32E,0x03);
    DispX->WriteRegister(0xD32F,0x10);
    DispX->WriteRegister(0xD330,0x03);
    DispX->WriteRegister(0xD331,0x33);
    DispX->WriteRegister(0xD332,0x03);
    DispX->WriteRegister(0xD333,0x6D);
    DispX->WriteRegister(0xD400,0x00);
    DispX->WriteRegister(0xD401,0x33);
    DispX->WriteRegister(0xD402,0x00);
    DispX->WriteRegister(0xD403,0x34);
    DispX->WriteRegister(0xD404,0x00);
    DispX->WriteRegister(0xD405,0x3A);
    DispX->WriteRegister(0xD406,0x00);
    DispX->WriteRegister(0xD407,0x4A);
    DispX->WriteRegister(0xD408,0x00);
    DispX->WriteRegister(0xD409,0x5C);
    DispX->WriteRegister(0xD40A,0x00);
    DispX->WriteRegister(0xD40B,0x81);

    DispX->WriteRegister(0xD40C,0x00);
    DispX->WriteRegister(0xD40D,0xA6);
    DispX->WriteRegister(0xD40E,0x00);
    DispX->WriteRegister(0xD40F,0xE5);
    DispX->WriteRegister(0xD410,0x01);
    DispX->WriteRegister(0xD411,0x13);
    DispX->WriteRegister(0xD412,0x01);
    DispX->WriteRegister(0xD413,0x54);
    DispX->WriteRegister(0xD414,0x01);
    DispX->WriteRegister(0xD415,0x82);
    DispX->WriteRegister(0xD416,0x01);
    DispX->WriteRegister(0xD417,0xCA);
    DispX->WriteRegister(0xD418,0x02);
    DispX->WriteRegister(0xD419,0x00);
    DispX->WriteRegister(0xD41A,0x02);
    DispX->WriteRegister(0xD41B,0x01);
    DispX->WriteRegister(0xD41C,0x02);
    DispX->WriteRegister(0xD41D,0x34);
    DispX->WriteRegister(0xD41E,0x02);
    DispX->WriteRegister(0xD41F,0x67);
    DispX->WriteRegister(0xD420,0x02);
    DispX->WriteRegister(0xD421,0x84);
    DispX->WriteRegister(0xD422,0x02);
    DispX->WriteRegister(0xD423,0xA4);
    DispX->WriteRegister(0xD424,0x02);
    DispX->WriteRegister(0xD425,0xB7);
    DispX->WriteRegister(0xD426,0x02);
    DispX->WriteRegister(0xD427,0xCF);
    DispX->WriteRegister(0xD428,0x02);
    DispX->WriteRegister(0xD429,0xDE);
    DispX->WriteRegister(0xD42A,0x02);
    DispX->WriteRegister(0xD42B,0xF2);
    DispX->WriteRegister(0xD42C,0x02);
    DispX->WriteRegister(0xD42D,0xFE);
    DispX->WriteRegister(0xD42E,0x03);
    DispX->WriteRegister(0xD42F,0x10);
    DispX->WriteRegister(0xD430,0x03);
    DispX->WriteRegister(0xD431,0x33);
    DispX->WriteRegister(0xD432,0x03);
    DispX->WriteRegister(0xD433,0x6D);
    DispX->WriteRegister(0xD500,0x00);
    DispX->WriteRegister(0xD501,0x33);
    DispX->WriteRegister(0xD502,0x00);
    DispX->WriteRegister(0xD503,0x34);
    DispX->WriteRegister(0xD504,0x00);
    DispX->WriteRegister(0xD505,0x3A);
    DispX->WriteRegister(0xD506,0x00);
    DispX->WriteRegister(0xD507,0x4A);
    DispX->WriteRegister(0xD508,0x00);
    DispX->WriteRegister(0xD509,0x5C);
    DispX->WriteRegister(0xD50A,0x00);
    DispX->WriteRegister(0xD50B,0x81);

    DispX->WriteRegister(0xD50C,0x00);
    DispX->WriteRegister(0xD50D,0xA6);
    DispX->WriteRegister(0xD50E,0x00);
    DispX->WriteRegister(0xD50F,0xE5);
    DispX->WriteRegister(0xD510,0x01);
    DispX->WriteRegister(0xD511,0x13);
    DispX->WriteRegister(0xD512,0x01);
    DispX->WriteRegister(0xD513,0x54);
    DispX->WriteRegister(0xD514,0x01);
    DispX->WriteRegister(0xD515,0x82);
    DispX->WriteRegister(0xD516,0x01);
    DispX->WriteRegister(0xD517,0xCA);
    DispX->WriteRegister(0xD518,0x02);
    DispX->WriteRegister(0xD519,0x00);
    DispX->WriteRegister(0xD51A,0x02);
    DispX->WriteRegister(0xD51B,0x01);
    DispX->WriteRegister(0xD51C,0x02);
    DispX->WriteRegister(0xD51D,0x34);
    DispX->WriteRegister(0xD51E,0x02);
    DispX->WriteRegister(0xD51F,0x67);
    DispX->WriteRegister(0xD520,0x02);
    DispX->WriteRegister(0xD521,0x84);
    DispX->WriteRegister(0xD522,0x02);
    DispX->WriteRegister(0xD523,0xA4);
    DispX->WriteRegister(0xD524,0x02);
    DispX->WriteRegister(0xD525,0xB7);
    DispX->WriteRegister(0xD526,0x02);
    DispX->WriteRegister(0xD527,0xCF);
    DispX->WriteRegister(0xD528,0x02);
    DispX->WriteRegister(0xD529,0xDE);
    DispX->WriteRegister(0xD52A,0x02);
    DispX->WriteRegister(0xD52B,0xF2);
    DispX->WriteRegister(0xD52C,0x02);
    DispX->WriteRegister(0xD52D,0xFE);
    DispX->WriteRegister(0xD52E,0x03);
    DispX->WriteRegister(0xD52F,0x10);
    DispX->WriteRegister(0xD530,0x03);
    DispX->WriteRegister(0xD531,0x33);
    DispX->WriteRegister(0xD532,0x03);
    DispX->WriteRegister(0xD533,0x6D);
    DispX->WriteRegister(0xD600,0x00);
    DispX->WriteRegister(0xD601,0x33);
    DispX->WriteRegister(0xD602,0x00);
    DispX->WriteRegister(0xD603,0x34);
    DispX->WriteRegister(0xD604,0x00);
    DispX->WriteRegister(0xD605,0x3A);
    DispX->WriteRegister(0xD606,0x00);
    DispX->WriteRegister(0xD607,0x4A);
    DispX->WriteRegister(0xD608,0x00);
    DispX->WriteRegister(0xD609,0x5C);
    DispX->WriteRegister(0xD60A,0x00);
    DispX->WriteRegister(0xD60B,0x81);

    DispX->WriteRegister(0xD60C,0x00);
    DispX->WriteRegister(0xD60D,0xA6);
    DispX->WriteRegister(0xD60E,0x00);
    DispX->WriteRegister(0xD60F,0xE5);
    DispX->WriteRegister(0xD610,0x01);
    DispX->WriteRegister(0xD611,0x13);
    DispX->WriteRegister(0xD612,0x01);
    DispX->WriteRegister(0xD613,0x54);
    DispX->WriteRegister(0xD614,0x01);
    DispX->WriteRegister(0xD615,0x82);
    DispX->WriteRegister(0xD616,0x01);
    DispX->WriteRegister(0xD617,0xCA);
    DispX->WriteRegister(0xD618,0x02);
    DispX->WriteRegister(0xD619,0x00);
    DispX->WriteRegister(0xD61A,0x02);
    DispX->WriteRegister(0xD61B,0x01);
    DispX->WriteRegister(0xD61C,0x02);
    DispX->WriteRegister(0xD61D,0x34);
    DispX->WriteRegister(0xD61E,0x02);
    DispX->WriteRegister(0xD61F,0x67);
    DispX->WriteRegister(0xD620,0x02);
    DispX->WriteRegister(0xD621,0x84);
    DispX->WriteRegister(0xD622,0x02);
    DispX->WriteRegister(0xD623,0xA4);
    DispX->WriteRegister(0xD624,0x02);
    DispX->WriteRegister(0xD625,0xB7);
    DispX->WriteRegister(0xD626,0x02);
    DispX->WriteRegister(0xD627,0xCF);
    DispX->WriteRegister(0xD628,0x02);
    DispX->WriteRegister(0xD629,0xDE);
    DispX->WriteRegister(0xD62A,0x02);
    DispX->WriteRegister(0xD62B,0xF2);
    DispX->WriteRegister(0xD62C,0x02);
    DispX->WriteRegister(0xD62D,0xFE);
    DispX->WriteRegister(0xD62E,0x03);
    DispX->WriteRegister(0xD62F,0x10);
    DispX->WriteRegister(0xD630,0x03);
    DispX->WriteRegister(0xD631,0x33);
    DispX->WriteRegister(0xD632,0x03);
    DispX->WriteRegister(0xD633,0x6D);
    //LV2 Page 0 enable
    DispX->WriteRegister(0xF000,0x55);
    DispX->WriteRegister(0xF001,0xAA);
    DispX->WriteRegister(0xF002,0x52);
    DispX->WriteRegister(0xF003,0x08);
    DispX->WriteRegister(0xF004,0x00);
    //Display control
    DispX->WriteRegister(0xB100, 0xCC);
    DispX->WriteRegister(0xB101, 0x00);
    //Source hold time
    DispX->WriteRegister(0xB600,0x05);
    //Gate EQ control
    DispX->WriteRegister(0xB700,0x70);
    DispX->WriteRegister(0xB701,0x70);
    //Source EQ control (Mode 2)
    DispX->WriteRegister(0xB800,0x01);
    DispX->WriteRegister(0xB801,0x03);
    DispX->WriteRegister(0xB802,0x03);
    DispX->WriteRegister(0xB803,0x03);
    //Inversion mode (2-dot)
    DispX->WriteRegister(0xBC00,0x02);
    DispX->WriteRegister(0xBC01,0x00);
    DispX->WriteRegister(0xBC02,0x00);
    //Timing control 4H w/ 4-delay
    DispX->WriteRegister(0xC900,0xD0);
    DispX->WriteRegister(0xC901,0x02);
    DispX->WriteRegister(0xC902,0x50);
    DispX->WriteRegister(0xC903,0x50);
    DispX->WriteRegister(0xC904,0x50);
    DispX->WriteRegister(0x3500,0x00);
    DispX->WriteRegister(0x3A00,0x55);  //16-bit/pixel
    DispX->SelectReg(0x1100);
    delay_us(120);
    DispX->SelectReg(0x2900);
}

/**
 *******************************************************************************
 * @brief       ILI9320 Chip Inital.
 * @return      None
 *******************************************************************************
 */
void LCD_ILI93xx_SPI_SelectRegister(uint8_t Register)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Register));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_ILI93xx_SPI_WriteData(uint16_t Data)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Data));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_ILI93xx_SPI_Write8BitsData(uint8_t Data)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Data));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
uint16_t LCD_ILI93xx_SPI_ReadData(void)
{
    return 0;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
uint8_t LCD_ILI93xx_SPI_Read8BitsData(void)
{
    return 0;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_ILI93xx_SPI_WriteRegister(uint16_t Register, uint16_t Value)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Register));
    ((void)(Value));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_ILI9320_Initial(Display_HandleTypeDef *DispX)
{
    DispX->WriteRegister(0x00,0x0000);
    DispX->WriteRegister(0x01,0x0100);          //Driver Output Contral.
    DispX->WriteRegister(0x02,0x0700);          //LCD Driver Waveform Contral.
    DispX->WriteRegister(0x03,0x1030);          //Entry Mode Set.
    //DispX->WriteRegister(0x03,0x1018);        //Entry Mode Set.

    DispX->WriteRegister(0x04,0x0000);          //Scalling Contral.
    DispX->WriteRegister(0x08,0x0202);          //Display Contral 2.(0x0207)
    DispX->WriteRegister(0x09,0x0000);          //Display Contral 3.(0x0000)
    DispX->WriteRegister(0x0a,0x0000);          //Frame Cycle Contal.(0x0000)
    DispX->WriteRegister(0x0c,(1<<0));          //Extern Display Interface Contral 1.(0x0000)
    DispX->WriteRegister(0x0d,0x0000);          //Frame Maker Position.
    DispX->WriteRegister(0x0f,0x0000);          //Extern Display Interface Contral 2.
    MID_Delay(50); 
    DispX->WriteRegister(0x07,0x0101);          //Display Contral.
    MID_Delay(50);
    DispX->WriteRegister(0x10,(1<<12)|(0<<8)|(1<<7)|(1<<6)|(0<<4));     //Power Control 1.(0x16b0)
    DispX->WriteRegister(0x11,0x0007);          //Power Control 2.(0x0001)
    DispX->WriteRegister(0x12,(1<<8)|(1<<4)|(0<<0));    //Power Control 3.(0x0138)
    DispX->WriteRegister(0x13,0x0b00);          //Power Control 4.
    DispX->WriteRegister(0x29,0x0000);          //Power Control 7.

    DispX->WriteRegister(0x2b,(1<<14)|(1<<4));
    DispX->WriteRegister(0x50,0);               //Set X Star
    //水平GRAM終止位置Set X End.
    DispX->WriteRegister(0x51,239);             //Set Y Star
    DispX->WriteRegister(0x52,0);               //Set Y End.t.
    DispX->WriteRegister(0x53,319);             //

    DispX->WriteRegister(0x60,0x2700);          //Driver Output Control.
    DispX->WriteRegister(0x61,0x0001);          //Driver Output Control.
    DispX->WriteRegister(0x6a,0x0000);          //Vertical Srcoll Control.

    DispX->WriteRegister(0x80,0x0000);          //Display Position? Partial Display 1.
    DispX->WriteRegister(0x81,0x0000);          //RAM Address Start? Partial Display 1.
    DispX->WriteRegister(0x82,0x0000);          //RAM Address End-Partial Display 1.
    DispX->WriteRegister(0x83,0x0000);          //Displsy Position? Partial Display 2.
    DispX->WriteRegister(0x84,0x0000);          //RAM Address Start? Partial Display 2.
    DispX->WriteRegister(0x85,0x0000);          //RAM Address End? Partial Display 2.

    DispX->WriteRegister(0x90,(0<<7)|(16<<0));  //Frame Cycle Contral.(0x0013)
    DispX->WriteRegister(0x92,0x0000);          //Panel Interface Contral 2.(0x0000)
    DispX->WriteRegister(0x93,0x0001);          //Panel Interface Contral 3.
    DispX->WriteRegister(0x95,0x0110);          //Frame Cycle Contral.(0x0110)
    DispX->WriteRegister(0x97,(0<<8));          //
    DispX->WriteRegister(0x98,0x0000);          //Frame Cycle Contral.
    DispX->WriteRegister(0x07,0x0173);          //(0x0173)
}

/**
 *******************************************************************************
 * @brief       ILI9325 Chip Inital.
 * @return      None
 *******************************************************************************
 */
void LCD_ILI9325_Initial(Display_HandleTypeDef *DispX)
{
//************* Start Initial Sequence **********//
    DispX->WriteRegister(0xE5, 0x78F0);       // Set uint16_ternal timing
    DispX->WriteRegister(0x01, 0x0000);       // set SS and SM bit
    DispX->WriteRegister(0x02, 0x0700);       // set 1 line inversion
    DispX->WriteRegister(0x03, 0x1030);       // ColorBar OK, Location OK
    //DispX->WriteRegister(0x03, 0x5030);       // set GRAM write direction and BGR=1.
    DispX->WriteRegister(0x04, 0x0000);       // Resize register
    DispX->WriteRegister(0x08, 0x0202);       // Ver1. set the back porch and front porch
    //DispX->WriteRegister(0x08, 0x0404);       // Ver1. set the back porch and front porch
    DispX->WriteRegister(0x09, 0x0000);
    // DispX->WriteRegister(0x0009, 0x000F);      // set non-display area refresh cycle ISC[3:0]
    DispX->WriteRegister(0x0A, 0x0000);       // FMARK function
    DispX->WriteRegister(0x0C, 0x0000);       // RGB uint16_terface setting
    DispX->WriteRegister(0x0D, 0x0000);       // Frame marker Position
    DispX->WriteRegister(0x0F, 0x0000);       // RGB uint16_terface polarity

//************* power on sequence VGHVGL ****************//
    DispX->WriteRegister(0x10, 0x0000);       // SAP, BT[3:0], AP, DSTB, SLP, STB
    DispX->WriteRegister(0x11, 0x0007);       // DC1[2:0], DC0[2:0], VC[2:0]
    DispX->WriteRegister(0x12, 0x0000);       // VREG1OUT voltage
    DispX->WriteRegister(0x13, 0x0000);       // VDV[4:0] for VCOM amplitude
    DispX->WriteRegister(0x07, 0x0000);       // Dis-charge capacitor power voltage
    //vgh 
    DispX->WriteRegister(0x10, 0x1690);       // SAP, BT[3:0], AP, DSTB, SLP, STB
    DispX->WriteRegister(0x11, 0x0227);       // R11h=0x0221 at VCI=3.3V, DC1[2:0], DC0[2:0], VC[2:0]
    // DispX->WriteRegister(0x11, 0x0037);      // R11h=0x0221 at VCI=3.3V, DC1[2:0], DC0[2:0], VC[2:0]
    //delayms(100);
    //vregiout 
    DispX->WriteRegister(0x12, 0x009D);       // 0x001b
    //DispX->WriteRegister(0x12, 0x013A);       // External reference voltage = Vci;
    //delayms(100); 
    //vom amplitude
    DispX->WriteRegister(0x13,0x1900);
    //DispX->WriteRegister(0x13, 0x1600);       // R13=0F00 when R12=009E;VDV[4:0] for VCOM amplitude  
    //delayms(100); 
    //vom H
    DispX->WriteRegister(0x29,0x0025);
    //DispX->WriteRegister(0x29, 0x000C);       // R29=0019 when R12=009E;VCM[5:0] for VCOMH 
    DispX->WriteRegister(0x2B,0x000D);
    //DispX->WriteRegister(0x20, 0x0000);       // GRAM horizontal Address  
    //DispX->WriteRegister(0x21, 0x0000);       // GRAM Vertical Address 

//************* Adjust the Gamma Curve ****************//
    DispX->WriteRegister(0x30,0x0007);
    DispX->WriteRegister(0x31,0x0303);
    DispX->WriteRegister(0x32,0x0003);// 0006
    DispX->WriteRegister(0x35,0x0206);
    DispX->WriteRegister(0x36,0x0008);
    DispX->WriteRegister(0x37,0x0406); 
    DispX->WriteRegister(0x38,0x0304);//0200
    DispX->WriteRegister(0x39,0x0007); 
    DispX->WriteRegister(0x3C,0x0602);// 0504
    DispX->WriteRegister(0x3D,0x0008); 

//************* Set GRAM area ****************// 
    DispX->WriteRegister(0x50,0x0000);        // Horizontal GRAM Start Address  
    DispX->WriteRegister(0x51,0x00EF);        // Horizontal GRAM End Address 
    DispX->WriteRegister(0x52,0x0000);        // Vertical GRAM Start Address
    DispX->WriteRegister(0x53,0x013F);        // Vertical GRAM Start Address  
    DispX->WriteRegister(0x60,0x2700);        // Gate Scan Line  
    DispX->WriteRegister(0x61,0x0001);        // NDL,VLE, REV  
    DispX->WriteRegister(0x6A,0x0000);        // set scrolling line  

//************* Partial Display Control ****************//
    DispX->WriteRegister(0x80,0x0000); 
    DispX->WriteRegister(0x81,0x0000); 
    DispX->WriteRegister(0x82,0x0000); 
    DispX->WriteRegister(0x83,0x0000); 
    DispX->WriteRegister(0x84,0x0000); 
    DispX->WriteRegister(0x85,0x0000);

//************* Panel Control ****************//
    DispX->WriteRegister(0x90,0x0010); 
    DispX->WriteRegister(0x92,0x0600); 
    
    DispX->WriteRegister(0x07,0x0133);        // 262K color and display ON
    DispX->WriteRegister(0x00,0x0022);          //
}

/**
 *******************************************************************************
 * @brief       ILI9328 Chip Inital.
 * @return      None
 *******************************************************************************
 */
void LCD_ILI9328_Initial(Display_HandleTypeDef *DispX)
{
    DispX->WriteRegister(0x00EC,0x108F);// internal timeing
    DispX->WriteRegister(0x00EF,0x1234);// ADD
    //DispX->WriteRegister(0x00e7,0x0010);
    //DispX->WriteRegister(0x0000,0x0001);//開啟內部時鐘
    DispX->WriteRegister(0x0001,0x0000);
    //DispX->WriteRegister(0x0001,0x0100);
    DispX->WriteRegister(0x0002,0x0700);//電源開啟
    //DispX->WriteRegister(0x0003,(1<<3)|(1<<4) );    //65K  RGB
    //DRIVE TABLE(寄存器 03H)
    //BIT3=AM BIT4:5=ID0:1
    //AM ID0 ID1   FUNCATION
    // 0  0   0     R->L D->U
    // 1  0   0     D->U R->L
    // 0  1   0     L->R D->U
    // 1  1   0     D->U L->R
    // 0  0   1     R->L U->D
    // 1  0   1     U->D R->L
    // 0  1   1     L->R U->D 正常就用這個.
    // 1  1   1     U->D L->R
    DispX->WriteRegister(0x03,(1<<12)|(3<<4)|(0<<3) );//65K
    DispX->WriteRegister(0x04,0x0000);
    DispX->WriteRegister(0x08,0x0202);
    DispX->WriteRegister(0x09,0x0000);
    DispX->WriteRegister(0x0a,0x0000);//display setting
    DispX->WriteRegister(0x0c,0x0001);//display setting
    DispX->WriteRegister(0x0d,0x0000);//0f3c
    DispX->WriteRegister(0x0f,0x0000);
    //電源配置
    DispX->WriteRegister(0x10,0x0000);
    DispX->WriteRegister(0x11,0x0007);
    DispX->WriteRegister(0x12,0x0000);
    DispX->WriteRegister(0x13,0x0000);
    DispX->WriteRegister(0x07,0x0001);
    MID_Delay(50); 
    DispX->WriteRegister(0x10,0x1490);
    DispX->WriteRegister(0x11,0x0227);
    MID_Delay(50); 
    DispX->WriteRegister(0x12,0x008A);
    MID_Delay(50); 
    DispX->WriteRegister(0x13,0x1a00);
    DispX->WriteRegister(0x29,0x0006);
    DispX->WriteRegister(0x2b,0x000d);
    MID_Delay(50); 
    DispX->WriteRegister(0x20,0x0000);
    DispX->WriteRegister(0x21,0x0000);
    MID_Delay(50); 
    //伽馬校正
    DispX->WriteRegister(0x30,0x0000);
    DispX->WriteRegister(0x31,0x0604);
    DispX->WriteRegister(0x32,0x0305);
    DispX->WriteRegister(0x35,0x0000);
    DispX->WriteRegister(0x36,0x0C09);
    DispX->WriteRegister(0x37,0x0204);
    DispX->WriteRegister(0x38,0x0301);
    DispX->WriteRegister(0x39,0x0707);
    DispX->WriteRegister(0x3c,0x0000);
    DispX->WriteRegister(0x3d,0x0a0a);
    MID_Delay(50); 
    DispX->WriteRegister(0x50,0x0000); //水平GRAM起始位置
    DispX->WriteRegister(0x51,0x00ef); //水平GRAM終止位置
    DispX->WriteRegister(0x52,0x0000); //垂直GRAM起始位置
    DispX->WriteRegister(0x53,0x013f); //垂直GRAM終止位置

    DispX->WriteRegister(0x60,0x2700);
    //DispX->WriteRegister(0x60,0xA700);
    DispX->WriteRegister(0x61,0x0001); 
    DispX->WriteRegister(0x6a,0x0000);
    DispX->WriteRegister(0x80,0x0000);
    DispX->WriteRegister(0x81,0x0000);
    DispX->WriteRegister(0x82,0x0000);
    DispX->WriteRegister(0x83,0x0000);
    DispX->WriteRegister(0x84,0x0000);
    DispX->WriteRegister(0x85,0x0000);

    DispX->WriteRegister(0x90,0x0010);
    DispX->WriteRegister(0x92,0x0600);
    //開啟顯示設置
    DispX->WriteRegister(0x07,0x0133);
}

/**
 *******************************************************************************
 * @brief       ILI9331 Chip Inital.
 * @return      None
 *******************************************************************************
 */
void LCD_ILI9331_Initial(Display_HandleTypeDef *DispX)
{
    DispX->WriteRegister(0xE7, 0x1014);
    DispX->WriteRegister(0x01, 0x0100); // set SS and SM bit
    DispX->WriteRegister(0x02, 0x0200); // set 1 line inversion
    DispX->WriteRegister(0x03,(1<<12)|(3<<4)|(1<<3));//65K    
    //DispX->WriteRegister(0x03, 0x1030); // set GRAM write direction and BGR=1.
    DispX->WriteRegister(0x08, 0x0202); // set the back porch and front porch
    DispX->WriteRegister(0x09, 0x0000); // set non-display area refresh cycle ISC[3:0]
    DispX->WriteRegister(0x0A, 0x0000); // FMARK function
    DispX->WriteRegister(0x0C, 0x0000); // RGB interface setting
    DispX->WriteRegister(0x0D, 0x0000); // Frame marker Position
    DispX->WriteRegister(0x0F, 0x0000); // RGB interface polarity
    //*************Power On sequence ****************//
    DispX->WriteRegister(0x10, 0x0000); // SAP, BT[3:0], AP, DSTB, SLP, STB
    DispX->WriteRegister(0x11, 0x0007); // DC1[2:0], DC0[2:0], VC[2:0]
    DispX->WriteRegister(0x12, 0x0000); // VREG1OUT voltage
    DispX->WriteRegister(0x13, 0x0000); // VDV[4:0] for VCOM amplitude
    MID_Delay(200); // Dis-charge capacitor power voltage
    DispX->WriteRegister(0x10, 0x1690); // SAP, BT[3:0], AP, DSTB, SLP, STB
    DispX->WriteRegister(0x11, 0x0227); // DC1[2:0], DC0[2:0], VC[2:0]
    MID_Delay(50); // Delay 50ms
    DispX->WriteRegister(0x12, 0x000C); // Internal reference voltage= Vci;
    MID_Delay(50); // Delay 50ms
    DispX->WriteRegister(0x13, 0x0800); // Set VDV[4:0] for VCOM amplitude
    DispX->WriteRegister(0x29, 0x0011); // Set VCM[5:0] for VCOMH
    DispX->WriteRegister(0x2B, 0x000B); // Set Frame Rate
    MID_Delay(50); // Delay 50ms
    DispX->WriteRegister(0x20, 0x0000); // GRAM horizontal Address
    DispX->WriteRegister(0x21, 0x013f); // GRAM Vertical Address
    // ----------- Adjust the Gamma Curve ----------//
    DispX->WriteRegister(0x30, 0x0000);
    DispX->WriteRegister(0x31, 0x0106);
    DispX->WriteRegister(0x32, 0x0000);
    DispX->WriteRegister(0x35, 0x0204);
    DispX->WriteRegister(0x36, 0x160A);
    DispX->WriteRegister(0x37, 0x0707);
    DispX->WriteRegister(0x38, 0x0106);
    DispX->WriteRegister(0x39, 0x0707);
    DispX->WriteRegister(0x3C, 0x0402);
    DispX->WriteRegister(0x3D, 0x0C0F);
    //------------------ Set GRAM area ---------------//
    DispX->WriteRegister(0x50, 0x0000); // Horizontal GRAM Start Address
    DispX->WriteRegister(0x51, 0x00EF); // Horizontal GRAM End Address
    DispX->WriteRegister(0x52, 0x0000); // Vertical GRAM Start Address
    DispX->WriteRegister(0x53, 0x013F); // Vertical GRAM Start Address
    DispX->WriteRegister(0x60, 0x2700); // Gate Scan Line
    DispX->WriteRegister(0x61, 0x0001); // NDL,VLE, REV 
    DispX->WriteRegister(0x6A, 0x0000); // set scrolling line
    //-------------- Partial Display Control ---------//
    DispX->WriteRegister(0x80, 0x0000);
    DispX->WriteRegister(0x81, 0x0000);
    DispX->WriteRegister(0x82, 0x0000);
    DispX->WriteRegister(0x83, 0x0000);
    DispX->WriteRegister(0x84, 0x0000);
    DispX->WriteRegister(0x85, 0x0000);
    //-------------- Panel Control -------------------//
    DispX->WriteRegister(0x90, 0x0010);
    DispX->WriteRegister(0x92, 0x0600);
    DispX->WriteRegister(0x07, 0x0133); // 262K color and display ON
}

/**
 *******************************************************************************
 * @brief       ILI9341 Chip Inital.
 * @return      None
 *******************************************************************************
 */
void LCD_ILI9341_SPI_SelectRegister(uint8_t Register)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Register));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_ILI9341_SPI_WriteData(uint16_t Data)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Data));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_ILI9341_SPI_Write8BitsData(uint8_t Data)
{
    //=========================================================
    //Prevent unused argument compilcation warning
    ((void)(Data));
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
uint16_t LCD_ILI9341_SPI_ReadData(void)
{
    return 0;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
uint8_t LCD_ILI9341_SPI_Read8BitsData(void)
{
    return 0;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_ILI9341_Initial(Display_HandleTypeDef *DispX)
{
    DispX->SelectReg(0xCF);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0xC1);
    DispX->WriteData8(0X30);

    DispX->SelectReg(0xED);
    DispX->WriteData8(0x64);
    DispX->WriteData8(0x03);
    DispX->WriteData8(0X12);
    DispX->WriteData8(0X81);

    DispX->SelectReg(0xE8);
    DispX->WriteData8(0x85);
    DispX->WriteData8(0x10);
    DispX->WriteData8(0x7A);

    DispX->SelectReg(0xCB);
    DispX->WriteData8(0x39);
    DispX->WriteData8(0x2C);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0x34);
    DispX->WriteData8(0x02);

    DispX->SelectReg(0xF7);
    DispX->WriteData8(0x20);

    DispX->SelectReg(0xEA);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0x00);

    DispX->SelectReg(0xC0);          // Power control
    DispX->WriteData8(0x1B);          // VRH[5:0]

    DispX->SelectReg(0xC1);          // Power control
    DispX->WriteData8(0x01);          // SAP[2:0];BT[3:0]

    DispX->SelectReg(0xC5);          // VCM control
    DispX->WriteData8(0x30);          // 3F
    DispX->WriteData8(0x30);          // 3C

    DispX->SelectReg(0xC7);          // VCM control2
    DispX->WriteData8(0XB7);

    DispX->SelectReg(0x36);          // Memory Access Control
    DispX->WriteData8(0x48);

    DispX->SelectReg(0x3A);
    DispX->WriteData8(0x55);

    DispX->SelectReg(0xB1);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0x1A);

    DispX->SelectReg(0xB6);      // Display Function Control
    DispX->WriteData8(0x0A);
    if(DispX->HWIF == 0)
    {
//        if((EMB->CR1.W & EMB_CR1_MAD_SWAP_enable_w) == 0)
//            DispX->WriteData8(0xC2);
//        else
            DispX->WriteData8(0xC2);
    }
    if(DispX->HWIF != 0)
        DispX->WriteData8(0xE2);

    DispX->SelectReg(0xF2);          // 3Gamma Function Disable
    DispX->WriteData8(0x00);

    DispX->SelectReg(0x26);          // Gamma curve selected
    DispX->WriteData8(0x01);

    DispX->SelectReg(0xE0);          // Set Gamma
    DispX->WriteData8(0x0F);
    DispX->WriteData8(0x2A);
    DispX->WriteData8(0x28);
    DispX->WriteData8(0x08);
    DispX->WriteData8(0x0E);
    DispX->WriteData8(0x08);
    DispX->WriteData8(0x54);
    DispX->WriteData8(0XA9);
    DispX->WriteData8(0x43);
    DispX->WriteData8(0x0A);
    DispX->WriteData8(0x0F);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0x00);

    DispX->SelectReg(0XE1);          // Set Gamma
    DispX->WriteData8(0x00);
    DispX->WriteData8(0x15);
    DispX->WriteData8(0x17);
    DispX->WriteData8(0x07);
    DispX->WriteData8(0x11);
    DispX->WriteData8(0x06);
    DispX->WriteData8(0x2B);
    DispX->WriteData8(0x56);
    DispX->WriteData8(0x3C);
    DispX->WriteData8(0x05);
    DispX->WriteData8(0x10);
    DispX->WriteData8(0x0F);
    DispX->WriteData8(0x3F);
    DispX->WriteData8(0x3F);
    DispX->WriteData8(0x0F);

    DispX->SelectReg(0x2B);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0x01);
    DispX->WriteData8(0x3f);

    DispX->SelectReg(0x2A);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0x00);
    DispX->WriteData8(0xef);

    DispX->SelectReg(0x11);          // Exit Sleep
    MID_Delay(10);
    DispX->SelectReg(0x29);          // display on
}

/**
 *******************************************************************************
 * @brief       Set LCD automatic scan direction 設置LCD的自動掃描方向
 * @param       Dir 0~7,代表8個方向(具體定義見lcd.h)
 *      @arg        L2R_U2D : From left to right, top to bottom.
 *      @arg        L2R_D2U : From left to right, bottom to top.
 *      @arg        R2L_U2D : From right to left, top to bottom.
 *      @arg        R2L_D2U : From right to left, bottom to top.
 *      @arg        U2D_L2R : From top to bottom, left to right.
 *      @arg        U2D_R2L : From top to bottom, right to left.
 *      @arg        D2U_L2R : From bottom to top, left to right.
 *      @arg        D2U_R2L : From bottom to top, right to left.
 * @return      None
 * @note        其他函數可能會受到此函數設置的影響(尤其是9341/6804這兩個奇葩),
 *              一般設置為L2R_U2D即可,如果設置為其他掃描方式,可能導致顯示不正常.
 *              9320/9325/9328/4531/4535/1505/b505/8989/5408/9341/5310/5510等IC已經實際測試
 *******************************************************************************
 */
void LCD_Scan_Dir(Display_HandleTypeDef *DispX, uint8_t dir)
{
    uint16_t regval=0;
    uint16_t dirreg=0;
    uint16_t temp;  
    if((DispX->Direct == 1) && (DispX->ID != 0X6804))//橫屏時，對6804不改變掃描方向！
    {
        switch(dir)//方向轉換
        {
            case 0: dir = 6;break;
            case 1: dir = 7;break;
            case 2: dir = 4;break;
            case 3: dir = 5;break;
            case 4: dir = 1;break;
            case 5: dir = 0;break;
            case 6: dir = 3;break;
            case 7: dir = 2;break;
        }
    }
    if(DispX->ID==0x9341||DispX->ID==0X6804||DispX->ID==0X5310||DispX->ID==0X5510)//9341/6804/5310/5510,很特殊
    {
        switch(dir)
        {
            case L2R_U2D://從左到右,從上到下
                regval|=(0<<7)|(0<<6)|(0<<5);
                break;
            case L2R_D2U://從左到右,從下到上
                regval|=(1<<7)|(0<<6)|(0<<5);
                break;
            case R2L_U2D://從右到左,從上到下
                regval|=(0<<7)|(1<<6)|(0<<5);
                break;
            case R2L_D2U://從右到左,從下到上
                regval|=(1<<7)|(1<<6)|(0<<5);
                break;
            case U2D_L2R://從上到下,從左到右
                regval|=(0<<7)|(0<<6)|(1<<5);
                break;
            case U2D_R2L://從上到下,從右到左
                regval|=(0<<7)|(1<<6)|(1<<5);
                break;
            case D2U_L2R://從下到上,從左到右
                regval|=(1<<7)|(0<<6)|(1<<5);
                break;
            case D2U_R2L://從下到上,從右到左
                regval|=(1<<7)|(1<<6)|(1<<5);
                break;
        }

        if(DispX->ID == 0X5510) dirreg = 0X3600;
        else dirreg = 0X36;

        if((DispX->ID != 0X5310) && (DispX->ID != 0X5510))regval |= 0X08;//5310/5510不需要BGR

        if(DispX->ID == 0X6804)regval |= 0x02;//6804的BIT6和9341的反了

        DispX->WriteRegister(dirreg, regval);

        if(regval & 0X20)
        {
            if(DispX->Width < DispX->Height)//交換X,Y
            {
                temp = DispX->Width;
                DispX->Width = DispX->Height;
                DispX->Height = temp;
            }
        }else
        {
            if(DispX->Width > DispX->Height)//交換X,Y
            {
                temp = DispX->Width;
                DispX->Width = DispX->Height;
                DispX->Height = temp;
            }
        }

        if(DispX->ID == 0X5510)
        {
            DispX->SelectReg(DispX->SetXcmd);
            DispX->WriteData(0);
            DispX->SelectReg(DispX->SetXcmd + 1);
            DispX->WriteData(0);
            DispX->SelectReg(DispX->SetXcmd + 2);
            DispX->WriteData((DispX->Width - 1) >> 8);
            DispX->SelectReg(DispX->SetXcmd + 3);
            DispX->WriteData((DispX->Width - 1) & 0XFF);
            DispX->SelectReg(DispX->SetYcmd);
            DispX->WriteData(0);
            DispX->SelectReg(DispX->SetYcmd + 1);
            DispX->WriteData(0);
            DispX->SelectReg(DispX->SetYcmd + 2);
            DispX->WriteData((DispX->Height-1)>>8);
            DispX->SelectReg(DispX->SetYcmd + 3);
            DispX->WriteData((DispX->Height-1)&0XFF);
        }else
        {
            DispX->SelectReg(DispX->SetXcmd);
            DispX->WriteData8(0);
            DispX->WriteData8(0);
            DispX->WriteData8((DispX->Width-1)>>8);
            DispX->WriteData8((DispX->Width-1)&0XFF);

            DispX->SelectReg(DispX->SetYcmd);
            DispX->WriteData8(0);
            DispX->WriteData8(0);
            DispX->WriteData8((DispX->Height-1)>>8);
            DispX->WriteData8((DispX->Height-1)&0XFF);
        }
    }else 
    {
        switch(dir)
        {
            case L2R_U2D://從左到右,從上到下
                regval|=(1<<5)|(1<<4)|(0<<3);
                break;
            case L2R_D2U://從左到右,從下到上
                regval|=(0<<5)|(1<<4)|(0<<3);
                break;
            case R2L_U2D://從右到左,從上到下
                regval|=(1<<5)|(0<<4)|(0<<3);
                break;
            case R2L_D2U://從右到左,從下到上
                regval|=(0<<5)|(0<<4)|(0<<3);
                break;
            case U2D_L2R://從上到下,從左到右
                regval|=(1<<5)|(1<<4)|(1<<3);
                break;
            case U2D_R2L://從上到下,從右到左
                regval|=(1<<5)|(0<<4)|(1<<3);
                break;
            case D2U_L2R://從下到上,從左到右
                regval|=(0<<5)|(1<<4)|(1<<3);
                break;
            case D2U_R2L://從下到上,從右到左
                regval |= (0 << 5) | (0 << 4) | (1 << 3 );
                break;
        }
        if(DispX->ID==0x8989)//8989 IC
        {
            dirreg = 0X11;
            regval |= 0X6040;   //65K
        }else//其他驅動IC
        {
            dirreg=0X03;
            regval |= 1 << 12;
        }
        DispX->WriteRegister(dirreg, regval);
    }
}

/**
 *******************************************************************************
 * @brief       Set LCD display direction
 * @param       Dir
*      @arg        0: Vertical screen 豎屏
*      @arg        1: Horizontal screen 橫屏
 * @return      None
 *******************************************************************************
 */
 void LCD_Display_Dir(Display_HandleTypeDef *DispX, uint8_t dir)
{
    if(dir == 0)                    //豎屏
    {
        DispX->Direct = 0;       //豎屏
        DispX->Width = 240;
        DispX->Height = 320;

        if(DispX->ID == 0X9341 || DispX->ID == 0X6804 || DispX->ID == 0X5310)
        {
            DispX->GRAMcmd = 0X2C;
            DispX->SetXcmd = 0X2A;
            DispX->SetYcmd = 0X2B;
            if(DispX->ID == 0X6804 || DispX->ID == 0X5310)
            {
                DispX->Width = 320;
                DispX->Height = 480;
            }
        }else if(DispX->ID == 0X8989)
        {
            DispX->GRAMcmd = R34;
            DispX->SetXcmd = 0X4E;
            DispX->SetYcmd = 0X4F;
        }else if(DispX->ID == 0x5510)
        {
            DispX->GRAMcmd = 0X2C00;
            DispX->SetXcmd = 0X2A00;
            DispX->SetYcmd = 0X2B00;
            DispX->Width = 480;
            DispX->Height = 800;
        }else
        {
            DispX->GRAMcmd = R34;
            DispX->SetXcmd = R32;
            DispX->SetYcmd = R33;
        }
    }else                           //橫屏
    {
        DispX->Direct = 1;       //橫屏
        DispX->Width = 320;
        DispX->Height = 240;
        if(DispX->ID == 0X9341 || DispX->ID == 0X5310)
        {
            DispX->GRAMcmd = 0X2C;
            DispX->SetXcmd = 0X2A;
            DispX->SetYcmd = 0X2B;
        }else if(DispX->ID == 0X6804)
        {
            DispX->GRAMcmd = 0X2C;
            DispX->SetXcmd = 0X2B;
            DispX->SetYcmd = 0X2A; 
        }else if(DispX->ID == 0X8989)
        {
            DispX->GRAMcmd = R34;
            DispX->SetXcmd = 0X4F;
            DispX->SetYcmd = 0X4E;
        }else if(DispX->ID == 0x5510)
        {
            DispX->GRAMcmd = 0X2C00;
            DispX->SetXcmd = 0X2A00;
            DispX->SetYcmd = 0X2B00; 
            DispX->Width = 800;
            DispX->Height = 480;
        }else
        {
            DispX->GRAMcmd = R34;
            DispX->SetXcmd = R33;
            DispX->SetYcmd = R32;
        }
        if(DispX->ID == 0X6804 || DispX->ID == 0X5310)
        {
            DispX->Width = 480;
            DispX->Height = 320;
        }
    }
    LCD_Scan_Dir(DispX, DFT_SCAN_DIR);     //默認掃描方向
}

/**
 *******************************************************************************
 * @brief       LCD Initial 初始化lcd
 * @return      None
 * @note        本函數占用較大flash,用戶可以根據自己的實際情況,刪掉未用到的LCD初始化代碼.以節省空間.
 *******************************************************************************
 */
void LCD_Init(Display_HandleTypeDef *DispX)
{
    DispX->Reset(0);
  #if LCDPanel_Type == 0
    DispX->ID = 0x9341;
    LCD_ILI9341_Initial(DispX);

  #elif LCDPanel_Type == 1
    DispX->ID = 0x9328;
    LCD_ILI9328_Initial(DispX);

  #elif LCDPanel_Type == 2
    DispX->ID = 0x9325;
    LCD_ILI9325_Initial(DispX);

  #elif LCDPanel_Type == 3
    DispX->ID = 0x9320;
    LCD_ILI9320_Initial(DispX);

  #elif LCDPanel_Type == 4
    DispX->ID = 0x5510;
    LCD_NT35510_Initial(DispX);

  #elif LCDPanel_Type == 5
  #elif LCDPanel_Type == 6
  #elif LCDPanel_Type == 7
  #elif LCDPanel_Type == 8
  #elif LCDPanel_Type == 9
    __IO uint16_t lTemp;
    DispX->ID = 0x0000;
    DispX->DataReverse(DISABLE);
    //MID_Delay(50); // delay 50 ms

    DispX->WriteRegister(0x0000, 0x0001);
    //DispX->WriteRegister(0x0001,0x0001);
    
    if(DispX->HWIF == 0)
    {
        if(DispX->ID == 0)
        {
            DispX->SelectReg(0x0000);
            lTemp = DispX->ReadData();
            
            switch (lTemp)
            {
                case 0x04C9:
                     DispX->DataReverse(ENABLE);
                     goto LCD_Init_0x9320;
                case 0x9320:
LCD_Init_0x9320:
                     DispX->ID = 0x9320;
                     LCD_ILI9320_Initial(DispX);
                     break;
    
                case 0xA4C9:
                     DispX->DataReverse(ENABLE);
                     goto LCD_Init_0x9325;
                case 0x9325:
LCD_Init_0x9325:                    
                     DispX->ID = 0x9325;
                     LCD_ILI9325_Initial(DispX);
                     break;
    
                case 0x14C9:
                     DispX->DataReverse(ENABLE);
                     goto LCD_Init_0x9328;
                case 0x9328:
LCD_Init_0x9328:
                     DispX->ID = 0x9328;
                     LCD_ILI9328_Initial(DispX);
                     break;
            }
        }
    
        if(DispX->ID == 0)
        {
            DispX->SelectReg(0xD3);                          //寫地址
            DispX->ReadData();                               //dummy read 
            DispX->ReadData();                               //Read 0x00
            if(((DispX->ReadData() << 8) | DispX->ReadData()) == 0x9341)
            {
                DispX->ID = 0x9341;
                LCD_ILI9341_Initial(DispX);
            }
        }
    
        if(DispX->ID == 0)
        {
            DispX->DataReverse(ENABLE);
            DispX->SelectReg(0xD3);                          //寫地址
            DispX->ReadData();                               //dummy read 
            DispX->ReadData();                               //Read 0x00
            lTemp = DispX->ReadData();
            lTemp <<= 8;
            lTemp |= DispX->ReadData();
            if(lTemp == 0x9341)
            {
                DispX->ID = 0x9341;
                LCD_ILI9341_Initial(DispX);
            }
            else
                DispX->DataReverse(DISABLE);
        }
    
        if(DispX->ID == 0)
        {
            DispX->SelectReg(0XDA00);
            DispX->ReadData();                       // dummy read 讀回0X00
            DispX->SelectReg(0XDB00);
            lTemp = DispX->ReadData();               //讀回0X80
            lTemp <<= 8;
            DispX->SelectReg(0XDC00);
            lTemp |= DispX->ReadData();              //讀回0X00
            if(lTemp == 0x8000)
            {
                DispX->ID = 0x5510;                  //NT35510讀回的ID是8000H,為方便區分,我們強制設置為5510
                LCD_NT35510_Initial(DispX);
            }
        }
    
        if(DispX->ID == 0)
        {
            DispX->DataReverse(ENABLE);
            DispX->SelectReg(0XDA00);
            DispX->ReadData();                       // dummy read 讀回0X00
            DispX->SelectReg(0XDB00);
            lTemp = DispX->ReadData();               //讀回0X80
            lTemp <<= 8;
            DispX->SelectReg(0XDC00);
            lTemp |= DispX->ReadData();              //讀回0X00
            if(lTemp == 0x8000)
            {
                DispX->ID = 0x5510;                  //NT35510讀回的ID是8000H,為方便區分,我們強制設置為5510
                LCD_NT35510_Initial(DispX);
            }
            else
                DispX->DataReverse(DISABLE);
        }
    }
    else
    {
        //DispX->SelectReg(0xD3);                          //寫地址
        //DispX->ReadData();                               //dummy read 
        //DispX->ReadData();                               //Read 0x00
        //if(((DispX->ReadData() << 8) | DispX->ReadData()) == 0x9341)
        //{
            DispX->ID = 0x9341;
            LCD_ILI9341_Initial(DispX);
        //}
    }
    //printf("  LCD ID:%x\r\n",DispX->ID); //打印LCD ID
  #endif

    LCD_Display_Dir(DispX, 0);                   //默認為豎屏
    DispX->BackLight(0x0000);
    DispX->BackLight(0xFFFF);                    //點亮背光
    LCD_Clear(DispX, WHITE);
    LCD_Clear(DispX, BLACK);
//    LCD_Clear(DispX, BLUE);
//    LCD_Clear(DispX, RED);
//    LCD_Clear(DispX, GREEN);
//    LCD_Clear(DispX, BLACK);
}

/**
 *******************************************************************************
 * @brief       Color format BGR comver to RGB
 *              從ILI93xx讀出的數據為GBR格式，而我們寫入的時候為RGB格式。
 * @param       BGRData BGR format Data Value BGR格式的顏色值
 * @return      uint16_t RGB格式的顏色值
 *******************************************************************************
 */
uint16_t LCD_BGR2RGB(uint16_t c)
{
    uint16_t  r,g,b,rgb;
    
    b   = (c>>0)&0x1f;
    g   = (c>>5)&0x3f;
    r   = (c>>11)&0x1f;
    rgb = ((uint16_t)(b<<11))+((uint16_t)(g<<5))+((uint16_t)(r<<0));
    return(rgb);
}

/**
 *******************************************************************************
 * @brief       Read Point Color Value 讀取個某點的顏色值
 * @param       x X-axes 坐標
 * @param       y Y-axes 坐標
 * @return      Color Value 此點的顏色
 *******************************************************************************
 */
uint16_t LCD_ReadPoint(Display_HandleTypeDef *DispX, uint16_t x, uint16_t y)
{
    __IO uint16_t r;
    __IO uint16_t g = 0;
    __IO uint16_t b = 0;

    if(x >= DispX->Width || y >= DispX->Height)
        return 0;  //超過了範圍,直接返回
    LCD_SetCursor(DispX, x, y);

    if(DispX->ID == 0X9341 || DispX->ID == 0X6804 || DispX->ID == 0X5310)
        DispX->SelectReg(0X2E);                  //9341/6804/5310發送讀GRAM指令
    else if(DispX->ID == 0X5510)
        DispX->SelectReg(0X2E00);                //5510 發送讀GRAM指令
    else
        DispX->SelectReg(R34);                   //其他IC發送讀GRAM指令

    //讀取數據(讀GRAM時,第一次為假讀)
    //delay_us(1);//延時1us
    DispX->ReadData();           //dummy READ

    //delay_us(1);//延時1us
    r = DispX->ReadData();       //實際坐標顏色

    if(DispX->ID == 0X9341 || DispX->ID == 0X5310 || DispX->ID == 0X5510) //9341/NT35310/NT35510要分2次讀出
    {
        b = DispX->ReadData();//讀取藍色值
        g = r & 0XFF;//對於9341,第一次讀取的是RG的值,R在前,G在後,各占8位
        g <<= 8;
    }else if(DispX->ID == 0X6804)
    {
        r = DispX->ReadData();   //6804第二次讀取的才是真實值 
    }

    if(DispX->ID == 0X9325 || DispX->ID == 0X4535 || DispX->ID == 0X4531 || DispX->ID == 0X8989 || DispX->ID == 0XB505)
        return r;  //這幾種IC直接返回顏色值
    else if(DispX->ID == 0X9341 || DispX->ID == 0X5310 || DispX->ID == 0X5510)
        return (((uint16_t)((r>>11)<<11))|((uint16_t)((g>>10)<<5))|((uint16_t)(b>>11)));//ILI9341/NT35310/NT35510需要公式轉換一下
    else return 
        LCD_BGR2RGB(r);     //其他IC
}

/**
 *******************************************************************************
 * @brief       LCD Display ON LCD開啟顯示
 * @return      None
 *******************************************************************************
 */
void LCD_DisplayOn(Display_HandleTypeDef *DispX)
{
    if(DispX->ID == 0X9341 || DispX->ID == 0X6804 || DispX->ID == 0X5310)
        DispX->SelectReg(0X29);              //開啟顯示
    else if(DispX->ID == 0X5510)
        DispX->SelectReg(0X2900);            //開啟顯示
    else 
        DispX->WriteRegister(R7, 0x0173);         //開啟顯示
}

/**
 *******************************************************************************
 * @brief       LCD Display OFF LCD關閉顯示
 * @return      None
 *******************************************************************************
 */
void LCD_DisplayOff(Display_HandleTypeDef *DispX)
{
    if(DispX->ID == 0X9341 || DispX->ID == 0X6804 || DispX->ID == 0X5310)
        DispX->SelectReg(0X28);              //關閉顯示
    else if(DispX->ID == 0X5510)
        DispX->SelectReg(0X2800);            //關閉顯示
    else
        DispX->WriteRegister(R7, 0x0);            //關閉顯示 
}

/**
 *******************************************************************************
 * @brief       Set Cursor 設置光標位置
 * @param       Xpos X-axes 坐標
 * @param       Ypos Y-axes 坐標
 * @return      None
 *******************************************************************************
 */
void LCD_SetCursor(Display_HandleTypeDef *DispX, uint16_t Xpos, uint16_t Ypos)
{
    if((DispX->ID == 0X9341) || (DispX->ID == 0X5310))
    {
        DispX->SelectReg(DispX->SetXcmd);
        DispX->WriteData8(Xpos>>8);
        DispX->WriteData8(Xpos&0XFF);

        DispX->SelectReg(DispX->SetYcmd);
        DispX->WriteData8(Ypos>>8);
        DispX->WriteData8(Ypos&0XFF);
    }else if(DispX->ID == 0X6804)
    {
        if(DispX->Direct != 0)
            Xpos = DispX->Width - 1 - Xpos;//橫屏時處理

        DispX->SelectReg(DispX->SetXcmd);
        DispX->WriteData(Xpos >> 8);
        DispX->WriteData(Xpos & 0XFF);

        DispX->SelectReg(DispX->SetYcmd);
        DispX->WriteData(Ypos >> 8);
        DispX->WriteData(Ypos & 0XFF);
    }else if(DispX->ID == 0X5510)
    {
        DispX->SelectReg(DispX->SetXcmd);
        DispX->WriteData(Xpos >> 8);

        DispX->SelectReg(DispX->SetXcmd + 1);
        DispX->WriteData(Xpos & 0XFF);

        DispX->SelectReg(DispX->SetYcmd);
        DispX->WriteData(Ypos >> 8);

        DispX->SelectReg(DispX->SetYcmd + 1);
        DispX->WriteData(Ypos & 0XFF);
    }else
    {
        if(DispX->Direct != 0)
            Xpos = DispX->Width - 1 - Xpos;//橫屏其實就是調轉x,y坐標

        //DispX->WriteRegister(lcddev.setxcmd, Xpos);
        DispX->SelectReg(DispX->SetXcmd);
        DispX->WriteData(Xpos);

        //DispX->WriteRegister(lcddev.setycmd, Ypos);
        DispX->SelectReg(DispX->SetYcmd);
        DispX->WriteData(Ypos);
    }
}

/**
 *******************************************************************************
 * @brief       Clear 清屏函數
 * @param       Color 要清屏的填充色
 * @return      None
 *******************************************************************************
 */
void LCD_Clear2(Display_HandleTypeDef *DispX, uint16_t sx, uint16_t sy, uint16_t width, uint16_t height, uint16_t color)
{
    __IO uint32_t lCount = 0;
    __IO uint32_t totalpoint;

    totalpoint = width;
    totalpoint *= height;                               //敺蝮賡???

    LCD_Set_Window(DispX, sx, sy, width, height);
    DispX->SelectReg(DispX->GRAMcmd);                   //??撖怠GRAM

    for(lCount = 0; lCount < totalpoint; lCount++)
    {
        DispX->WriteData(color);
    }
    LCD_Set_Window(DispX, 0, 0, DispX->Width, DispX->Height);
}

static uint16_t gFputcX, gFputcY;
/**
 *******************************************************************************
 * @brief       Clear 清屏函数
 * @param       Color 要清屏的颜色
 * @return      None
 *******************************************************************************
 */
void LCD_Clear(Display_HandleTypeDef *DispX, uint16_t color)
{
    __IO uint32_t index = 0;
    __IO uint32_t totalpoint;

    gFputcX = gFputcY = 0;
    totalpoint = DispX->Width;
    totalpoint *= DispX->Height;                          //得到總點數
    if((DispX->ID == 0X6804)&&(DispX->Direct != 0))       //6804橫屏的時候特殊處理  
    {
        DispX->Direct = 0;
        DispX->SetXcmd = 0X2A;
        DispX->SetYcmd = 0X2B;
        LCD_SetCursor(DispX, 0x00,0x0000);                //設置光標位置  
        DispX->Direct = 1;
        DispX->SetXcmd = 0X2B;
        DispX->SetYcmd = 0X2A;
    }else LCD_SetCursor(DispX, 0x00, 0x0000);             //設置光標位置 
    DispX->SelectReg(DispX->GRAMcmd);                     //開始寫入GRAM

    for(index = 0; index < totalpoint; index++)
    {
        DispX->WriteData(color);
    }
}

/**
 *******************************************************************************
 * @brief       Set Windows 設置窗口,並自動設置畫點坐標到窗口左上角(sx,sy).
 * @param       sx windows Start X-axes 窗口起始坐標(左上角)
 * @param       sy windows Start Y-axes 窗口起始坐標(左上角)
 * @param       width windows width 窗體大小
 * @param       height windows height 窗體大小
 * @return      None
 * @note        68042,橫屏時不支持窗口設置!! 
 *******************************************************************************
 */
void LCD_Set_Window(Display_HandleTypeDef *DispX, uint16_t sx, uint16_t sy, uint16_t width, uint16_t height)
{
    uint8_t hsareg, heareg, vsareg, veareg;
    uint16_t hsaval, heaval, vsaval, veaval;
    width = sx + width - 1;
    height = sy + height - 1;
    if((DispX->ID == 0X9341) || (DispX->ID == 0X5310) || (DispX->ID == 0X6804))//6804橫屏不支持
    {
        DispX->SelectReg(DispX -> SetXcmd);
        DispX->WriteData(sx >> 8);
        DispX->WriteData(sx & 0XFF);
        DispX->WriteData(width >> 8);
        DispX->WriteData(width & 0XFF);

        DispX->SelectReg(DispX -> SetYcmd);
        DispX->WriteData(sy >> 8);
        DispX->WriteData(sy & 0XFF);
        DispX->WriteData(height >> 8);
        DispX->WriteData(height & 0XFF);
    }else if(DispX->ID == 0X5510)
    {
        DispX->SelectReg(DispX->SetXcmd);
        DispX->WriteData(sx >> 8);

        DispX->SelectReg(DispX->SetXcmd + 1);
        DispX->WriteData(sx & 0XFF);

        DispX->SelectReg(DispX->SetXcmd + 2);
        DispX->WriteData(width >> 8);

        DispX->SelectReg(DispX->SetXcmd + 3);
        DispX->WriteData(width & 0XFF);

        DispX->SelectReg(DispX->SetYcmd);
        DispX->WriteData(sy >> 8);

        DispX->SelectReg(DispX->SetYcmd + 1);
        DispX->WriteData(sy & 0XFF);

        DispX->SelectReg(DispX->SetYcmd + 2);
        DispX->WriteData(height >> 8);

        DispX->SelectReg(DispX->SetYcmd + 3);
        DispX->WriteData(height & 0XFF);
    }else   //其他驅動IC
    {
        if(DispX->Direct == 1)//橫屏
        {
            //窗口值
            hsaval = sy;
            heaval = height;
            vsaval = DispX->Width - width - 1;
            veaval = DispX->Width - sx - 1;
        }else
        { 
            hsaval = sx;
            heaval = width;
            vsaval = sy;
            veaval = height;
        }
        if(DispX->ID == 0X8989)//8989 IC
        {
            hsareg = 0X44; heareg = 0X44;//水平方向窗口寄存器 (1289的由一個寄存器控制)
            hsaval |=( heaval << 8);//得到寄存器值.
            heaval = hsaval;
            vsareg = 0X45; veareg = 0X46;//垂直方向窗口寄存器
        }else  //其他驅動IC
        {
            hsareg = 0X50; heareg = 0X51;//水平方向窗口寄存器
            vsareg = 0X52; veareg = 0X53;//垂直方向窗口寄存器
        }
        //設置寄存器值
        DispX->WriteRegister(hsareg, hsaval);
        DispX->WriteRegister(heareg, heaval);
        DispX->WriteRegister(vsareg, vsaval);
        DispX->WriteRegister(veareg, veaval);
        LCD_SetCursor(DispX, sx, sy);//設置光標位置
    }
}

///**
// *******************************************************************************
// * @brief       Draw Point on Display 畫點
// * @param       x X-axes 坐標
// * @param       y Y-axes 坐標
// * @param       Color Value 此點的顏色
// * @return      None
// *******************************************************************************
// */
//void LCD_DrawPoint(Display_HandleTypeDef *DispX, uint16_t x,uint16_t y, uint16_t Color)
//{
//    LCD_SetCursor(DispX, x,y);                        //設置光標位置
//    DispX->SelectReg(DispX->GRAMcmd);                 //開始寫入GRAM
//    DispX->WriteData(Color); 
//}

/**
 *******************************************************************************
 * @brief       Fast Draw Point on Display 畫點
 * @param       x X-axes 坐標
 * @param       y Y-axes 坐標
 * @param       Color Value 此點的顏色
 * @return      None
 *******************************************************************************
 */
void LCD_DrawPoint(Display_HandleTypeDef *DispX, uint16_t x, uint16_t y, uint16_t color)
{
    if((y >= DispX->Height) || (x >= DispX->Width))
        return;

    if((DispX->ID == 0X9341)|| (DispX->ID == 0X5310))
    {
        DispX->SelectReg(DispX->SetXcmd);
        DispX->WriteData8(x >> 8);
        DispX->WriteData8(x & 0XFF);
        DispX->SelectReg(DispX->SetYcmd);
        DispX->WriteData8(y >> 8);
        DispX->WriteData8(y & 0XFF);
    }else if(DispX->ID == 0X5510)
    {
        DispX->SelectReg(DispX->SetXcmd);
        DispX->WriteData(x >> 8);
        DispX->SelectReg(DispX->SetXcmd + 1);
        DispX->WriteData(x & 0XFF);
        DispX->SelectReg(DispX->SetYcmd);
        DispX->WriteData(y >> 8);
        DispX->SelectReg(DispX->SetYcmd + 1);
        DispX->WriteData(y & 0XFF);
    }else if(DispX->ID == 0X6804)
    {
        if(DispX->Direct == 1)
            x = DispX->Width - 1 - x;                   //橫屏時處理

        DispX->SelectReg(DispX->SetXcmd);
        DispX->WriteData(x >> 8);
        DispX->WriteData(x & 0XFF);
        DispX->SelectReg(DispX->SetYcmd);
        DispX->WriteData(y >> 8);
        DispX->WriteData(y & 0XFF);
    }else
    {
        if(DispX->Direct == 1)
            x = DispX->Width - 1 - x;                   //橫屏其實就是調轉x,y坐標

        DispX->WriteRegister(DispX->SetXcmd, x);
        DispX->WriteRegister(DispX->SetYcmd, y);
    }
    DispX->SelectReg(DispX->GRAMcmd);                   //寫指令 選至GRAM
    DispX->WriteData(color);                            //寫數據
}

/**
 *******************************************************************************
 * @brief       Draw Line 畫線
 * @param       x1 Start X-axes 坐標
 * @param       y1 Start Y-axes 坐標
 * @param       x2 end X-axes 坐標
 * @param       y2 end Y-axes 坐標
 * @param       color : Line Color 要填充的顏色
 * @return      None
 * @note        (sx,sy),(ex,ey):填充矩形對角坐標,區域大小為:(ex-sx+1)*(ey-sy+1)
 *******************************************************************************
 */
void LCD_DrawLine(Display_HandleTypeDef *DispX, uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t Color)
{
    uint16_t t; 
    int xerr = 0, yerr = 0, delta_x, delta_y, distance; 
    int incx, incy, uRow, uCol; 

    delta_x = x2 - x1;                              //計算坐標增量
    delta_y = y2 - y1; 
    uRow = x1; 
    uCol = y1; 
    if(delta_x > 0)incx=1;                          //設置單步方向
    else if(delta_x==0)incx=0;                      //垂直線
    else {incx=-1;delta_x=-delta_x;} 
    if(delta_y>0)incy=1; 
    else if(delta_y==0)incy=0;                      //水平線
    else{incy = -1; delta_y = -delta_y;} 
    if(delta_x > delta_y) distance = delta_x;       //選取基本增量坐標軸
    else distance = delta_y; 
    for(t = 0; t <= distance + 1; t++ )             //?怎?頛詨
    {  
        LCD_DrawPoint(DispX, (uint16_t)uRow, (uint16_t)uCol, (uint16_t)Color);    //畫點
        xerr += delta_x ; 
        yerr += delta_y ; 
        if(xerr > distance) 
        { 
            xerr -= distance; 
            uRow += incx; 
        } 
        if(yerr > distance) 
        { 
            yerr -= distance; 
            uCol += incy; 
        } 
    }  
}

/**
 *******************************************************************************
 * @brief       Draw Rectangle 畫矩形
 * @param       x1 Start X-axes 坐標
 * @param       y1 Start Y-axes 坐標
 * @param       x2 end X-axes 坐標
 * @param       y2 end Y-axes 坐標
 * @param       color : Rectangle Color
 * @return      None
 * @note        (sx,sy),(ex,ey):矩形對角坐標
 *******************************************************************************
 */
void LCD_DrawRectangle(Display_HandleTypeDef *DispX, uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t Color)
{
    LCD_DrawLine(DispX, x1, y1, x2, y1, Color);
    LCD_DrawLine(DispX, x1, y1, x1, y2, Color);
    LCD_DrawLine(DispX, x1, y2, x2, y2, Color);
    LCD_DrawLine(DispX, x2, y1, x2, y2, Color);
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_ButtonUp(Display_HandleTypeDef *DispX, uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t Color)
{
    LCD_DrawRectangle(DispX, (x1 + 3), (y1 + 0), (x2 - 3), (y2 - 0), Color);
    LCD_DrawRectangle(DispX, (x1 + 2), (y1 + 1), (x2 - 2), (y2 - 1), Color);
    LCD_DrawRectangle(DispX, (x1 + 1), (y1 + 2), (x2 - 1), (y2 - 2), Color);
    LCD_DrawRectangle(DispX, (x1 + 0), (y1 + 3), (x2 - 0), (y2 - 3), Color);
    //LCD_Fill(DispX, (x1 + 4), (y1 + 4), (x2 - 4), (y2 - 4), Color);
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_ButtonDone(Display_HandleTypeDef *DispX, uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t Color)
{
    LCD_DrawRectangle(DispX, (x1 + 3), (y1 + 0), (x2 - 3), (y2 - 0), Color);
    LCD_DrawRectangle(DispX, (x1 + 2), (y1 + 1), (x2 - 2), (y2 - 1), Color);
    LCD_DrawRectangle(DispX, (x1 + 1), (y1 + 2), (x2 - 1), (y2 - 2), Color);
    LCD_DrawRectangle(DispX, (x1 + 0), (y1 + 3), (x2 - 0), (y2 - 3), Color);
    //LCD_Fill(DispX, (x1 + 4), (y1 - 4), (x2 - 4), (y2 - 4), Color);
}

//在指定區域內填充指定顏色
//區域大小:(xend-xsta+1)*(yend-ysta+1)
//xsta
//color:要填充的顏色
void LCD_Fill(Display_HandleTypeDef *DispX, uint16_t sx, uint16_t sy, uint16_t ex, uint16_t ey, uint16_t color)
{
    uint16_t i, j;
    uint16_t xlen = 0;
    uint16_t temp;
    if((DispX->ID == 0X6804)&&(DispX->Direct == 1))    //6804橫屏的時候特殊處理  
    {
        temp = sx;
        sx = sy;
        sy = DispX->Width - ex - 1;
        ex = ey;
        ey = DispX->Width - temp - 1;
        DispX->Direct = 0;
        DispX->SetXcmd = 0X2A;
        DispX->SetYcmd = 0X2B;
        LCD_Fill(DispX, sx, sy, ex, ey, color);
        DispX->Direct = 1;
        DispX->SetXcmd = 0X2B;
        DispX->SetYcmd = 0X2A;
    }else
    {
        xlen = ex - sx + 1;
        for(i = sy; i <= ey; i++)
        {
            LCD_SetCursor(DispX, sx, i);                //設置光標位置 
            DispX->SelectReg(DispX->GRAMcmd);           //開始寫入GRAM
            for(j = 0; j < xlen; j++)
                DispX->WriteData(color);                //設置光標位置
        }
    }
}

/**
 *******************************************************************************
 * @brief       Rectangle Fill Color 在指定區矩形域內填充指定顏色塊
 * @param       sx Start X-axes 坐標
 * @param       sy Start Y-axes 坐標
 * @param       ex end X-axes 坐標
 * @param       ey end Y-axes 坐標
 * @param       color 要填充的顏色
 * @return      None
 * @note        (sx,sy),(ex,ey):填充矩形對角坐標,區域大小為:(ex-sx+1)*(ey-sy+1)
 *******************************************************************************
 */
void LCD_Color_Fill(Display_HandleTypeDef *DispX, uint16_t sx, uint16_t sy, uint16_t width, uint16_t height, __I uint16_t *color)
{
    //uint16_t height, width;
    uint16_t i, j;
    //width = ex - sx + 1;                            //得到填充的寬度
    //height = ey - sy + 1;                           //高度
    for(i = 0; i < height; i++)
    {
        LCD_SetCursor(DispX, sx, sy + i);           //設置光標位置 
        DispX->SelectReg(DispX->GRAMcmd);           //開始寫入GRAM

        for(j = 0; j < width; j++)
            DispX->WriteData(color[i * width + j]); //寫入數據 
    }
}

/**
 *******************************************************************************
 * @brief       Rectangle Fill Color 在指定區矩形域內填充指定顏色塊
 * @param       sx Start X-axes 坐標
 * @param       sy Start Y-axes 坐標
 * @param       ex end X-axes 坐標
 * @param       ey end Y-axes 坐標
 * @param       color 要填充的顏色
 * @return      None
 * @note        (sx,sy),(ex,ey):填充矩形對角坐標,區域大小為:(ex-sx+1)*(ey-sy+1)
 *******************************************************************************
 */
void LCD_Color_Fill_DMA(Display_HandleTypeDef *DispX, uint16_t sx, uint16_t sy, uint16_t width, uint16_t height, uint16_t *color)
{
    //uint16_t height, width;
    uint16_t i, j;
    //width = ex - sx + 1;                          //得到填充的寬度
    //height = ey - sy + 1;                         //高度
    for(i = 0; i < height; i++)
    {
        LCD_SetCursor(DispX, sx, sy + i);           //設置光標位置 
        DispX->SelectReg(DispX->GRAMcmd);           //開始寫入GRAM

        for(j = 0; j < width; j++)
            DispX->WriteData(color[i * width + j]); //寫入數據 
    }
}

/**
 *******************************************************************************
 * @brief       Rectangle Fill Color 在指定區矩形域內填充指定顏色塊
 * @param       sx Start X-axes 坐標
 * @param       sy Start Y-axes 坐標
 * @param       ex end X-axes 坐標
 * @param       ey end Y-axes 坐標
 * @param       color 要填充的顏色
 * @return      None
 * @note        (sx,sy),(ex,ey):填充矩形對角坐標,區域大小為:(ex-sx+1)*(ey-sy+1)
 *******************************************************************************
 */
void LCD_Color_FillX(Display_HandleTypeDef *DispX, uint16_t sx, uint16_t sy, uint16_t ex, uint16_t ey, uint16_t *color)
{
    uint32_t iColor;
    uint16_t height, width;
    uint16_t i, j;

    width = ex - sx + 1;                            //得到填充的寬度
    height = ey - sy + 1;                           //高度
    for(i = 0; i < height; i++)
    {
        LCD_SetCursor(DispX, sx, sy + i);           //設置光標位置 
        DispX->SelectReg(DispX->GRAMcmd);           //開始寫入GRAM

        for(j = 0; j < width; j++){
            iColor = color[i * width + j];
            DispX->WriteData(iColor); //寫入數據
        }
    }
}

/**
 *******************************************************************************
 * @brief       Draw Circle 在指定位置畫一個指定大小的圓
 * @param       x0 Center of circle X-axes 坐標
 * @param       y0 Center of circle Y-axes 坐標
 * @param       r Radius 半徑
 * @param       color : Circle Color
 * @return      None
 * @note        (x0, y0):中心點
 *******************************************************************************
 */
void LCD_Draw_Circle(Display_HandleTypeDef *DispX, uint16_t x0, uint16_t y0, uint8_t r, uint16_t Color)
{
    int a, b;
    int di;
    a = 0; b = r;
    di = 3 - (r << 1);             //判斷下個點位置的標志
    while(a <= b)
    {
        LCD_DrawPoint(DispX, (uint16_t)(x0-b), (uint16_t)(y0-a), Color);             //3
        LCD_DrawPoint(DispX, (uint16_t)(x0+b), (uint16_t)(y0-a), Color);             //0
        LCD_DrawPoint(DispX, (uint16_t)(x0-a), (uint16_t)(y0+b), Color);             //1
        LCD_DrawPoint(DispX, (uint16_t)(x0-b), (uint16_t)(y0-a), Color);             //7
        LCD_DrawPoint(DispX, (uint16_t)(x0-a), (uint16_t)(y0-b), Color);             //2
        LCD_DrawPoint(DispX, (uint16_t)(x0+b), (uint16_t)(y0+a), Color);             //4
        LCD_DrawPoint(DispX, (uint16_t)(x0+a), (uint16_t)(y0-b), Color);             //5
        LCD_DrawPoint(DispX, (uint16_t)(x0+a), (uint16_t)(y0+b), Color);             //6 
        LCD_DrawPoint(DispX, (uint16_t)(x0-b), (uint16_t)(y0+a), Color);
        a++;
        //使用Bresenham算法畫圓
        if(di<0)di +=4*a+6;	  
        else
        {
            di += 10 + 4 * (a - b);
            b--;
        } 
        LCD_DrawPoint(DispX, (uint16_t)(x0 + a), (uint16_t)(y0 + b), Color);
    }
}

/**
 *******************************************************************************
 * @brief       Print Number 顯示數字
 * @param       x : Start X-axes 起始坐標
 * @param       y : Start Y-axes 起始坐標
 * @param       num : Number value(0~4294967295) 數值
 * @param       len : Number of digits 數字的位數
 * @param       size : Font size 字體大小
 * @param       color : Number Color 顏色
 * @return      None
 * @note        高位為0不顯示
 *******************************************************************************
 */
void LCD_ShowChar(Display_HandleTypeDef *DispX, uint16_t x, uint16_t y, uint8_t num, uint8_t size, uint8_t mode, uint16_t Color)
{
    uint8_t temp,t1,t;
    uint16_t y0 = y;
    uint8_t csize = (size / 8 + ((size % 8) ? 1 : 0)) * (size / 2); //得到字體一個字符對應點陣集所占的字節數
    //設置窗口
    num = num - ' ' ;//得到偏移後的值
    for(t = 0; t < csize; t++)
    {   
        if(size == 12)temp = asc2_1206[num][t];         //調用1206字體
        else if(size == 16)temp = asc2_1608[num][t];    //調用1608字體
        else if(size == 24)temp = asc2_2412[num][t];    //調用2412字體
        else if(size == 32)temp = asc2_3216[num][t];    //調用3216字體
        else return;                                    //沒有的字庫
        for(t1 = 0; t1 < 8; t1++)
        {
            if(temp & 0x80) LCD_DrawPoint(DispX, x, y, Color);
            else if(mode == 0) LCD_DrawPoint(DispX, x, y, DispX->BackColor);
            temp <<= 1;
            y++;
            if(x >= DispX->Width)return;                //超區域了
            if((y - y0) == size)
            {
                y = y0;
                x++;
                if(x >= DispX->Width)return;            //超區域了
                break;
            }
        }
    }
}

/**
 *******************************************************************************
 * @brief       Print Number 顯示數字
 * @param       x : Start X-axes 起始坐標
 * @param       y : Start Y-axes 起始坐標
 * @param       num : Number value(0~4294967295) 數值
 * @param       len : Number of digits 數字的位數
 * @param       size : Font size 字體大小
 * @param       color : Number Color 顏色
 * @return      None
 * @note        高位為0不顯示
 *******************************************************************************
 */
void LCD_ShowCharX(Display_HandleTypeDef *DispX, uint16_t x, uint16_t y, uint8_t num, uint8_t size, uint16_t Color, uint16_t BackColor)
{
    uint8_t temp,t1,t;
    uint16_t y0 = y;
    uint8_t csize = (size / 8 + ((size % 8) ? 1 : 0)) * (size / 2); //得到字體一個字符對應點陣集所占的字節數
    //設置窗口
    num = num - ' ' ;//得到偏移後的值
    for(t = 0; t < csize; t++)
    {   
        if(size == 12)temp = asc2_1206[num][t];         //調用1206字體
        else if(size == 16)temp = asc2_1608[num][t];    //調用1608字體
        else if(size == 24)temp = asc2_2412[num][t];    //調用2412字體
        else if(size == 32)temp = asc2_3216[num][t];    //調用3216字體
        else return;                                    //沒有的字庫
        for(t1=0;t1<8;t1++)
        {
            if(temp & 0x80) LCD_DrawPoint(DispX, x, y, Color);
            else LCD_DrawPoint(DispX, x, y, BackColor);
            temp <<= 1;
            y++;
            if(x >= DispX->Width)return;                //超區域了
            if((y - y0) == size)
            {
                y = y0;
                x++;
                if(x >= DispX->Width)return;            //超區域了
                break;
            }
        }
    }
}

/**
 *******************************************************************************
 * @brief       m to the power of n. m^n次方函數
 * @param       m : 
 * @param       n : 
 * @return      uint32_t m^n次方
 *******************************************************************************
 */
uint32_t LCD_Pow(uint8_t m,uint8_t n)
{
    uint32_t result = 1;
    while(n--)result*=m;
    return result;
}

/**
 *******************************************************************************
 * @brief       Print Number 顯示數字
 * @param       x : Start X-axes 起始坐標
 * @param       y : Start Y-axes 起始坐標
 * @param       num : Number value(0~4294967295) 數值
 * @param       len : Number of digits 數字的位數
 * @param       size : Font size 字體大小
 * @param       mode : overlay 疊加方式
 *      @arg        bit7 is 0 Don't fill.
 *      @arg        bit7 is 1 Fill.
 *      @arg        bit0 is 0 No overlay.
 *      @arg        bit0 is 1 overlay.
 * @param       color : Number Color 顏色
 * @return      None
 * @note        高位為0顯示
 *******************************************************************************
 */
void LCD_ShowNum(Display_HandleTypeDef *DispX, uint16_t x, uint16_t y, uint32_t num, uint8_t len, uint8_t size, uint16_t Color)
{
    uint8_t t,temp;
    uint8_t enshow=0;
    for(t = 0; t < len; t++)
    {
        temp = (num / LCD_Pow(10, len-t-1)) % 10;
        if(enshow == 0 && t < (len-1))
        {
            if(temp == 0)
            {
                LCD_ShowChar(DispX, x + (size / 2) * t, y, ' ', size, 0, Color);
                continue;
            }else enshow=1; 
        }
        LCD_ShowChar(DispX, x+(size/2)*t, y, temp + '0', size, 0, Color); 
    }
}

/**
 *******************************************************************************
 * @brief       Print one char 在指定位置顯示一個字符
 * @param       x : Start X-axes 起始坐標
 * @param       y : Start Y-axes 起始坐標
 * @param       num : ASCII Number " "--->"~"
 * @param       size : Font size 字體大小
 * @param       mode 疊加方式
 *      @arg        0:No overlay
 *      @arg        1:overlay
 * @param       color : Font Color
 * @return      None
 *******************************************************************************
 */
void LCD_ShowxNum(Display_HandleTypeDef *DispX, uint16_t x, uint16_t y, uint32_t num, uint8_t len, uint8_t size, uint8_t mode, uint16_t Color)
{
    uint8_t t,temp;
    uint8_t enshow=0;
    for(t=0;t<len;t++)
    {
        temp=(num/LCD_Pow(10,len-t-1))%10;
        if(enshow==0&&t<(len-1))
        {
            if(temp==0)
            {
                if(mode&0X80)LCD_ShowChar(DispX, x+(size/2)*t, y, '0', size, mode&0X01, Color);
                else LCD_ShowChar(DispX, x+(size/2)*t, y, ' ', size, mode&0X01, Color);
                continue;
            }else enshow=1;
        }
        LCD_ShowChar(DispX, x+(size/2)*t, y, temp+'0', size, mode&0X01, Color); 
    }
}

/**
 *******************************************************************************
 * @brief       Print String 顯示字符串
 * @param       x : Start X-axes 起始坐標
 * @param       y : Start Y-axes 起始坐標
 * @param       width : Displayable width range 區域寬度大小
 * @param       height : Displayable height range 區域高度大小
 * @param       Color : String Color 顏色
 * @param       size : Font size 字體大小
 * @param       p : String start address
 * @return      None
 *******************************************************************************
 */
void LCD_ShowString(Display_HandleTypeDef *DispX, uint16_t x, uint16_t y, uint16_t width, uint16_t height, uint16_t Color, uint8_t size, uint8_t *p)
{
    uint16_t x0 = x;
    width += x;
    height += y;
    while((*p <= '~')&&(*p >= ' '))//判斷是不是非法字符!
    {
        if(x >= width){x = x0; y += size;}
        if(y >= height)break;//退出
        LCD_ShowChar(DispX, x, y, *p, size, 1, Color);
        x += size / 2;
        p++;
    }
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
void LCD_ShowStringX(Display_HandleTypeDef *DispX, uint16_t x, uint16_t y, uint16_t width, uint16_t height, uint16_t Color, uint16_t BackColor, uint8_t size, __I uint8_t *p)
{
    uint16_t x0 = x;
    width += x;
    height += y;
    while((*p <= '~') && (*p >= ' '))//判斷是不是非法字符!
    {
        if(x >= width){x = x0; y += size;}
        if(y >= height)break;//退出
        LCD_ShowCharX(DispX, x, y, *p, size, Color, BackColor);
        x += size/2;
        p++;
    }
}

/**
 *******************************************************************************
 * @brief       Color Bar 蝘敶拇芋撘?
 * @return      None
 *******************************************************************************
 */
static uint16_t ColorBar_1[] = {0xFFFF, 0x001F, 0xF800, 0xF81F, 0x07E0, 0x07FF, 0xFFE0, 0x0000,};
void LCD_ColorBar_1(Display_HandleTypeDef *DispX)
{
    uint16_t i, j, lColorTemp, lColorCount;  
//    uint32_t totalpoint;

//    totalpoint = DispX->Width;
//    totalpoint *= DispX->Height;                         //得到總點數
    if((DispX->ID == 0X6804)&&(DispX->Direct != 0))        //6804橫屏的時候特殊處理  
    {
        DispX->Direct = 0;
        DispX->SetXcmd = 0X2A;
        DispX->SetYcmd = 0X2B;
        LCD_SetCursor(DispX, 0x00,0x0000);                 //設置光標位置
        DispX->Direct = 1;
        DispX->SetXcmd = 0X2B;
        DispX->SetYcmd = 0X2A;
    }else LCD_SetCursor(DispX, 0x00,0x0000);               //設置光標位置
    DispX->SelectReg(DispX->GRAMcmd);                      //開始寫入GRAM

    lColorCount = 0;

    do{
        lColorTemp = ColorBar_1[lColorCount];
        i = (uint16_t)(DispX->Height / 8);
        do{
            j = DispX->Width;
            do{
                DispX->WriteData(lColorTemp);
            }while(-- j != 0);
        }while(-- i != 0);
    }while(++ lColorCount != 0x0008);
}

/**
 *******************************************************************************
 * @brief       Color Bar 2 秀彩模式2
 * @return      None
 *******************************************************************************
 */
void LCD_ColorBar_2(Display_HandleTypeDef *DispX)
{
    float lColorInt = 0;
    float lColorScreen;
//    uint32_t totalpoint;
    uint16_t i, j, lColorTemp;

//    totalpoint = DispX->Width;
//    totalpoint *= DispX->Height;                         //得到總點數
    if((DispX->ID == 0X6804) && (DispX->Direct != 0))      //6804橫屏的時候特殊處理
    {
        DispX->Direct = 0;
        DispX->SetXcmd = 0X2A;
        DispX->SetYcmd = 0X2B;
        LCD_SetCursor(DispX, 0x00,0x0000);                 //設置光標位置
        DispX->Direct = 1;
        DispX->SetXcmd = 0X2B;
        DispX->SetYcmd = 0X2A;
    }else LCD_SetCursor(DispX, 0x00,0x0000);               //設置光標位置
    DispX->SelectReg(DispX->GRAMcmd);                      //開始寫入GRAM

    // B = 0 ~ 255;
    lColorScreen = DispX->Height;
    lColorScreen /= 6;
    lColorScreen = 32 / lColorScreen;

    lColorInt = 0;
    lColorTemp = 0xF800;
    i = (uint16_t)(DispX->Height / 6);
    do{
        j = DispX->Width;
        do{
            DispX->WriteData(lColorTemp);
        }while(-- j != 0);
        lColorInt += lColorScreen;
        lColorTemp = 0xF800 + (uint16_t)lColorInt;
    }while(-- i != 0);

    // R = 255 ~ 0;
    lColorInt = 0;
    lColorTemp = 0xF81F;
    i = (uint16_t)(DispX->Height / 6);
    do{
        j = DispX->Width;
        do{
            DispX->WriteData(lColorTemp);
        }while(-- j != 0);
        lColorInt += lColorScreen;
        lColorTemp = (uint16_t)(0xF81F - ((uint16_t)(((uint16_t)lColorInt) << 11)));
    }while(-- i != 0);

    // G = 0 ~ 255;
    lColorScreen = DispX->Height;
    lColorScreen /= 6;
    lColorScreen = 64 / lColorScreen;
    lColorInt = 0;
    lColorTemp = 0x001F;
    i = (uint16_t)(DispX->Height / 6);
    do{
        j = DispX->Width;
        do{
            DispX->WriteData(lColorTemp);
        }while(-- j != 0);
        lColorInt += lColorScreen;
        lColorTemp = (uint16_t)(0x001F + (uint16_t)(((uint16_t)lColorInt) << 5));
    }while(-- i != 0);

    // B = 255 ~ 0;
    lColorScreen = DispX->Height;
    lColorScreen /= 6;
    lColorScreen = 32 / lColorScreen;
    lColorInt = 0;
    lColorTemp = 0x7FF;
    i = (uint16_t)(DispX->Height / 6);
    do{
        j = DispX->Width;
        do{
            DispX->WriteData(lColorTemp);
        }while(-- j != 0);
        lColorInt += lColorScreen;
        lColorTemp = 0x7FF - ((uint16_t)lColorInt);
    }while(-- i != 0);

    // R = 0 ~ 255;
    lColorScreen = DispX->Height;
    lColorScreen /= 6;
    lColorScreen = 32 / lColorScreen;
    lColorInt = 0;
    lColorTemp = 0x07E0;
    i = (uint16_t)(DispX->Height / 6);
    do{
        j = DispX->Width;
        do{
            DispX->WriteData(lColorTemp);
        }while(-- j != 0);
        lColorInt += lColorScreen;
        lColorTemp = (uint16_t)(0x07E0 + (uint16_t)(((uint16_t)lColorInt) << 11));
    }while(-- i != 0);

    // B = 255 ~ 0;
    lColorScreen = DispX->Height;
    lColorScreen /= 6;
    lColorScreen = 64 / lColorScreen;
    lColorInt = 0;
    lColorTemp = 0xFFE0;
    i = (uint16_t)(DispX->Height / 6);
    do{
        j = DispX->Width;
        do{
            DispX->WriteData(lColorTemp);
        }while(-- j != 0);
        lColorInt += lColorScreen;
        lColorTemp = (uint16_t)(0xFFE0 - (uint16_t)(((uint16_t)lColorInt) << 5));
    }while(-- i != 0);
}

/**
 *******************************************************************************
 * @brief       Show Gray Bar 對比模式
 * @return      None
 *******************************************************************************
 */
uint16_t GrayBar_1[] = {0xFFFF, 0xEF7D, 0xDEFB, 0xCE79, 0xBDF7, 0xAD75, 0x9CF3, 0x8C71,
                        0x7BEF, 0x6B6D, 0x5AEB, 0x4A69, 0x39E7, 0x2965, 0x18E3, 0x0000};
void LCD_GrayBar_1(Display_HandleTypeDef *DispX)
{
//    uint32_t totalpoint;
    uint16_t i, j, lColorTemp, lColorCount;

//    totalpoint = DispX->Width;
//    totalpoint *= DispX->Height;                         //得到總點數
    if((DispX->ID == 0X6804)&&(DispX->Direct != 0))        //6804橫屏的時候特殊處理
    {
        DispX->Direct = 0;
        DispX->SetXcmd = 0X2A;
        DispX->SetYcmd = 0X2B;
        LCD_SetCursor(DispX, 0x00,0x0000);                 //設置光標位置
        DispX->Direct = 1;
        DispX->SetXcmd = 0X2B;
        DispX->SetYcmd = 0X2A;
    }else LCD_SetCursor(DispX, 0x00,0x0000);               //設置光標位置
    DispX->SelectReg(DispX->GRAMcmd);                      //開始寫入GRAM

    lColorCount = 0;

    do{
        lColorTemp = GrayBar_1[lColorCount];
        i = (uint16_t)(DispX->Height / 16);
        do{
            j = DispX->Width;
            do{
                DispX->WriteData(lColorTemp);
            }while(-- j != 0);
        }while(-- i != 0);
    }while(++ lColorCount != 0x0010);
}

/**
 *******************************************************************************
 * @brief       Set Printf Windows 設置窗口,並自動設置畫點坐標到窗口左上角(sx,sy).
 * @param       sx windows Start X-axes 窗口起始坐標(左上角)
 * @param       sy windows Start Y-axes 窗口起始坐標(左上角)
 * @param       width windows width 窗體大小
 * @param       height windows height 窗體大小
 * @return      None
 * @note        68042,橫屏時不支持窗口設置!! 
 *******************************************************************************
 */
#if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) /* ARM Compiler V6 */
    #pragma clang diagnostic push
    #pragma clang diagnostic ignored "-Wmissing-variable-declarations"
        Display_HandleTypeDef *gPrintf;
        uint16_t gPrintfWindowXstart;
        uint16_t gPrintfWindowYstart;
        uint16_t gPrintfWindowXsize;
        uint16_t gPrintfWindowYsize;
    #pragma clang diagnostic pop
#else
    Display_HandleTypeDef *gPrintf;
    uint16_t gPrintfWindowXstart;
    uint16_t gPrintfWindowYstart;
    uint16_t gPrintfWindowXsize;
    uint16_t gPrintfWindowYsize;
#endif

void Set_Printf_Window(Display_HandleTypeDef *DispX, uint16_t sx, uint16_t sy, uint16_t width, uint16_t height)
{
    gPrintf = DispX;
    gPrintfWindowXstart = sx;
    gPrintfWindowYstart = sy;
    gPrintfWindowXsize = width;
    gPrintfWindowYsize = height;
}

/**
 *******************************************************************************
 * @brief       
 * @details     
 * @return      
 * @exception   No
 * @note
 * @par         Example
 * @code
 * @endcode
 *******************************************************************************
 */
#define DefaultFontSize 16                                          // Font Size 12 / 16 / 24 / 32
#define FputcBuffSize   1024                                        // LCD ILI9341 240x320 / FontSize 16x8  

int fputc(int ch, FILE* f)
{
    int num;

    uint16_t y0 = gFputcY;
    uint8_t temp, t1, t;
#if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) /* ARM Compiler V6 */
    #pragma clang diagnostic push
    #pragma clang diagnostic ignored "-Wunreachable-code"
        uint8_t csize = (DefaultFontSize / 8 + ((DefaultFontSize % 8) ? 1 : 0)) * (DefaultFontSize / 2); //得到字體一個字符對應點陣集所占的字節數
    #pragma clang diagnostic pop
#else
    uint8_t csize = (DefaultFontSize / 8 + ((DefaultFontSize % 8) ? 1 : 0)) * (DefaultFontSize / 2); //得到字體一個字符對應點陣集所占的字節數
#endif

    (void)f;

    num = ch - ' ';                                                                                 //得到偏移後的值

    if(ch == '\n')   // Line Feed
    {
        
        gFputcY += DefaultFontSize;
        LCD_Fill(&LCD1st, 0, gFputcY, LCD1st.Width, (gFputcY + DefaultFontSize), LCD1st.BackColor);
        goto fputc_exit;
    }

    if(ch == '\r')   // Carriage return
    {
        gFputcX = 0;
        //gFputcY += DefaultFontSize;
        LCD_Fill(&LCD1st, 0, gFputcY, LCD1st.Width, (gFputcY + DefaultFontSize), LCD1st.BackColor);
        goto fputc_exit;
    }

    for(t = 0; t < csize; t++)
    {
//        if(DefaultFontSize == 12)temp = asc2_1206[num][t];         //調用1206字體
//        else if(DefaultFontSize == 16)temp = asc2_1608[num][t];    //調用1608字體
//        else if(DefaultFontSize == 24)temp = asc2_2412[num][t];    //調用2412字體
//        else if(DefaultFontSize == 32)temp = asc2_3216[num][t];    //調用3216字體
//        else return ch;                                            //沒有的字庫
        temp = asc2_1608[num][t];
        for(t1 = 0; t1 < 8; t1 ++)
        {
            if(temp & 0x80) LCD_DrawPoint(&LCD1st, gFputcX, gFputcY, LCD1st.PointColor);
            else LCD_DrawPoint(&LCD1st, gFputcX, gFputcY, LCD1st.BackColor);

            temp <<= 1;
            gFputcY ++;
            if(gFputcY >= LCD1st.Height)
            {
                gFputcY = 0;
                LCD_Fill(&LCD1st, 0, gFputcY, LCD1st.Width, (gFputcY + DefaultFontSize), LCD1st.BackColor);
                goto fputc_exit;                                     //超區域了
            }

            if((gFputcY - y0) == DefaultFontSize)
            {
                gFputcY = y0;
                gFputcX ++;
                if(gFputcX >= LCD1st.Width)
                {
                    gFputcX = 0;
                    gFputcY = gFputcY + DefaultFontSize;
                    if((gFputcY + DefaultFontSize) >= LCD1st.Height)
                    {
                        gFputcY = 0;
                    }
                    LCD_Fill(&LCD1st, 0, gFputcY, LCD1st.Width, (gFputcY + DefaultFontSize), LCD1st.BackColor);
                    goto fputc_exit;                                      //超區域了
                }
                break;
            }
        }
    }

  fputc_exit:
    return ch;
}

/**
 *******************************************************************************
 * @brief       
 * @return      
 *******************************************************************************
 */

   // uint8_t gArrayU8[80];

void LCD_Interface_Init(void)
{
    URT_Struct* URTx = URT0;
  // GPIO IO Config
    // LCD Reset
    __SPI_LCD_RESET_PIN()->CR.W  = PX_CR_IOM_ppo_w | PX_CR_HS_enable_w | PX_CR_PU_disable_w | PX_CR_INV_disable_w | PX_CR_ODC_level0_w | PX_CR_FDIV_bypass_w | (((uint32_t)__SPI_LCD_RESET_PIN_AFS()) << 12);

    // LCD Back Lingt
    __SPI_LCD_BACKLIGHT_PIN()->CR.W  = PX_CR_IOM_ppo_w | PX_CR_HS_enable_w | PX_CR_PU_disable_w | PX_CR_INV_disable_w | PX_CR_ODC_level0_w | PX_CR_FDIV_bypass_w | (((uint32_t)__SPI_LCD_BACKLIGHT_PIN_AFS()) << 12);

    // SPI DC
    __SPI_LCD_DC_PIN()->CR.W  = PX_CR_IOM_ppo_w | PX_CR_HS_enable_w | PX_CR_PU_disable_w | PX_CR_INV_disable_w | PX_CR_ODC_level0_w | PX_CR_FDIV_bypass_w | (((uint32_t)__SPI_LCD_DC_PIN_AFS()) << 12);

    // SPI NSS
    __SPI_LCD_CS_PIN()->CR.W  = PX_CR_IOM_ppo_w | PX_CR_HS_enable_w | PX_CR_PU_disable_w | PX_CR_INV_disable_w | PX_CR_ODC_level0_w | PX_CR_FDIV_bypass_w | (((uint32_t)__SPI_LCD_CS_PIN_AFS()) << 12);

    // SPI CLK
    __SPI_LCD_CLK_PIN()->CR.W  = PX_CR_IOM_ppo_w | PX_CR_HS_enable_w | PX_CR_PU_disable_w | PX_CR_INV_disable_w | PX_CR_ODC_level0_w | PX_CR_FDIV_bypass_w | (((uint32_t)__SPI_LCD_CLK_PIN_AFS()) << 12);

    // SPI MOSI
    __SPI_LCD_SDI_PIN()->CR.W  = PX_CR_IOM_ppo_w | PX_CR_HS_enable_w | PX_CR_PU_disable_w | PX_CR_INV_disable_w | PX_CR_ODC_level0_w | PX_CR_FDIV_bypass_w | (((uint32_t)__SPI_LCD_SDI_PIN_AFS()) << 12);

    // SPI MISO
    __SPI_LCD_SDO_PIN()->CR.W  = PX_CR_IOM_din_w | PX_CR_HS_enable_w | PX_CR_PU_disable_w | PX_CR_INV_disable_w | PX_CR_ODC_level0_w | PX_CR_FDIV_bypass_w | (((uint32_t)__SPI_LCD_SDO_PIN_AFS()) << 12);

    // URT0 SPI Initial

    URTx->CLK.W = URT_CLK_BR_CKS_psc_w | \
                  URT_CLK_CKO_LCK_un_locked_w | \
                  URT_CLK_CKO_STA_1_w | \
                  URT_CLK_BR_MDS_combined_w | \
                  URT_CLK_BR_EN_enable_w | \
                  URT_CLK_TX_CKS_internal_w | \
                  URT_CLK_RX_CKS_internal_w | \
                  URT_CLK_ECK_CKS_clk_w | \
                  URT_CLK_CLK_CKS_out_w | \
                  URT_CLK_CLK_EN_enable_w | \
                  URT_CLK_CK_SEL_proc_w;

    URTx->CR0.W = URT_CR0_DMA_TXEN_disable_w | \
                  URT_CR0_DMA_RXEN_disable_w | \
                  URT_CR0_DDTX_EN_disable_w | \
                  URT_CR0_SDT_EN_disable_w | \
                  URT_CR0_LBM_EN_disable_w | \
                  URT_CR0_NCHAR_DIS_enable_w | \
                  URT_CR0_NCHAR_HE_disable_w | \
                  URT_CR0_IDL_MDS_no_w | \
                  URT_CR0_RX_TH_1byte_w | \
                  URT_CR0_DE_GT_1_4_w | \
                  URT_CR0_DE_INV_disable_w | \
                  URT_CR0_DE_EN_disable_w | \
                  URT_CR0_TX_INV_disable_w | \
                  URT_CR0_RX_INV_disable_w | \
                  URT_CR0_SYNC_MDS_master_w | \
                  URT_CR0_IO_SWP_disable_w | \
                  0 << URT_CR0_GSA_EN_shift_w | \
                  URT_CR0_MDS_sync_w | \
                  URT_CR0_DAT_LINE_2_w | \
                  URT_CR0_HDX_EN_disable_w | \
                  URT_CR0_OS_MDS_three_w | \
                  URT_CR0_EN_disable_w;

    URTx->CR1.W = 1 << URT_CR1_TXOS_NUM_shift_w | \
                  URT_CR1_TXSTP_LEN_1bit_w | \
                  URT_CR1_TXMSB_EN_enable_w | \
                  URT_CR1_TXPAR_STK_disable_w | \
                  URT_CR1_TXPAR_POL_even_w | \
                  URT_CR1_TXPAR_EN_disable_w | \
                  URT_CR1_TXDSIZE_8bit_w | \
                  1 << URT_CR1_RXOS_NUM_shift_w | \
                  URT_CR1_RXSTP_LEN_1bit_w | \
                  URT_CR1_RXMSB_EN_enable_w | \
                  URT_CR1_RXPAR_STK_disable_w | \
                  URT_CR1_RXPAR_POL_even_w | \
                  URT_CR1_RXPAR_EN_disable_w | \
                  URT_CR1_RXDSIZE_8bit_w;

    URTx->CR2.W = URT_CR2_DOUT_IDL_lbit_w | \
                  URT_CR2_DOUT_MDS_disable_w | \
                  URT_CR2_NSSI_EN_disable_w | \
                  URT_CR2_NSS_SWEN_disable_w | \
                  URT_CR2_NSS_INV_disable_w | \
                  URT_CR2_NSSI_INV_disable_w | \
                  1 << URT_CR2_NSS_SWI_shift_w | \
                  0 << URT_CR2_NSS_SWO_shift_w | \
                  URT_CR2_TX_HALT_disable_w | \
                  URT_CR2_TX_EN_enable_w | \
                  URT_CR2_RX_EN_enable_w | \
                  URT_CR2_ADR_TX_normal_w | \
                  URT_CR2_BK_TX_normal_w;

    URTx->CR3.W = 0 | \
                  (0xA << URT_CR3_DET_IDL_shift_w) | \
                  URT_CR3_DET_BK_1bit_w | \
                  URT_CR3_CPHA_leading_edge_w | \
                  URT_CR3_CPOL_low_w;

    URTx->CR4.W = 0;
    URTx->RLR.W = 0 << URT_RLR_PSR_shift_w | \
                  2 << URT_RLR_RLR_shift_w;
    URTx->CNT.W = 0;
    URTx->RCAP.W = 0;
    URTx->RDAT.W = 0;
    URTx->TDAT.W = 0;
    URTx->TDAT3.W = 0;
    URTx->TMOUT.W = 0;
    URTx->TMOUT2.W = 0;
    URTx->SC.W = 0;
    URTx->SADR.W = 0;
    URTx->CAL.W = 0;
    URTx->IRDA.W = 0x00000300;
    URTx->HFC.W = 0;
    URTx->MUTE.W = 0x00010100;

    URTx->CR0.W |= URT_CR0_EN_enable_w;

    // LCD Initial
    LCD1st.HWIF = 1;
    LCD1st.Reset         = SPI_LCD_Reset;
    LCD1st.BackLight     = SPI_LCD_BackLight;
    LCD1st.DataReverse   = SPI_LCD_DataReverse;
    LCD1st.SelectReg     = SPI_LCD_SelectRegister;
    LCD1st.WriteData     = SPI_LCD_WriteData;
    LCD1st.WriteData8    = SPI_LCD_WriteData8;
    LCD1st.ReadData      = SPI_LCD_ReadData;
    LCD1st.WriteRegister = SPI_LCD_WriteRegister;
    LCD1st.PointColor    = WHITE;
    LCD1st.BackColor     = BLACK;
    LCD_Init(&LCD1st);
}

