/**
 ******************************************************************************
 *
 * @file        Sample_WWDT_GeneralTimer.c
 *
 * @brief       WWDT general timer sample code, trigger warning event interrupt.
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2022/12/19
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2018 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer
 *      The Demo software is provided "AS IS"  without any warranty, either
 *      expressed or implied, including, but not limited to, the implied warranties
 *      of merchantability and fitness for a particular purpose.  The author will
 *      not be liable for any special, incidental, consequential or indirect
 *      damages due to loss of data or any other reason.
 *      These statements agree with the world wide and local dictated laws about
 *      authorship and violence against these laws.
 ******************************************************************************
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32_DRV.h"
/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
uint32_t Sample_WWDT_GeneralTimer(void);
void WWDT_IRQHandler(void);
void WWDT_IRQ(void);

/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief  	    Sample_IWDT_GeneralTimer
 * @details     1. Enable CSC to WWDT clock
 *      \n      2. WWDT Clock Configure
 *      \n      3. WWDT Interrupt Configure
 *      \n      4. WWDT Reload Counter Value Setting
 *      \n      5. WWDT Function Command
 * @return	    None	
 * @note        
 * @par         Example
 * @code
    Sample_WWDT_GeneralTimer();
 * @endcode
 *******************************************************************************
 */
uint32_t Sample_WWDT_GeneralTimer(void)
{
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12Mz
    //Select CK_MAIN Source = CK_HS
    //Select CK_LS Source = CK_ILRCO
    //Configure Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->WWDT = Enable

    /*=== 1. Enable CSC to WWDT clock ===*/
    UnProtectModuleReg(CSCprotect);                                 // Unprotect CSC module
    CSC_PeriphOnModeClock_Config(CSC_ON_WWDT, ENABLE);              // Enable WWDT module clock
    ProtectModuleReg(CSCprotect);                                   // protect CSC module
    
    // WWDT Module Protect Disable.
    UnProtectModuleReg(WWDTprotect);                                // WWDT module protect disable.
    
    /*=== 2. WWDT Clock Configure ===*/
    WWDT_DIV_Select(WWDT_DIV_4);                                    // WWDT clock source divider select.
    WWDT_PDIV_Select(WWDT_PDIV_256);                                // WWDT prescaler divider select.
    WWDT_CLK_Select(WWDT_CK_UT);                                    // WWDT clock source select.
    
    /*=== 3. WWDT Interrupt Configure ===*/
    WWDT_IT_Config(WWDT_INT_WRN, ENABLE);                           // WWDT interrupt even enable.
    NVIC_EnableIRQ(WWDT_IRQn);                                      // M0 NVIC configure.
    
    /*=== 4. WWDT Reload Counter Value Setting ===*/
    WWDT_SetReloadReg(0x3FF);                                       // WWDT reload value set.
    WWDT_SetWarningThreshold(0x3FE);                                // WWDT warning detecter
    WWDT_RefreshCounter();                                          // WWDT counter value reset(0x3FF).
    
    /*=== 5. WWDT Function Command ===*/
    WWDT_Cmd(ENABLE);                                               // WWDT module function enable.
    
    // WWDT Module Protect Enable.
    ProtectModuleReg(WWDTprotect);                                  // WWDT module protect enable.
    
    return DRV_Success;
}

/**
 *******************************************************************************
 * @brief       WWDT: An interrupt, or IRQ, is an exception signalled by a
 *              peripheral, or generated by a software request. 
 * @details     WWDT interrupt source.
 * @return      No
 *******************************************************************************
 */
void WWDT_IRQHandler(void)
{
    WWDT_IRQ();                                                     // WWDT interrupt servive routine.
}

/**
 *******************************************************************************
 * @brief       WWDT module interrupt service routine.
 * @details     Refresh WWDT counter & clear interrup flag.
 * @return      No
 *******************************************************************************
 */
void WWDT_IRQ(void)
{
    // WWDT Counter Reload.
    WWDT_RefreshCounter();                                          // WWDT counter value reset(0x3FF).
    // Clear WWDT Flag
    WWDT_ClearFlag(WWDT_WRNF);                                      // WWDT waring interrupt flag clear.
}





