/**
  ******************************************************************************
 *
 * @file        Sample_IR_Transmit.c
 *
 * @brief       IR transmit NEC data.
 *
 * @par         Project
 *              MG32
 * @version     V1.10
 * @date        2022/02/23
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
* @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_DRV.h"


/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/**
 * @enum    IR_TxStepTypeDef
 * @brief   IR Transmit Num Definition
 */
typedef enum
{
    START_Code = 0,
    Addr_Code = 1,
    BAddr_Code = 2,
    Data_Code = 3,
    BData_Code = 4,
    STOP_Code = 5
}IR_TxStepTypeDef;

/* Private define ------------------------------------------------------------*/
/**
 * @name    IR_Compare_Value_Definition
 *          IR Compare Value Definition
 */ 
//!@{
#define IR_START_CountRange   40500U              /*!< Start Code Counter Range Value Definition */
#define IR_STOP_CountRange    65535U              /*!< Stop Code Counter Range Value Definition */
#define IR_Repeat_CountRange  34500U              /*!< Repeat Code Counter Range Value Definition */
#define IR_DAT0_CountRange    3360U               /*!< Data bit transmit 0 Code Counter Range Value Definition */
#define IR_DAT1_CountRange    6720U               /*!< Data bit transmit 1 Code Counter Range Value Definition */

#define IR_START_PULSE1       27000U              /*!< Start Code Compare Value Definition */
#define IR_STOP_PULSE1        1680U               /*!< Stop Code Compare Value Definition */
#define IR_Repeat_PULSE1      27000U              /*!< Repeat Code Compare Value Definition */
#define IR_DAT0_PULSE1        1680U               /*!< Date bit transmit 0, Compare Value Definition */
#define IR_DAT1_PULSE1        1680U               /*!< Date bit transmit 1, Compare Value Definition */

#define IR_Repeat_Delay	      96U
//!@}

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
void Delay_ms(uint8_t dly_0);
uint32_t Sample_STOPDelayTime(uint32_t Dly_TM);
void Sample_TM00_CKO_OUTPUT_38KHz(void);
void Sample_TM36_PWMInit(void);
void Sample_IR_Start_Transmit(void);
void Sample_IR_Stop_Transmit(uint8_t tDLY);
void Sample_IR_Repeat_Transmit(void);
void Sample_IR_DAT0_Transmit(void);
void Sample_IR_DAT1_Transmit(void);
DRV_Return Sample_IR_DAT_Transmit_BYTE(uint8_t Addr_0, uint8_t Addr_1, uint8_t Data_0, uint8_t Data_1);
void Sample_APB_IR_Transmit_Init(void);
void Sample_APB_IR_Transmit(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/


/**
 * @name    Delay function
 *
 */ 
///@{  
/**
 *******************************************************************************
 * @brief       Delay ms function
* @param[in]    dly_0 : Delay ms count
 * @arg         0~255
 * @par         Example
 * @code
                Delay_ms(30);
 * @endcode
 *******************************************************************************
 */
void Delay_ms(uint8_t dly_0)
{
    uint16_t dly_1;
    for(;dly_0>0;dly_0--)
    {
        for(dly_1=0;dly_1<1200;dly_1++)
        {
            __NOP();
        }
    }
}

/**
 *******************************************************************************
 * @brief       Stop Delay Time Check
 * @param[in]   Dly_TM : Transmit IR data
 * @return      uint32_t
 * @note        Auto calculate stop delay time.
 * @par         Example
 * @code
                Delay = Sample_STOPDelayTime(IR_DATA);
 * @endcode
 *******************************************************************************
 */
uint32_t Sample_STOPDelayTime(uint32_t Dly_TM)
{
    uint8_t jCNT,DATA_0 = 0,DATA_1 = 0;
    uint16_t ALLD;
    
    for(jCNT=0;jCNT<32;jCNT++)
    {
        if((Dly_TM&(0x00000001<<jCNT))!=0)
            DATA_1++;
        else
            DATA_0++;
    }
    ALLD = ((DATA_1*2)+((DATA_1*24)/100))+((DATA_0)+((DATA_0*12)/100));
    ALLD = 108-ALLD;
    
    return ALLD;
}
///@}

/**
 * @name     IR Carrier Clock Initialize
 *
 */ 
///@{  
/**
 *******************************************************************************
 * @brief       IR NEC Carrier Clock Initial
 * @par         Example
 * @code
                Sample_TM00_CKO_OUTPUT_38KHz();
 * @endcode
 *******************************************************************************
 */
void Sample_TM00_CKO_OUTPUT_38KHz(void)
{
    uint16_t vTM_H,vTM_L;
    uint32_t TM_VAL;
    
    /*----- TM00 CKO IO Set -----*/
    PIN_InitTypeDef PINX_InitStruct;
 
    PINX_InitStruct.PINX_Mode               = PINX_Mode_PushPull_O;
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;
    PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;
    PINX_InitStruct.PINX_Alternate_Function = 4;
    GPIO_PinMode_Config(PIND(12),&PINX_InitStruct);
    /*----- TM00 Counter Value Set -----*/
    TM_VAL = (12000000/38000/2);
    vTM_H = (uint16_t)(TM_VAL>>16);
    vTM_L = (uint16_t) TM_VAL;
    /*----- TM00 CKO Init -----*/
    TM_DeInit(TM00);
    TM00->CLK.MBIT.CKI_SEL = 0x00;
    TM00->CR0.MBIT.MDS = 0x00;
    TM00->CNT.W = vTM_L;
    TM00->ARR.W = vTM_L;
    TM00->PSCNT.W = vTM_H;
    TM00->PSARR.W = vTM_H;
    /*----- TM00 output by Main counter -----*/
    TM00->CKO.MBIT.CKO_SEL = 1;
    TM00->CKO.MBIT.CKO_STA = SET;
    TM00->CKO.MBIT.CKO_EN = SET;
    TM00->CR0.MBIT.EN = SET;
    TM00->CR0.MBIT.EN2 = SET;
    
}
///@}

/**
 * @name    IR Data Transmitting Set
 *
 */ 
///@{  
/**
 *******************************************************************************
 * @brief       IR Data Transmitter Initialize (for IR NEC protocol)
 * @par         Example
 * @code
                Sample_TM36_PWMInit();
 * @endcode
 *******************************************************************************
 */
void Sample_TM36_PWMInit(void)
{
    TM_TimeBaseInitTypeDef TM_TMBaseInitStruct;
    
    // ----------------------------------------------------
    /*!< TM36 register initial */
    TM_TimeBaseStruct_Init(&TM_TMBaseInitStruct);
    TM_TMBaseInitStruct.TM_Prescaler = 0x0000;
    TM_TMBaseInitStruct.TM_Period = 0x0000;
    TM_TMBaseInitStruct.TM_CounterMode = Cascade;
    TM_TMBaseInitStruct.TM_MainClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_2ndClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_IntClockDivision = TM_IntDIV4;
    TM_TMBaseInitStruct.TM_MainClockSource = TM_CK_INT;
    TM_TMBaseInitStruct.TM_2ndClockSource = TM_CK_INT;
    TM_TimeBase_Init(TM36, &TM_TMBaseInitStruct);
    
    // ----------------------------------------------------
    /*!< TM36 PWM initial */
    TM_CH0Function_Config(TM36, TM_16bitPWM);
    TM_AlignmentMode_Select(TM36, TM_EdgeAlign);
    TM_ClearFlag(TM36, TMx_CF0A);
    TM_Timer_Cmd(TM36,DISABLE);
    
    // ----------------------------------------------------
    /*!< PWM CH0 Output Config */
    TM_OC00Output_Cmd(TM36,ENABLE);
    TM_InverseOC0z_Cmd(TM36, DISABLE);
    TM_OC0zOutputState_Init(TM36, CLR);
    
    TM_OC0NOutput_Cmd(TM36, DISABLE);
    TM_InverseOC0N_Cmd(TM36, DISABLE);
    
    // ----------------------------------------------------
    /*!< TM36 TRGO Initial */
    TM_TRGOOut_SW(TM36, ENABLE);
    TM_TRGO_Select(TM36, TRGO_OS0);
    
}

/**
 *******************************************************************************
 * @brief       IR Start Code Tansmit (for IR NEC protocol)
 * @par         Example
 * @code
                Sample_IR_Start_Transmit();
 * @endcode
 *******************************************************************************
 */
void Sample_IR_Start_Transmit(void)
{
    TM_TimeBaseInitTypeDef TM_TMBaseInitStruct;
    
    // ----------------------------------------------------
    /* TM36 register initial */
    TM_TMBaseInitStruct.TM_Prescaler = 0x0000U;
    TM_TMBaseInitStruct.TM_Period = IR_START_CountRange;
    TM_TMBaseInitStruct.TM_CounterMode = Cascade;
    TM_TMBaseInitStruct.TM_MainClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_2ndClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_IntClockDivision = TM_IntDIV4;
    TM_TMBaseInitStruct.TM_MainClockSource = TM_CK_INT;
    TM_TMBaseInitStruct.TM_2ndClockSource = TM_CK_INT;
    TM_TimeBase_Init(TM36, &TM_TMBaseInitStruct);
    
    // ----------------------------------------------------
    /* Clear TM36 comparator flag and enable TM36 counter */
    TM_SetCC0A(TM36, IR_START_PULSE1);
    TM_SetCC0B(TM36, IR_START_PULSE1);
    
    // ----------------------------------------------------
    /* Clear TM36 comparator flag and enable TM36 counter */
    TM_ClearFlag(TM36, TMx_CF0A);
    TM_Timer_Cmd(TM36,ENABLE);
    
    // ----------------------------------------------------
    /* TM36 enven check */
    while(TM_GetSingleFlagStatus(TM36, TMx_CF0A)!=DRV_Happened);
    while(TM_GetSingleFlagStatus(TM36, TMx_TOF)!=DRV_Happened);
    
    // ----------------------------------------------------
    /* Clear TM36 event flag and disable TM36 */
    TM_ClearFlag(TM36, (TMx_CF0A|TMx_TOF));
    TM_Timer_Cmd(TM36,DISABLE);
}

/**
 *******************************************************************************
 * @brief       IR Stop Code Tansmit (for IR NEC protocol)
 * @par         Example
 * @code
                Sample_IR_Stop_Transmit(IR_Repeat_Delay);
 * @endcode
 *******************************************************************************
 */
void Sample_IR_Stop_Transmit(uint8_t tDLY)
{
    // ----------------------------------------------------
    /* TM36 register initial */
    TM_TimeBaseInitTypeDef TM_TMBaseInitStruct;
    
    TM_TMBaseInitStruct.TM_Prescaler = 0x0000U;
    TM_TMBaseInitStruct.TM_Period = IR_STOP_CountRange;
    TM_TMBaseInitStruct.TM_CounterMode = Cascade;
    TM_TMBaseInitStruct.TM_MainClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_2ndClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_IntClockDivision = TM_IntDIV4;
    TM_TMBaseInitStruct.TM_MainClockSource = TM_CK_INT;
    TM_TMBaseInitStruct.TM_2ndClockSource = TM_CK_INT;
    TM_TimeBase_Init(TM36, &TM_TMBaseInitStruct);
    
    // ----------------------------------------------------
    /* IR stop bit output */
    
    /* TM36 PWM comparator value set */
    TM_SetCC0A(TM36, IR_STOP_PULSE1);
    TM_SetCC0B(TM36, IR_STOP_PULSE1);
    
    // ----------------------------------------------------
    /* Clear TM36 comparator flag and enable TM36 counter */
    TM_ClearFlag(TM36, TMx_CF0A);
    TM_Timer_Cmd(TM36,ENABLE);
    
    
    while(TM_GetSingleFlagStatus(TM36, TMx_CF0A)!=DRV_Happened);
    TM_SetCC0A(TM36, 0x0000U);
    TM_SetCC0B(TM36, 0x0000U);
    while(TM_GetSingleFlagStatus(TM36, TMx_TOF)!=DRV_Happened);
    TM_ClearFlag(TM36, (TMx_CF0A|TMx_TOF|TMx_TOF2));
    TM_Timer_Cmd(TM36,DISABLE);
    Delay_ms(tDLY);
}

/**
 *******************************************************************************
 * @brief       IR Repeat Code Tansmit (for IR NEC protocol)
 * @par         Example
 * @code
                Sample_IR_Repeat_Transmit();
 * @endcode
 *******************************************************************************
 */
void Sample_IR_Repeat_Transmit(void)
{
    TM_TimeBaseInitTypeDef TM_TMBaseInitStruct;
    
    // ----------------------------------------------------
    /* TM36 Initial */
    TM_TMBaseInitStruct.TM_Prescaler = 0x0000U;
    TM_TMBaseInitStruct.TM_Period = IR_Repeat_CountRange;
    TM_TMBaseInitStruct.TM_CounterMode = Cascade;
    TM_TMBaseInitStruct.TM_MainClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_2ndClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_IntClockDivision = TM_IntDIV4;
    TM_TMBaseInitStruct.TM_MainClockSource = TM_CK_INT;
    TM_TMBaseInitStruct.TM_2ndClockSource = TM_CK_INT;
    TM_TimeBase_Init(TM36, &TM_TMBaseInitStruct);
    
    // ----------------------------------------------------
    /* TM36 PWM comparator value set */
    TM_SetCC0A(TM36, IR_Repeat_PULSE1);
    TM_SetCC0B(TM36, IR_Repeat_PULSE1);
    
    // ----------------------------------------------------
    /* Clear TM36 comparator flag and enable TM36 counter */
    TM_ClearFlag(TM36, TMx_CF0A);
    TM_Timer_Cmd(TM36,ENABLE);
    
    // ----------------------------------------------------
    /* TM36 enven check */
    while(TM_GetSingleFlagStatus(TM36, TMx_CF0A)!=DRV_Happened);
    while(TM_GetSingleFlagStatus(TM36, TMx_TOF)!=DRV_Happened);
    
    // ----------------------------------------------------
    /* Clear TM36 event flag and disable TM36 */
    TM_ClearFlag(TM36, (TMx_CF0A|TMx_TOF));
    TM_Timer_Cmd(TM36,DISABLE);
    
    Sample_IR_Stop_Transmit(IR_Repeat_Delay);
}

/**
 *******************************************************************************
 * @brief       IR Bit Data Transmit 0 (for IR NEC protocol)
 * @par         Example
 * @code
                Sample_IR_DAT0_Transmit();
 * @endcode
 *******************************************************************************
 */
void Sample_IR_DAT0_Transmit(void)
{
    TM_TimeBaseInitTypeDef TM_TMBaseInitStruct;
    
    // ----------------------------------------------------
    /* TM36 Initial */
    TM_TMBaseInitStruct.TM_Prescaler = 0x0000U;
    TM_TMBaseInitStruct.TM_Period = IR_DAT0_CountRange;
    TM_TMBaseInitStruct.TM_CounterMode = Cascade;
    TM_TMBaseInitStruct.TM_MainClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_2ndClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_IntClockDivision = TM_IntDIV4;
    TM_TMBaseInitStruct.TM_MainClockSource = TM_CK_INT;
    TM_TMBaseInitStruct.TM_2ndClockSource = TM_CK_INT;
    TM_TimeBase_Init(TM36, &TM_TMBaseInitStruct);
    
    // ----------------------------------------------------
    /* TM36 PWM comparator value set */
    TM_SetCC0A(TM36, IR_DAT0_PULSE1);
    TM_SetCC0B(TM36, IR_DAT0_PULSE1);
    
    // ----------------------------------------------------
    /* Clear TM36 comparator flag and enable TM36 counter */
    TM_ClearFlag(TM36, TMx_CF0A);
    TM_Timer_Cmd(TM36,ENABLE);
    
    // ----------------------------------------------------
    /* TM36 enven check */
    while(TM_GetSingleFlagStatus(TM36, TMx_CF0A)!=DRV_Happened);
    while(TM_GetSingleFlagStatus(TM36, TMx_TOF)!=DRV_Happened);
    
    // ----------------------------------------------------
    /* Clear TM36 event flag and disable TM36 */
    TM_ClearFlag(TM36, (TMx_CF0A|TMx_TOF));
    TM_Timer_Cmd(TM36,DISABLE);
}

/**
 *******************************************************************************
 * @brief       IR Bit Data Transmit 1 (for IR NEC protocol)
 * @par         Example
 * @code
                Sample_IR_DAT1_Transmit();
 * @endcode
 *******************************************************************************
 */
void Sample_IR_DAT1_Transmit(void)
{
    TM_TimeBaseInitTypeDef TM_TMBaseInitStruct;
    
    // ----------------------------------------------------
    /* TM36 Initial */
    TM_TMBaseInitStruct.TM_Prescaler = 0x0000U;
    TM_TMBaseInitStruct.TM_Period = IR_DAT1_CountRange;
    TM_TMBaseInitStruct.TM_CounterMode = Cascade;
    TM_TMBaseInitStruct.TM_MainClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_2ndClockDirection = TM_UpCount;
    TM_TMBaseInitStruct.TM_IntClockDivision = TM_IntDIV4;
    TM_TMBaseInitStruct.TM_MainClockSource = TM_CK_INT;
    TM_TMBaseInitStruct.TM_2ndClockSource = TM_CK_INT;
    TM_TimeBase_Init(TM36, &TM_TMBaseInitStruct);
    
    // ----------------------------------------------------
    /* TM36 PWM comparator value set */
    TM_SetCC0A(TM36, IR_DAT1_PULSE1);
    TM_SetCC0B(TM36, IR_DAT1_PULSE1);
    
    // ----------------------------------------------------
    /* Clear TM36 comparator flag and enable TM36 counter */
    TM_ClearFlag(TM36, TMx_CF0A);
    TM_Timer_Cmd(TM36,ENABLE);
    
    // ----------------------------------------------------
    /* TM36 enven check */
    while(TM_GetSingleFlagStatus(TM36, TMx_CF0A)!=DRV_Happened);
    while(TM_GetSingleFlagStatus(TM36, TMx_TOF)!=DRV_Happened);
    
    // ----------------------------------------------------
    /* Clear TM36 event flag and disable TM36 */
    TM_ClearFlag(TM36, (TMx_CF0A|TMx_TOF));
    TM_Timer_Cmd(TM36,DISABLE);
}

/**
 *******************************************************************************
 * @brief       IR Byte Transmit. (for NEC protocol)
 * @param[in]   Addr_0
 * @arg         0~255
 * @param[in]   Addr_1
 * @arg         0~255
 * @param[in]   Data_0
 * @arg         0~255
 * @param[in]   Data_1
 * @arg         0~255
 * @par         Example
 * @code
                Sample_APB_IR_DAT_Transmit(0xAA, 0x55, 0xAA, 0x55);
 * @endcode
 *******************************************************************************
 */
DRV_Return Sample_IR_DAT_Transmit_BYTE(uint8_t Addr_0, uint8_t Addr_1, uint8_t Data_0, uint8_t Data_1)
{
    uint8_t TxCNT,CodeCNT,tDLY;
    uint32_t wDATA = 0x00000000;
    
    // ----------------------------------------------------
    /* Save data to calculate delay time */
    wDATA |= Addr_0;
    wDATA |= (uint32_t)(Addr_1<<8);
    wDATA |= (uint32_t)(Data_0<<16);
    wDATA |= (uint32_t)(Data_1<<24);
    tDLY = (uint8_t)Sample_STOPDelayTime((uint32_t)wDATA);
    
    // ----------------------------------------------------
    /* Start transmit IR data */
    for(TxCNT=0;TxCNT<=5;TxCNT++)
    {
        switch(TxCNT)
        {
            case START_Code:
                            Sample_IR_Start_Transmit();
                            break;
            case Addr_Code:
                            for(CodeCNT=0;CodeCNT<=7;CodeCNT++)
                            {
                                if((Addr_0&(0x01<<CodeCNT))!=0x00U)Sample_IR_DAT1_Transmit();
                                else Sample_IR_DAT0_Transmit();
                            }
                            break;
            case BAddr_Code:
                            for(CodeCNT=0;CodeCNT<=7;CodeCNT++)
                            {
                                if((Addr_1&(0x01<<CodeCNT))!=0x00U)Sample_IR_DAT1_Transmit();
                                else Sample_IR_DAT0_Transmit();
                            }
                            break;
            case Data_Code:
                            for(CodeCNT=0;CodeCNT<=7;CodeCNT++)
                            {
                                if((Data_0&(0x01<<CodeCNT))!=0x00U)Sample_IR_DAT1_Transmit();
                                else Sample_IR_DAT0_Transmit();
                            }
                            break;
            case BData_Code:
                            for(CodeCNT=0;CodeCNT<=7;CodeCNT++)
                            {
                                if((Data_1&(0x01<<CodeCNT))!=0x00)Sample_IR_DAT1_Transmit();
                                else Sample_IR_DAT0_Transmit();
                            }
                            break;
            case STOP_Code:
                            Sample_IR_Stop_Transmit(tDLY);
                            break;
            default:
                    break;
            
        }
    }
    
    return DRV_Ready;
}
///@}

/**
 *******************************************************************************
 * @brief       IR Transmit Initial
 * @par         Example
 * @code
                Sample_APB_IR_Transmit_Init();
 * @endcode
 *******************************************************************************
 */
void Sample_APB_IR_Transmit_Init(void)
{
    //APB IR output PIN initialize.
    PIN_InitTypeDef PINX_InitStruct;
 
    PINX_InitStruct.PINX_Mode				= PINX_Mode_PushPull_O;
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Disable;
    PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;
    PINX_InitStruct.PINX_Alternate_Function = 1;
    GPIO_PinMode_Config(PINB(15),&PINX_InitStruct);
    
    //APB IR carrier clock source select
    APB_IRCLK_Select(IRCLK_TM00_CKO);
    
    //APB IR data output IR waveform module select
    APB_IRDAT_Select(IRDAT_TM36_TRGO);
    
    //IR carrier clock initialize
    Sample_TM00_CKO_OUTPUT_38KHz();
    
    //IR data output module initialize
    Sample_TM36_PWMInit();
    
}

/**
 *******************************************************************************
 * @brief       IR Transmit Sample Code
 * @par         Example
 * @code
                Sample_APB_IR_Transmit();
 * @endcode
 *******************************************************************************
 */
void Sample_APB_IR_Transmit(void)
{
    // make sure :
    
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    
    //===Set GPIO init====
    //Use Configwizard Initial GPIOB 15 is "IR_OUT", push pull output mode.
    
    //===Set TM00 init====
    //TM00_CKO output 38kHz to IR carrier clock.
    //Sample_TM00_CKO_OUTPUT_38KHz();
    
    //===Set APB Init====
    //Select IR carrier clock source
    //Select IR data
    
    // ------------------------------------------------------------------------
    // 1. GPIO Initial
    // 2. APB IR function initialize
    // 3. TM36 PWM output
    // ------------------------------------------------------------------------
    
    //APB IR transmit module initialize
    Sample_APB_IR_Transmit_Init();
    
    //IR data transmit
    if(Sample_IR_DAT_Transmit_BYTE(0x55,0xA1,0xB2,0x4D) != DRV_Ready)
    {
        //IR transmit is unready to do ...
    }
//    Sample_IR_Repeat_Transmit();
//    while(1);
}


