
/**
 ******************************************************************************
 *
 * @file        BSP_DHT_AM2302.c
 * @brief       Read AM2302 humidity and temperature Code. 
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2024/01/08
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2022 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/*==============================================================================
                                 User NOTES
How To use this function:
-----------------------
   + DHT_begin() - Initial sensor and TM20.
   + DHT_read() - Get enviroment Humidity & Temperature by AM2302.
   + DHT_readHumidity - Get Humidity.
   + DHT_readTemperature - Get Temperature. (default Celcius)
   + DHT_convertCtoF - Converts Celcius to Fahrenheit.

Example:
-----------------------
    DHT_begin();
    
    // Wait a few seconds between measurements.
    delay(2000);

    // Start AM2302 peripheral
    if (DHT_read() == FALSE)
        break;
    
    // Get Humidity
    Humidity = DHT_readHumidity();
    
    // Get Temperature
    Temperature = DHT_readTemperature(Celcius);


Driver architecture:
--------------------
   + MG32_GPIO_DRV
   + MG32_TM_DRV

Known Limitations:
------------------
   1- Run in 48MHz. (CSC control).
   2- GPIO PB4 must be GPIO mode.
==============================================================================*/

/*==============================================================================
                                 Require parameter
Require module : CSC / GPIO / TM20

CSC Module :
    Clock source                : CK_IHRCO (12M Hz)
    Clock frequency             : CK_MAIN select CK_PLLO (PLL to 48MHz)
    CK_APB frequency            : 48MHz
    CK_AHB frequency            : 48MHz
    Clock enable in 'On Mode'   : GPIOB / TM20

GPIO pin configuration : 
    Pin /   IO mode   /   AFS
    ---   ------------  --------
    PB4 / ODO+Pull up / TM20_IC0
                                           
TM20 Module :                   
    TM20 input capture 
    
==============================================================================*/

/* Includes ------------------------------------------------------------------*/
#include "BSP_DHT_AM2302.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
DHT_Structure dht;

/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/
/* audio file is Audio_raw_data.c' */


/**
 *******************************************************************************
 * @brief	    Setup sensor pins and Timer 
 * @param[in]   usec : Optionally pass pull-up time (in microseconds) before DHT reading
 * @return		No
 *******************************************************************************
 */
void DHT_begin(void) 
{
    TM_TimeBaseInitTypeDef TM_TimeBase_InitStruct;
     
    
    // ------------------------------------------------------------------------
    // Config AM2302 pin mode
    // ------------------------------------------------------------------------
    PINB(4)->CR.W = PX_CR_IOM_odo_w | PX_CR_HS_enable_w | PX_CR_PU_enable_w | PX_CR_INV_disable_w | PX_CR_ODC_level0_w | PX_CR_FDIV_bypass_w | PX_CR_AFS_af0_w;
    GPIO_CommunicationPin();
    CommunicationPin = 1;
    
    // ------------------------------------------------------------------------
    // Timer period generator via TM20 (Cascade mode)
    // ------------------------------------------------------------------------
    // 1.TimeBase structure initial
    TM_TimeBaseStruct_Init(&TM_TimeBase_InitStruct);
    
    // modify parameter
    TM_TimeBase_InitStruct.TM_Period = 0x55; 
    TM_TimeBase_InitStruct.TM_Prescaler = 0x55;
    TM_TimeBase_InitStruct.TM_CounterMode = Cascade;
    
    TM_TimeBase_Init(TM20, &TM_TimeBase_InitStruct);
    
    // ----------------------------------------------------
    // 2.clear TOF flag
    TM_ClearFlag(TM20, TMx_TOF);
        
    // ----------------------------------------------------
    // 3.Start TM20 
    TM_Timer_Cmd(TM20, ENABLE);
}

/**
 *******************************************************************************
 * @brief	    Converts Celcius to Fahrenheit.
 * @param[in]   c : value in Celcius.
 * @return		float value in Fahrenheit.
 *******************************************************************************
 */
float DHT_convertCtoF(float c) 
{ 
    return ((c * 1.8) + 32); 
}

/**
 *******************************************************************************
 * @brief	    Converts Fahrenheit to Celcius.
 * @param[in]   f : value in Fahrenheit.
 * @return		float value in Celcius.
 *******************************************************************************
 */
float DHT_convertFtoC(float f) 
{ 
    return ((f - 32) * 0.55555); 
}
/**
 *******************************************************************************
 * @brief	    Read temperature
 * @param[in]   S : Scale. Boolean value.
 *  @arg\b	    Fahrenheit : One shot 
 *	@arg\b	    Celcius : scan mode
 * @param[in]   force : true if in force mode.
 * @return		emperature value in selected scale
 *******************************************************************************
 */
float DHT_readTemperature(boolean S) 
{
    ctype HT_Convert;                           // HT_Convert : Combine Humidity / Temperature
    
    // ----------------------------------------------------------------
    //  HT[2] = Temp. high byte
    //  LT[3] = Temp. low byte
    // ----------------------------------------------------------------
    // Temperature resolution of 16Bit, the previous high; temperature 
    // sensor string value is 10 times the actual temperature value;
    // The temperature is the highest bit (Bit15) is equal to 1 indicates 
    // a negative temperature, the temperature is the highest bit (Bit15) 
    // is equal to 0 indicates a positive temperature;
    // Temperature in addition to the most significant bit (Bit14 ~ bit 0) temperature values.
    // ----------------------------------------------------------------
    HT_Convert.B[1] = (uint8_t) (dht.data[Temperature_HighByte_IDX]);
    HT_Convert.B[0] = (uint8_t) (dht.data[Temperature_LowByte_IDX]);
    
    if (S == Fahrenheit) 
        return DHT_convertCtoF((float) (HT_Convert.H[0] / 10.0));
    
    return (float) (HT_Convert.H[0] / 10.0);
}


/**
 *******************************************************************************
 * @brief	    Read Humidity.
 * @return		float value - humidity in percent
 *******************************************************************************
 */
float DHT_readHumidity(void) 
{
    ctype HT_Convert;                           // HT_Convert : Combine Humidity / Temperature
    
    // ------------------------------------------------------------------------
    // Humidity resolution of 16Bit, the previous high; humidity sensor 
    // string value is 10 times the actual humidity values.
    // ------------------------------------------------------------------------
    HT_Convert.B[1] = (uint8_t) (dht.data[Humidity_HighByte_IDX]);
    HT_Convert.B[0] = (uint8_t) (dht.data[Humidity_LowByte_IDX]);
    
    return (float) (HT_Convert.H[0] / 10.0);
}

/**
 *******************************************************************************
 * @brief	    Read value from sensor or return last one from less than two seconds.
 * @param[in]   temperature : 
 *  @arg\b	    force : true if using force mode
 * @return		float value
 *******************************************************************************
 */
boolean DHT_read(void) 
{
    uint8_t dat, i, j, ChkSum;
    
    // Reset 40 bits of received data to zero.
    dht.data[0] = dht.data[1] = dht.data[2] = dht.data[3] = dht.data[4] = 0;
 
    GPIO_CommunicationPin();                    // Normal GPIO configuration
    CommunicationPin = 1;                       // CommunicationPin = PB4 = High
    
    // ------------------------------------------------------------------------
    // (Start signal)
    // 1. Pull low 800us to wakeup AM2302 
    //      Host the start signal down time : min=0.8ms / typ=1ms / Max=20ms
    // ------------------------------------------------------------------------
//    Timer Alarm (Time800us);                  // count ~800us  
    TM20->PSARR.H[0]    = TimerPrescaler_800us;
    TM20->ARR.H[0]      = TimerCounter_800us;
    TM20->CNT.H[0]      = 0;
    TM20->PSCNT.H[0]    = 0;    
    TM_ClearFlag(TM20, 0xFFFFFFFF);

    CommunicationPin = 0;                       // CommunicationPin = PB4 = Low
    
    while(Timer_GetTimeout() == DRV_UnHappened);
    
    // ------------------------------------------------------------------------
    // (Release)
    // 2. release SDA pin 20us 
    //      Bus master has released time : min=20us / typ=30us / Max=200us
    // ------------------------------------------------------------------------
    CommunicationPin = 1;                       // CommunicationPin = PB4 = High

    // ------------------------------------------------------------------------
    // (Response signal)
    // 3.   Response to low time : min=75us / typ=80us / Max=85us
    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
    // 4.   In response to high time : min=75us / typ=80us / Max=85us
    // ------------------------------------------------------------------------
//    Timer_CapturePolling_Init(Capture_us);      // Config to TM20_IC0 & Capture mode    
                                                // TM20_IC0 from PB4 (Dual edge capture)
                                                // First capture edge event -> Trigger startup TM20 counter
    // Set CommunicationPin AFS state (TM20_IC0)
    GPIO_PinFunction_Select(PINB(4), TM20_IC0_Pin_AFS);
    
    // Disable TM20 & clear flag
    TM_Timer_Cmd(TM20, DISABLE);
    TM_ClearFlag(TM20, 0xFFFFFFFF);
    
    // Set TM20 CNT & PreCNT startup condition
    TM_TriggerSource_Select(TM20,TRGI_IN0);
    TM_TRGICounter_Select(TM20, TRGI_StartupFalling);
    TM_TRGIPrescaler_Select(TM20, TRGI_StartupFalling);
    
    // Config TM20_IC0 is capture function
    TM_CH0Function_Config(TM20, TM_InputCapture);    
    TM_IN0Source_Select(TM20, IC0);            
    TM_IN0TriggerEvent_Select(TM20, IC_DualEdge);
    
    // Keep capture data
    TM_IC0OverWritten_Cmd(TM20, TM_Keep);
    
    //
    TM20->CNT.H[0]      = 0;
    TM20->PSCNT.H[0]    = 0;    
    TM20->PSARR.H[0]    = TimerPrescaler_800us;
    TM20->ARR.H[0]      = 0xFFFF;
    
    //
    TM_Timer_Cmd(TM20, ENABLE);
    {
        // --------------------------------------------------------------------
        // first capture event (Low to High)
        // --------------------------------------------------------------------
        while(Timer_Get1stCapture() == DRV_UnHappened);

        Timer_LowEventResetCounter();           // Low event to reset TM20 counter (TM20 TRGI function)
        
        // --------------------------------------------------------------------
        // second capture event (High to Low)
        // --------------------------------------------------------------------
        while(Timer_Get2ndCapture() == DRV_UnHappened);
        
        // Clear TM20's STA & calculate TM20 capture period. (TM20_CC0B = TM20_CC0B - TM20_CC0A)
        Timer_Corrected();
        
        // --------------------------------------------------------------------
        // Check rule: 
        //  1. Low to High time 80us 
        //  2. High to Low time 80us
        // --------------------------------------------------------------------
        if (Timer_GetDelta() > 5)               // if (TM20->CC0A.H[0] - TM20->CC0B.H[0]) then 
            return FALSE;                       //  function fail -> continue again.
        
        // --------------------------------------------------------------------
        // check rule:
        //  CC0A (High level time) = CC0B (Low level time)
        // --------------------------------------------------------------------
        if ((Timer_GetHighCount() < 75) || (Timer_GetHighCount() > 85))
            return FALSE;                       //  function fail -> continue again.
    
    }
        
    // ------------------------------------------------------------------------
    // (Data format)
    // 5.   Pooling CommunicationPin state 
    //          5.1 Signal "0", "1" low time : min=48us / type=50us / Max=55us
    //          5.2 Signal "0" high time : min=22us / typ=26us / Max=30us
    //          5.3 Signal "1" high time : min=68us / typ=70us / Max=75us
    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
    // Total receive 5 bytes
    // 1st byte : Humidity high byte
    // 2nd byte : Humidity low byte
    // 3rd byte : Temp. high byte
    // 4th byte : Temp. low byte
    // 5th byte : Check Sum verify - (Parity bit)
    // ------------------------------------------------------------------------
    for (i=0; i<5; i++)
    {
        // --------------------------------------------------------------------
        // MSB first
        //  bit 0 : Low 50us High 26~28us
        //  bit 1 : Low 50us High 70us
        // --------------------------------------------------------------------
        for (j = dat = 0; j < 8; j++)
        {
            dat = (uint8_t) (dat << 1);
            
            // until assert dual capture event
            while(Timer_Get2ndCapture() == DRV_UnHappened);

            // Clear TM20's STA & calculate TM20 capture period. (TM20_CC0B = TM20_CC0B - TM20_CC0A)
            // TM20_CC0A = Low to High time
            // TM20_CC0B = High to Low time
            Timer_Corrected();
            
            // if (TM20->CC0A.H[0] > TM20->CC0B.H[0]) then ...
            if (Timer_HmoreL())
                dat |= 0x00;                    // bit 0 : Low 50us High 26~28us
            else    
                dat |= 0x01;                    // bit 1 : Low 50us High 70us
        
        }
        dht.data[i] = dat;
    }
    
    // 
    GPIO_CommunicationPin();

    // ------------------------------------------------------------------------
    // Check CHKSUM & Output sensor value
    //  HT[0] : Humidity high byte
    //  HT[1] : Humidity low byte
    //  HT[2] : Temp. high byte
    //  HT[3] : Temp. low byte
    //  HT[4] : Check Sum verify
    // ------------------------------------------------------------------------
    
    // Checksum rule check
    ChkSum = dht.data[ Humidity_HighByte_IDX    ] 
           + dht.data[ Humidity_LowByte_IDX     ] 
           + dht.data[ Temperature_HighByte_IDX ] 
           + dht.data[ Temperature_LowByte_IDX  ];
        
    if (ChkSum != dht.data[ ChecksumByte_IDX ])
        return FALSE;
    else
        return TRUE;
        
}

