/**
  ******************************************************************************
 *
 * @file        Sample_MID_SPI_Master_TXRX.c
 *
 * @brief       SPIx/URTx master transmit and receive sample code
 *
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2021/05/27
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2020 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */ 


/* Includes ------------------------------------------------------------------*/
#include "MG32_SPI_MID.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define SPI0_NSS            PX(GPIOB_Base,0)                                            /*!< SPI0 NSS pin*/
#define URT0_NSS            PX(GPIOB_Base,10)                                           /*!< URT0 NSS pin*/

/* Private macro -------------------------------------------------------------*/
#define PX(GPIOX,PinNum)    *((volatile uint8_t*)((uint32_t)GPIOX + PinNum + 0x10))     /*!< GPIOX pin (PinNum) read /write */

/* Private variables ---------------------------------------------------------*/
static SPI_HandleTypeDef     mSPI0;

/* Private function prototypes -----------------------------------------------*/
void Sample_MID_SPI_Master_Transfer (MID_SPI_Struct* MID_Module);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/



/**
 *******************************************************************************
 * @brief       SPI0/URTx master transmit and receive.
 * @details     1.Initial SPI/URTx.
 *    \n        2.Setting TX/RX Buffer Transmit data.
 *    \n        3.Transmit data.
 *    \n        4.Receive data.
 *    \n        5.Transmit data.
 *    \n        6.Receive data.
 * @return      None
 *******************************************************************************
 */
void Sample_MID_SPI_Master_Transfer (MID_SPI_Struct* MID_Module)
{  
    uint8_t     TX_BUF8[63]={0};
    uint8_t     RX_BUF8[63]={0};
    uint8_t     CNT;



    // make sure :
    //===Set CSC init===
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12M
    //Select CK_MAIN Source = CK_HS
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->SPI0 = Enable
    //Configure Peripheral On Mode Clock->URT0 = Enable
    //Configure Peripheral On Mode Clock->URT1 = Enable
    //Configure Peripheral On Mode Clock->GPIOB = Enable

    /*===Set GPIO init SPI0 for SPI0===*/
    //1. MOSI Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin3
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin3 mode is PPO (Push pull output)
    //    (4).Pin3 pull-up resister Enable
    //    (5).Pin3 function SPI0_MOSI
    //2. MISO Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin1
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin1 mode is DIN (Data input)
    //    (4).Pin1 pull-up resister Enable
    //    (5).Pin1 function SPI0_MISO
    //3. SPICLK Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin2
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin2 mode is PPO (Push pull output)
    //    (4).Pin2 pull-up resister Enable
    //    (5).Pin2 function SPI0_CLK
    //4. NSS Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin10
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin0 mode is PPO (Push pull output)
    //    (4).Pin0 pull-up resister Enable
    //    (5).Pin0 function GPB0

    /*===Set GPIO init SPI0 for URT0===*/
    //1. MOSI Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin8
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin8 mode is PPO (Push pull output)
    //    (4).Pin8 pull-up resister Enable
    //    (5).Pin8 function URT0_TX
    //2. MISO Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin9
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin9 mode is DIN (Data input)
    //    (4).Pin9 pull-up resister Enable
    //    (5).Pin9 function URT0_RX
    //3. SPICLK Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOC->Pin3
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin3 mode is PPO (Push pull output)
    //    (4).Pin3 pull-up resister Enable
    //    (5).Pin3 function URT0_CLK
    //4. NSS Pin
    //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin10
    //    (2).GPIO port initial is 0xFFFF
    //    (3).Pin10 mode is PPO (Push pull output)
    //    (4).Pin10 pull-up resister Enable
    //    (5).Pin10 function GPB10

    // ------------------------------------------------------------------------
    // 1.Initial SPI/URTx
    // ------------------------------------------------------------------------
    mSPI0.Instance = MID_Module;
    mSPI0.Init.Mode = SPI_MODE_MASTER;
    mSPI0.Init.NSSPMode = SPI_NSS_PULSE_DISABLE;
    mSPI0.Init.NSS = SPI_NSS_OUT_HARDWARE;
    mSPI0.Init.DataLine = SPI_STANDARD_SPI;
    mSPI0.Init.DataSize = SPI_DATASIZE_8BIT;
    mSPI0.Init.CLKPolarity = SPI_POLARITY_LOW;
    mSPI0.Init.CLKPhase = SPI_PHASE_1EDGE;
    mSPI0.Init.ClockDivider = SPI_CLOCKDIVIDER_32;
    mSPI0.Init.FirstBit = SPI_FIRSTBIT_MSB;
    mSPI0.Init.NSSPMode = SPI_NSS_PULSE_DISABLE;
    if (MID_SPI_Init(&mSPI0) != MID_OK)             // When SPI initial failure, stop at here.
    {
        while(1);
    }

    // ------------------------------------------------------------------------
    // 2.Setting TX/RX Buffer Transmit data
    // ------------------------------------------------------------------------
    for(CNT=0; CNT<63; CNT++)                       
    {
        TX_BUF8[CNT] = CNT;     // Set TX_BUF data
        RX_BUF8[CNT] = 0;       // Clear RX_BUF data
    }

    // ------------------------------------------------------------------------
    // 3.Transmit data
    // ------------------------------------------------------------------------
    URT0_NSS = CLR;
    SPI0_NSS = CLR; 
    MID_SPI_Transmit(&mSPI0, &TX_BUF8[0], 63, 0x80000000);

    for(CNT=50; CNT>0; CNT--)   // Delay
        __NOP();
    
    // ------------------------------------------------------------------------
    // 4.Receive data
    // ------------------------------------------------------------------------
    MID_SPI_Receive(&mSPI0, &RX_BUF8[0], 63, 0xf0000000);
    SPI0_NSS = SET;
    URT0_NSS = SET;
    
    for(CNT=50; CNT>0; CNT--)   // Delay
        __NOP();
    
    // ------------------------------------------------------------------------
    // 5.Transmit data
    // ------------------------------------------------------------------------
    URT0_NSS = CLR;
    SPI0_NSS = CLR; 
    MID_SPI_Transmit(&mSPI0, &TX_BUF8[0], 63, 0x80000000);
    SPI0_NSS = SET;
    URT0_NSS = SET;

    for(CNT=50; CNT>0; CNT--)   // Delay
        __NOP();
    
    // ------------------------------------------------------------------------
    // 6.Receive data
    // ------------------------------------------------------------------------
    URT0_NSS = CLR;
    SPI0_NSS = CLR;
    MID_SPI_Receive(&mSPI0, &RX_BUF8[0], 63, 0xf0000000);
    SPI0_NSS = SET;
    URT0_NSS = SET;
}



