/*
*********************************************************************************************************
*                                     MICIRUM BOARD SUPPORT PACKAGE
*
*                            (c) Copyright 2007-2012; Micrium, Inc.; Weston, FL
*
*                   All rights reserved.  Protected by international copyright laws.
*                   Knowledge of the source code may not be used to write a similar
*                   product.  This file may only be used in accordance with a license
*                   and should not be redistributed in any way.
*********************************************************************************************************
*/

/*
*********************************************************************************************************
*
*                                        BOARD SUPPORT PACKAGE
*
*                                           megawin MG32
*                                              on the
*
*                                          MG32F02A132-EVAL
*                                          Evaluation Board
*
* Filename      : bsp_periph.c
* Version       : V1.00
* Programmer(s) : MD
*
*********************************************************************************************************
*/

/*
*********************************************************************************************************
*                                             INCLUDE FILES
*********************************************************************************************************
*/

#define  BSP_PERIPH_MODULE
#include <bsp.h>


/*
*********************************************************************************************************
*                                            LOCAL DEFINES
*********************************************************************************************************
*/

#define BSP_PERIPH_REG_CSC_BASE                  0x4C010000
#define BSP_PERIPH_REG_CSC_STA                   (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x000))
#define BSP_PERIPH_REG_CSC_INT                   (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x004))
#define BSP_PERIPH_REG_CSC_PLL                   (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x008))
#define BSP_PERIPH_REG_CSC_KEY                   (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x00C))
#define BSP_PERIPH_REG_CSC_CR0                   (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x010))
#define BSP_PERIPH_REG_CSC_DIV                   (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x014))
#define BSP_PERIPH_REG_CSC_CKO                   (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x018))
#define BSP_PERIPH_REG_CSC_AHB                   (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x01C))
#define BSP_PERIPH_REG_CSC_APB0                  (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x020))
#define BSP_PERIPH_REG_CSC_APB1                  (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x024))
#define BSP_PERIPH_REG_CSC_SLP0                  (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x030))
#define BSP_PERIPH_REG_CSC_SLP1                  (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x034))
#define BSP_PERIPH_REG_CSC_STP0                  (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x038))
#define BSP_PERIPH_REG_CSC_STP1                  (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x03C))
#define BSP_PERIPH_REG_CSC_CKS0                  (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x040))
#define BSP_PERIPH_REG_CSC_CKS1                  (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x044))
#define BSP_PERIPH_REG_CSC_CKS2                  (*(CPU_REG32 *)(BSP_PERIPH_REG_CSC_BASE + 0x048))


#define BSP_PERIPH_REG_CSC_CKS0_ADC0             CSC_CKS0_ADC0_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS0_CMP              CSC_CKS0_CMP_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS0_DAC              CSC_CKS0_DAC_CKS_mask_w

#define BSP_PERIPH_REG_CSC_CKS1_I2C0             CSC_CKS1_I2C0_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS1_I2C1             CSC_CKS1_I2C1_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS1_SPI0             CSC_CKS1_SPI0_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS1_URT0             CSC_CKS1_URT0_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS1_URT1             CSC_CKS1_URT1_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS1_URT2             CSC_CKS1_URT2_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS1_URT3             CSC_CKS1_URT3_CKS_mask_w

#define BSP_PERIPH_REG_CSC_CKS2_TM00             CSC_CKS2_TM00_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS2_TM01             CSC_CKS2_TM01_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS2_TM10             CSC_CKS2_TM10_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS2_TM16             CSC_CKS2_TM16_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS2_TM20             CSC_CKS2_TM20_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS2_TM26             CSC_CKS2_TM26_CKS_mask_w
#define BSP_PERIPH_REG_CSC_CKS2_TM36             CSC_CKS2_TM36_CKS_mask_w

/*
*********************************************************************************************************
*                                           LOCAL CONSTANTS
*********************************************************************************************************
*/


/*
*********************************************************************************************************
*                                          LOCAL DATA TYPES
*********************************************************************************************************
*/


/*
*********************************************************************************************************
*                                            LOCAL TABLES
*********************************************************************************************************
*/


/*
*********************************************************************************************************
*                                       LOCAL GLOBAL VARIABLES
*********************************************************************************************************
*/


/*
*********************************************************************************************************
*                                      LOCAL FUNCTION PROTOTYPES
*********************************************************************************************************
*/


/*
*********************************************************************************************************
*                                     LOCAL CONFIGURATION ERRORS
*********************************************************************************************************
*/


/*
*********************************************************************************************************
*                                         BSP_PeriphClkFreqGet()
*
* Description : Get clock frequency of a peripheral.
*
* Argument(s) : pwr_clk_id      Power/clock ID.
*
* Return(s)   : clk_freq or 0.
*
* Caller(s)   : Application.
*
* Note(s)     : AHB Periphs divider chosen with HPRE[3:0] bits 7-4. APB Periphs
*               divider chosen with PPRE[2:0] bits 10-8.
*
*              ---PPRE PCLK Prescaler---        ---HPRE HCLK Prescaler---
*              0xx: HCLK not divided            0xxx: SYSCLK not divided
*              100: HCLK divided by 2           1000: SYSCLK divided by 2
*              101: HCLK divided by 4           1001: SYSCLK divided by 4
*              110: HCLK divided by 8           1010: SYSCLK divided by 8
*              111: HCLK divided by 16          1011: SYSCLK divided by 16
*                                               1100: SYSCLK divided by 64
*                                               1101: SYSCLK divided by 128
*                                               1110: SYSCLK divided by 256
*                                               1111: SYSCLK divided by 512
*
*********************************************************************************************************
*/

CPU_INT32U BSP_PeriphClkFreqGet(CPU_DATA  pwr_clk_id)
{
    CSC_ClockTypeDef lCSC_Clock;
    BSP_ClkFreq(&lCSC_Clock);

    if(pwr_clk_id >= 96)
        return 0;
    else
    {
        if(pwr_clk_id >= 64)
        {
            if((BSP_PERIPH_REG_CSC_APB1 & DEF_BIT(pwr_clk_id - 64)) == 0)
                return 0;
        }
        else
        {
            if(pwr_clk_id >= 32)
            {
                if((BSP_PERIPH_REG_CSC_APB0 & DEF_BIT(pwr_clk_id - 32)) == 0)
                    return 0;
            }
            else 
            {
                if(pwr_clk_id < 32)
                {
                    if((BSP_PERIPH_REG_CSC_AHB & DEF_BIT(pwr_clk_id)) == 0)
                        return 0;
                }
            }
        }
    }

    switch (pwr_clk_id) {
                                                                /*  AHB values for HCLK                                 */
        /* AHB Enable Peripherals                               */
        case BSP_PERIPH_ID_IOPAEN:
        case BSP_PERIPH_ID_IOPBEN:
        case BSP_PERIPH_ID_IOPCEN:
        case BSP_PERIPH_ID_IOPDEN:
        case BSP_PERIPH_ID_IOPEEN:
        case BSP_PERIPH_ID_IOPFEN:
        case BSP_PERIPH_ID_IOPGEN:
        case BSP_PERIPH_ID_IOPHEN:
        case BSP_PERIPH_ID_GPLEN:
        case BSP_PERIPH_ID_EMBEN:
        case BSP_PERIPH_ID_DMAEN:
            return (lCSC_Clock.CK_AHB);

        /*  APB0 values for PCLK */
        case BSP_PERIPH_ID_ADC0EN:
            if((BSP_PERIPH_REG_CSC_CKS0 & BSP_PERIPH_REG_CSC_CKS0_ADC0) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

        case BSP_PERIPH_ID_CMPEN:
            if((BSP_PERIPH_REG_CSC_CKS0 & BSP_PERIPH_REG_CSC_CKS0_CMP) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

      #if defined(MG32F02A132) || defined(MG32F02A072) || defined(MA862)
        case BSP_PERIPH_ID_DACEN:
            if((BSP_PERIPH_REG_CSC_CKS0 & BSP_PERIPH_REG_CSC_CKS0_DAC) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);
      #endif

        case BSP_PERIPH_ID_RTCEN:
        case BSP_PERIPH_ID_IWDTEN:
        case BSP_PERIPH_ID_WWDTEN:
            return(lCSC_Clock.CK_APB);

        case BSP_PERIPH_ID_I2C0EN:
            if((BSP_PERIPH_REG_CSC_CKS1 & BSP_PERIPH_REG_CSC_CKS1_I2C0) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

      #if defined(MG32F02A132) || defined(MG32F02A072) || defined(MA862)
        case BSP_PERIPH_ID_I2C1EN:
            if((BSP_PERIPH_REG_CSC_CKS1 & BSP_PERIPH_REG_CSC_CKS1_I2C1) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);
      #endif

        case BSP_PERIPH_ID_SPI0EN:
            if((BSP_PERIPH_REG_CSC_CKS1 & BSP_PERIPH_REG_CSC_CKS1_SPI0) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

        case BSP_PERIPH_ID_URT0:
            if((BSP_PERIPH_REG_CSC_CKS1 & BSP_PERIPH_REG_CSC_CKS1_URT0) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

        case BSP_PERIPH_ID_URT1:
            if((BSP_PERIPH_REG_CSC_CKS1 & BSP_PERIPH_REG_CSC_CKS1_URT1) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

      #if defined(MG32F02A132) || defined(MG32F02A072) || defined(MA862)
        case BSP_PERIPH_ID_URT2:
            if((BSP_PERIPH_REG_CSC_CKS1 & BSP_PERIPH_REG_CSC_CKS1_URT2) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

        case BSP_PERIPH_ID_URT3:
            if((BSP_PERIPH_REG_CSC_CKS1 & BSP_PERIPH_REG_CSC_CKS1_URT3) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);
      #endif

      #if 0
        case BSP_PERIPH_ID_USB:
            if((BSP_PERIPH_REG_CSC_CKS1 & BSP_PERIPH_REG_CSC_CKS1_USB) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);
      #endif

                                                               /*  APB1 values for PCLK */
        case BSP_PERIPH_ID_TM00EN:
            if((BSP_PERIPH_REG_CSC_CKS2 & BSP_PERIPH_REG_CSC_CKS2_TM00) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

        case BSP_PERIPH_ID_TM01EN:
            if((BSP_PERIPH_REG_CSC_CKS2 & BSP_PERIPH_REG_CSC_CKS2_TM01) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

        case BSP_PERIPH_ID_TM10EN:
            if((BSP_PERIPH_REG_CSC_CKS2 & BSP_PERIPH_REG_CSC_CKS2_TM10) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

        case BSP_PERIPH_ID_TM16EN:
            if((BSP_PERIPH_REG_CSC_CKS2 & BSP_PERIPH_REG_CSC_CKS2_TM16) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

      #if defined(MG32F02A132) || defined(MG32F02A072) || defined(MA862)
        case BSP_PERIPH_ID_TM20EN:
            if((BSP_PERIPH_REG_CSC_CKS2 & BSP_PERIPH_REG_CSC_CKS2_TM20) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);

        case BSP_PERIPH_ID_TM26EN:
            if((BSP_PERIPH_REG_CSC_CKS2 & BSP_PERIPH_REG_CSC_CKS2_TM26) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);
      #endif

        case BSP_PERIPH_ID_TM36EN:
            if((BSP_PERIPH_REG_CSC_CKS2 & BSP_PERIPH_REG_CSC_CKS2_TM36) == 0)
                return(lCSC_Clock.CK_APB);
            else
                return(lCSC_Clock.CK_AHB);
    }
    return ((CPU_INT32U)0);
}


/*
*********************************************************************************************************
*                                             BSP_PeriphEn()
*
* Description : Enable clock for peripheral.
*
* Argument(s) : pwr_clk_id      Power/clock ID.
*
* Return(s)   : none.
*
* Caller(s)   : Application.
*
* Note(s)     : none.
*********************************************************************************************************
*/

void  BSP_PeriphEn (CPU_DATA  pwr_clk_id)
{
    BSP_PERIPH_REG_CSC_KEY = 0xA217UL;
    switch (pwr_clk_id) {
         /* AHB Enable Peripherals                               */
         case BSP_PERIPH_ID_IOPAEN:
         case BSP_PERIPH_ID_IOPBEN:
         case BSP_PERIPH_ID_IOPCEN:
         case BSP_PERIPH_ID_IOPDEN:
         case BSP_PERIPH_ID_IOPEEN:
         case BSP_PERIPH_ID_IOPFEN:
         case BSP_PERIPH_ID_IOPGEN:
         case BSP_PERIPH_ID_IOPHEN:
         case BSP_PERIPH_ID_GPLEN:
         case BSP_PERIPH_ID_EMBEN:
         case BSP_PERIPH_ID_DMAEN:
              BSP_PERIPH_REG_CSC_AHB |= DEF_BIT(pwr_clk_id);
              break;

         /* APB0 Enable Peripherals                              */
         case BSP_PERIPH_ID_ADC0EN:
         case BSP_PERIPH_ID_CMPEN:
         case BSP_PERIPH_ID_DACEN:
         case BSP_PERIPH_ID_RTCEN:
         case BSP_PERIPH_ID_IWDTEN:
         case BSP_PERIPH_ID_WWDTEN:
         case BSP_PERIPH_ID_I2C0EN:
         case BSP_PERIPH_ID_I2C1EN:
         case BSP_PERIPH_ID_SPI0EN:
         case BSP_PERIPH_ID_URT0:
         case BSP_PERIPH_ID_URT1: 
         case BSP_PERIPH_ID_URT2: 
         case BSP_PERIPH_ID_URT3: 
              BSP_PERIPH_REG_CSC_APB0 |= DEF_BIT(pwr_clk_id - 32);
              break;

         /* APB1 Enable Peripherals                              */
         case BSP_PERIPH_ID_USB:
         case BSP_PERIPH_ID_TM00EN:
         case BSP_PERIPH_ID_TM01EN:
         case BSP_PERIPH_ID_TM10EN:
         case BSP_PERIPH_ID_TM16EN:
         case BSP_PERIPH_ID_TM20EN:
         case BSP_PERIPH_ID_TM26EN:
         case BSP_PERIPH_ID_TM36EN:
              BSP_PERIPH_REG_CSC_APB1 |= DEF_BIT(pwr_clk_id - 64);
              break;
    }
    BSP_PERIPH_REG_CSC_KEY = 0x0UL;
}


/*
*********************************************************************************************************
*                                             BSP_PeriphDis()
*
* Description : Disable clock for peripheral.
*
* Argument(s) : pwr_clk_id      Power/clock ID.
*
* Return(s)   : none.
*
* Caller(s)   : Application.
*
* Note(s)     : none.
*********************************************************************************************************
*/

void  BSP_PeriphDis (CPU_DATA  pwr_clk_id)
{
    BSP_PERIPH_REG_CSC_KEY = 0xA217UL;
    switch (pwr_clk_id) {
         /* AHB Enable Peripherals                               */
         case BSP_PERIPH_ID_IOPAEN:
         case BSP_PERIPH_ID_IOPBEN:
         case BSP_PERIPH_ID_IOPCEN:
         case BSP_PERIPH_ID_IOPDEN:
         case BSP_PERIPH_ID_IOPEEN:
         case BSP_PERIPH_ID_IOPFEN:
         case BSP_PERIPH_ID_IOPGEN:
         case BSP_PERIPH_ID_IOPHEN:
         case BSP_PERIPH_ID_GPLEN:
         case BSP_PERIPH_ID_EMBEN:
         case BSP_PERIPH_ID_DMAEN:
              BSP_PERIPH_REG_CSC_AHB &= ~DEF_BIT(pwr_clk_id);
              break;

         /* APB0 Enable Peripherals                              */
         case BSP_PERIPH_ID_ADC0EN:
         case BSP_PERIPH_ID_CMPEN:
         case BSP_PERIPH_ID_DACEN:
         case BSP_PERIPH_ID_RTCEN:
         case BSP_PERIPH_ID_IWDTEN:
         case BSP_PERIPH_ID_WWDTEN:
         case BSP_PERIPH_ID_I2C0EN:
         case BSP_PERIPH_ID_I2C1EN:
         case BSP_PERIPH_ID_SPI0EN:
         case BSP_PERIPH_ID_URT0:
         case BSP_PERIPH_ID_URT1: 
         case BSP_PERIPH_ID_URT2: 
         case BSP_PERIPH_ID_URT3:
              BSP_PERIPH_REG_CSC_APB0 &= ~DEF_BIT(pwr_clk_id - 32);
              break;

         /* APB1 Enable Peripherals                              */
         case BSP_PERIPH_ID_USB:
         case BSP_PERIPH_ID_TM00EN:
         case BSP_PERIPH_ID_TM01EN:
         case BSP_PERIPH_ID_TM10EN:
         case BSP_PERIPH_ID_TM16EN:
         case BSP_PERIPH_ID_TM20EN:
         case BSP_PERIPH_ID_TM26EN:
         case BSP_PERIPH_ID_TM36EN:
              BSP_PERIPH_REG_CSC_APB1 &= ~DEF_BIT(pwr_clk_id - 64);
              break;
    }
    BSP_PERIPH_REG_CSC_KEY = 0x0UL;
}
