/*********************************************************************
*    Project: MG82F5B32-DEMO
*			MG82F5B32 LQFP32_V10 EV Board (TH156A)
*			CpuCLK=12MHz, SysCLK=12MHz
*    Description:
*			TWSI0 (SCL/P40, SDA/P41) Slave interrupt
*			TWSI1 (SCL/P10, SDA/P11) Slave interrupt
*    Note:
*
*    Creat time:
*    Modify:
*    
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MG82F5Bxx.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F5B32.H"
#include ".\include\API_Uart_BRGRL_MG82F5B32.H"

/*************************************************
*Set SysClk (MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.32MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define LED_G_0		P25
#define LED_R		P26
#define LED_G_1		P27

#define	TWI_OV_TIME_MAX			20

#define TWI0_SCL		P40
#define TWI0_SDA		P41

#define TWI1_SCL		P10
#define TWI1_SDA		P11

#define SLAVE_ADDRESS	0xA0
#define TEST_BUF_SIZE	16

#define	TWI_OV_TIME_MAX			20

u8	TWI0OvTime;
u8	TWI1OvTime;
bit bTWI0Error;
bit bTWI1Error;
bit bTWI0ADFlag;
bit bTWI1ADFlag;

bit bTWI0RXFlag;
bit bTWI1RXFlag;

WordTypeDef TWI0RegAddr;
WordTypeDef TWI1RegAddr;
xdata u8 TWI0Buf[TEST_BUF_SIZE];
xdata u8 TWI1Buf[TEST_BUF_SIZE];
u8 TWI0Inx;
u8 TWI1Inx;
bit bES0;
u8 LedTime;

/***********************************************************************************
*Function:   	void INT_T0(void)
*Description:	T0 Interrupt handler
*Input:   
*Output:     
*************************************************************************************/
void INT_T0(void) interrupt INT_VECTOR_T0
{
	TH0=TIMER_12T_1ms_TH;
	TL0=TIMER_12T_1ms_TL;
	if(TWI0OvTime!=0) 
	{
		TWI0OvTime--;
		if(TWI0OvTime==0) bTWI0Error=TRUE;
	}
	if(TWI1OvTime!=0) 
	{
		TWI1OvTime--;
		if(TWI1OvTime==0) bTWI1Error=TRUE;
	}
	if(LedTime!=0) LedTime--;
}

/***********************************************************************************
*Function:   	void INT_TWI(void)
*Description:	TWI0 Interrupt handler
*		 
*Input:   
*Output:     
*************************************************************************************/
void INT_TWSI0() interrupt INT_VECTOR_TWI0
{
	_push_(SFRPI);
	SFR_SetPage(0);
	if(SISTA==0x80)
	{ // 0x80: DATA has been received. ACK has been returned.
		TWI0OvTime = TWI_OV_TIME_MAX;			// restore TWI0 overtime
		if(bTWI0ADFlag==0)
		{// Reg Address
			if(TWI0Inx==0)
			{
				TWI0RegAddr.B.BHigh=SIDAT;
				TWI0Inx++;
			}
			else 
			{
				TWI0RegAddr.B.BLow=SIDAT;
				bTWI0ADFlag=1;					// The next byte is Data
				TWI0Inx=0;
			}
		}
		else
		{
			TWI0Buf[TWI0Inx]=SIDAT;				// Save received data into buffer
			TWI0Inx++;
			if(TWI0Inx>=TEST_BUF_SIZE) 
			{
				bTWI0RXFlag=TRUE;
				TWI0Inx=0;
				TWI0OvTime = 0; 				// buffer is full,set overtime to zero
			}
		}
	}
	else if((SISTA==0x60)||(SISTA==0x68))
	{// 0x60,0x68: Own SLA+W has been received. ACK has been returned
		bTWI0ADFlag=0;					// The next two byte is Reg address
		TWI0Inx=0;
		TWI0OvTime = TWI_OV_TIME_MAX;			// restore TWI0 overtime
	}
	else if((SISTA==0xA8)||(SISTA==0xB0))
	{// 0x60,0x68: Own SLA+R has been received. ACK has been returned
		TWI0Inx=0;						// Set buffer index to zero
		goto _IIC_SET_SIDAT;
	}
	else if((SISTA==0xB8)||(SISTA==0xC0)||(SISTA==0xC8))
	{
        // 0xB8:  Data byte in SIDAT has been transmitted ACK has been received
		// 0xC0:  Data byte or Last data byte in SIDAT has been transmitted Not ACK has been received
		// 0xC8:  Last Data byte in SIDAT has been transmitted ACK has been received
_IIC_SET_SIDAT:
		TWI0OvTime = TWI_OV_TIME_MAX;			// restore TWI0 overtime
		if(TWI0Inx>=TEST_BUF_SIZE) 
		{
			TWI0Inx=0;
			TWI0OvTime = 0; 				// buffer is full,set overtime to zero
		}
		SIDAT=TWI0Buf[TWI0Inx]; 				// Load data to send
		TWI0Inx++;
	}
	TWI0_SCL=0;
	SICON=SICON|(AA);						// Set AA
	SICON=SICON&(~SI);						// clear TWI0 interrupt flag
	TWI0_SCL=1;

	_pop_(SFRPI);
}

/***********************************************************************************
*Function:   	void INT_TWI1(void)
*Description:	TWI1 Interrupt handler
*		 
*Input:   
*Output:     
*************************************************************************************/
void INT_TWSI1() interrupt INT_VECTOR_TWI1
{
	_push_(SFRPI);
	SFR_SetPage(1);
	if(SI1STA==0x80)
	{ // 0x80: DATA has been received. ACK has been returned.
		TWI1OvTime = TWI_OV_TIME_MAX;			// restore TWI1 overtime
		if(bTWI1ADFlag==0)
		{// Reg Address
			if(TWI1Inx==0)
			{
				TWI1RegAddr.B.BHigh=SI1DAT;
				TWI1Inx++;
			}
			else 
			{
				TWI1RegAddr.B.BLow=SI1DAT;
				bTWI1ADFlag=1;					// The next byte is Data
				TWI1Inx=0;
			}
		}
		else
		{
			TWI1Buf[TWI1Inx]=SI1DAT;				// Save received data into buffer
			TWI1Inx++;
			if(TWI1Inx>=TEST_BUF_SIZE) 
			{
				bTWI1RXFlag=TRUE;
				TWI1Inx=0;
				TWI1OvTime = 0; 				// buffer is full,set overtime to zero
			}
		}
	}
	else if((SI1STA==0x60)||(SI1STA==0x68))
	{// 0x60,0x68: Own SLA+W has been received. ACK has been returned
		bTWI1ADFlag=0;					// The next two byte is Reg address
		TWI1Inx=0;
		TWI1OvTime = TWI_OV_TIME_MAX;			// restore TWI1 overtime
	}
	else if((SI1STA==0xA8)||(SI1STA==0xB0))
	{// 0x60,0x68: Own SLA+R has been received. ACK has been returned
		TWI1Inx=0;						// Set buffer index to zero
		goto _IIC_SET_SI1DAT;
	}
	else if((SI1STA==0xB8)||(SI1STA==0xC0)||(SI1STA==0xC8))
	{
        // 0xB8:  Data byte in SIDAT has been transmitted ACK has been received
		// 0xC0:  Data byte or Last data byte in SIDAT has been transmitted Not ACK has been received
		// 0xC8:  Last Data byte in SIDAT has been transmitted ACK has been received
_IIC_SET_SI1DAT:
		TWI1OvTime = TWI_OV_TIME_MAX;			// restore TWI1 overtime
		if(TWI1Inx>=TEST_BUF_SIZE) 
		{
			TWI1Inx=0;
			TWI1OvTime = 0; 					// buffer is full,set overtime to zero
		}
		SI1DAT=TWI1Buf[TWI1Inx]; 				// Load data to send
		TWI1Inx++;
	}
	TWI1_SCL=0;
	SI1CON=SI1CON|(AA);						// Set AA
	SI1CON=SI1CON&(~SI1);					// clear TWI1 interrupt flag
	TWI1_SCL=1;
	_pop_(SFRPI);
}

/***********************************************************************************
*Function:		void SendByte(u8 tByte)
*Description:	send byte
*Input:			u8 tByte: the data to be send
*Output:     
*************************************************************************************/
void SendByte(BYTE tByte)
{
	bES0=ES0;
	ES0=0;
	S0BUF=tByte;
	while(TI0==0);
	TI0=0;
	ES0=bES0;
}

/***********************************************************************************
*Function:		void SendStr(BYTE* PStr)
*Description:	send string
*Input: 			u8* PStr:the string to be send
*Output:     
*************************************************************************************/
void SendStr(BYTE* PStr)
{
	while(*PStr != 0)
	{
		SendByte(*PStr);
		PStr ++;
	}
}

/***********************************************************************************
*Function:   	void SendHex(u8 ToSend)
*Description:	send data for HEX
*Input: 			u8 ToSend: the data to be send
*Output:     
*************************************************************************************/
void SendHex(BYTE ToSend)
{
	BYTE x;
	x = ToSend >> 4;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
	x= ToSend&0x0F;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
}

/*************************************************
*Function:     	void DelayXus(u16 xUs)
*Description:   	delay,unit:us
*Input:     		u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     	void DelayXms(u16 xMs)
*Description:    delay,unit:ms
*Input:     		u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

/***********************************************************************************
*Function:   	void InitUart0_S1BRG()
*Description:	Initialize Uart0, The source of baud rate was S1BRG
*Input:   
*Output:   		
*************************************************************************************/
void InitUart0_S1BRG(void)
{	
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS1BRG();			// B.R. source: S1BRG
	UART0_SetS1BRGBaudRateX2();			// S1BRG x2
	UART1_SetS1BRGSelSYSCLK();			// S1BRG clock source: SYSCLK

    // Sets B.R. value
	UART1_SetS1BRGValue(S1BRG_BRGRL_9600_2X_12000000_1T);
	
	UART1_EnS1BRG();					// Enable S1BRG
}

/***********************************************************************************
*Function:   	void InitTimer0(void)
*Description:	Initialize Timer0
*Input:   
*Output:   		
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// TIMER0 Mode: 16-bit
	TM_SetT0Clock_SYSCLKDiv12();			// TIMER0 Clock source: SYSCLK/12
	TM_SetT0Gate_Disable();

	TM_SetT0LowByte(TIMER_12T_1ms_TL);		// Set TL0 value
	TM_SetT0HighByte(TIMER_12T_1ms_TH);		// Set TH0 value

	TM_EnableT0();							// Enable TIMER0
}

/***********************************************************************************
*Function:   	void InitPort()
*Description:	Initialize IO Port
*Input:   
*Output:   		
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP1OpenDrain(BIT0|BIT1|BIT2|BIT3);			// Set P10,P11,P12,P13 as open drain
	PORT_EnP10P11P12P13PullHigh();						// Enable P10,P11,P12,P13 pull-high

	PORT_SetP2OpenDrain(BIT4|BIT5|BIT6|BIT7);			// Set P24,P25,P26,P27 as open drain
	PORT_EnP24P25P26P27PullHigh();						// Enable P24,P25,P26,P27 pull-high

	PORT_SetP2OpenDrain(BIT0|BIT1);						// Set P40,P41 as open drain
	PORT_EnP40P41PullHigh();							// Enable P40,P41 pull-high

}

/***********************************************************************************
*Function:   void InitTWI0_Slave(void)
*Description:Initialize TWI0 Slave mode
*Input:   
*Output:     
*************************************************************************************/
void InitTWI0_Slave()
{
	TWI0_Clear();
	TWI0_SetClock(TWI0_CLK_SYSCLK_32);
	TWI0_SetUseP40P41();
	TWI0_Enable();
	TWI0_SetSlaveAddr(SLAVE_ADDRESS);
	TWI0_SendACK();

}

/***********************************************************************************
*Function:   void InitTWI1_Slave(void)
*Description:Initialize TWI1 Slave mode
*Input:   
*Output:     
*************************************************************************************/
void InitTWI1_Slave()
{
	TWI1_Clear();
	TWI1_SetClock(TWI0_CLK_SYSCLK_32);
	TWI1_SetUseP10P11();
	TWI1_Enable();
	TWI1_SetSlaveAddr(SLAVE_ADDRESS);
	TWI1_SendACK();

}

/***********************************************************************************
*Function:   	void InitClock()
*Description:	Initialize clock
*Input:   
*Output:   		
*************************************************************************************/
void InitClock(void)
{
#if (MCU_SYSCLK==11059200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==12000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=12MHz CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==22118400)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==24000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==29491200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==32000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if (MCU_SYSCLK==48000000)
	// SysClk=48MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:   	void InitInterrupt()
*Description:	Initialize Interrupt
*Input:   
*Output:   		
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnTIMER0();				// Enable T0 interrupt
	INT_EnTWI0();				// Enable TWI0 interrupt
	INT_EnTWI1();				// Enable TWI1 interrupt
}	

/***********************************************************************************
*Function:       void InitSystem(void)
*Description:    Initialize MCU
*Input:   
*Output:     
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitTWI0_Slave();
	InitTWI1_Slave();
	InitTimer0();
	InitUart0_S1BRG();
	InitInterrupt();

	INT_EnAll();						// Enable global interrupt
}

void main()
{
	u8 i;
    InitSystem();


 	SendStr("\nStart");
 	
	LED_G_0=0;LED_G_1=0;LED_R=0;
	DelayXms(1000);
	LED_G_0=1;LED_G_1=1;LED_R=1;

	TWI0OvTime = 0; 				// Set buffer index to zero
	TWI1OvTime = 0; 				// Set buffer index to zero

    while(1)
    {
    	if(LedTime==0)
    	{
			LED_R = !LED_R;
			if(LED_R==0)
			{
				LedTime=20;
			}
			else
			{
				LedTime=250;
			}
    	}
    	if(bTWI0RXFlag==TRUE)
    	{
			LED_G_0=0;LED_R=1;
			SendStr("\nTWI0 RX 0x");
			SendHex(TWI0RegAddr.B.BHigh);
			SendHex(TWI0RegAddr.B.BLow);
			for(i=0;i<TEST_BUF_SIZE;i++)
			{
				SendByte(' ');
				SendHex(TWI0Buf[i]);
    	}
			bTWI0RXFlag=FALSE;
			LED_G_0=1;LED_R=1;
			LedTime=250;
    	}
    	if(bTWI1RXFlag==TRUE)
    	{
			LED_G_1=0;LED_R=1;
			SendStr("\nTWI1 RX 0x");
			SendHex(TWI1RegAddr.B.BHigh);
			SendHex(TWI1RegAddr.B.BLow);
			for(i=0;i<TEST_BUF_SIZE;i++)
			{
				SendByte(' ');
				SendHex(TWI1Buf[i]);
			}
			bTWI1RXFlag=FALSE;
			LED_G_1=1;LED_R=1;
			LedTime=250;
    	}
    	
    	if(bTWI0Error==TRUE)
    	{
    		bTWI0Error=FALSE;
			// error,reset TWI0
			LED_G_0=0;LED_R=0;
			InitTWI0_Slave();
			DelayXms(50);
			LED_G_0=1;LED_R=1;
			LedTime=250;
    	}
		if(bTWI1Error==TRUE)
		{
			bTWI1Error=FALSE;
			// error,reset TWI1
			LED_G_1=0;LED_R=0;
			InitTWI1_Slave();
			DelayXms(50);
			LED_G_1=1;LED_R=1;
			LedTime=250;
		}
    }
}


